/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.session.validation;

import static java.lang.String.format;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static org.mule.sdk.api.values.ValueResolvingException.MISSING_REQUIRED_PARAMETERS;

import org.mule.runtime.api.meta.NamedObject;
import org.mule.runtime.api.meta.model.parameter.ActingParameterModel;
import org.mule.runtime.app.declaration.api.ElementDeclaration;
import org.mule.runtime.app.declaration.api.ParameterElementDeclaration;
import org.mule.runtime.app.declaration.api.ParameterValue;
import org.mule.runtime.module.extension.internal.value.ValueProviderUtils;

import java.util.List;
import java.util.Map;
import java.util.Set;

public abstract class AbstractActingParametersValidator implements Validators.ComponentValidator {

  @Override
  public void validate(ComponentValidationContext<?> context) throws SessionCallValidationException {
    // Keep all the names of the acting parameters that are required
    Set<String> mandatoryActingParameterNames =
        doGetActingParameters(context)
            .stream()
            .filter(ActingParameterModel::isRequired)
            .map(ActingParameterModel::getExtractionExpression)
            .map(ValueProviderUtils::getParameterNameFromExtractionExpression)
            .collect(toSet());

    // Keep all the parameters that are optional and have a default value in the model, since if they are not populated
    // in the declaration, this value will be used
    Set<String> parametersWithDefaultValue = context.getParameterizedModel().getAllParameterModels()
        .stream()
        .filter(p -> !p.isRequired() && p.getDefaultValue() != null)
        .map(NamedObject::getName)
        .collect(toSet());

    // All populated parameters in the declaration for easy access
    Map<String, ParameterValue> populatedParameters =
        context.getComponentDeclaration()
            .getParameterGroups()
            .stream()
            .flatMap(g -> g.getParameters().stream())
            .collect(toMap(ElementDeclaration::getName, ParameterElementDeclaration::getValue));

    // The missing parameters will be all that do not have a default value (not caring if they are or not mandatory),
    // and that are not in the declaration.
    List<String> missingRequiredParameters = mandatoryActingParameterNames
        .stream()
        .filter(p -> !parametersWithDefaultValue.contains(p))
        .filter(rp -> !populatedParameters.containsKey(rp) || populatedParameters.get(rp) == null)
        .collect(toList());

    if (!missingRequiredParameters.isEmpty()) {
      throw new SessionCallValidationException(
                                               format("%s, the parameters %s are missing from the declaration",
                                                      getErrorMessagePrefix(context),
                                                      missingRequiredParameters),
                                               "Missing required parameters",
                                               MISSING_REQUIRED_PARAMETERS);
    }
  }

  protected abstract String getErrorMessagePrefix(ComponentValidationContext<?> context);

  protected abstract List<ActingParameterModel> doGetActingParameters(ComponentValidationContext<?> context);

}
