/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.lang.System.clearProperty;
import static java.lang.System.setProperty;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.collection.IsArrayWithSize.arrayWithSize;
import static org.junit.Assert.assertThat;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.runtime.api.meta.Category;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.XmlDslModel;
import org.mule.runtime.api.meta.model.display.DisplayModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.api.util.Reference;
import org.mule.runtime.extension.api.model.ImmutableExtensionModel;
import org.mule.tooling.client.test.utils.CheckedConsumer;

import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileTime;
import java.time.Duration;
import java.time.Instant;
import java.util.Optional;
import java.util.function.Consumer;

import com.google.common.cache.CacheStats;
import org.apache.commons.io.FileUtils;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.junit.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class ExtensionModelServiceCacheTestCase {

  private static final String ORG = "org";
  private static final String MULE = "mule";
  private static final String CONNECTORS = "connectors";
  private static final String GROUP_ID = ORG + "." + MULE + "." + CONNECTORS;
  private static final String ARTIFACT_ID = "mule-http-connector";
  private static final String JAR = "jar";
  private static final String MULE_PLUGIN = "mule-plugin";

  private static final String HTTP = "http";
  private static final String HTTP_MODIFIED_DESCRIPTION = "http modified";

  private static final String HTTP_XSD_CONTENT = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" +
      "\n" +
      "<xs:schema xmlns:mule=\"http://www.mulesoft.org/schema/mule/core\" xmlns:xs=\"http://www.w3.org/2001/XMLSchema\" targetNamespace=\"http://www.mulesoft.org/schema/mule/http\" attributeFormDefault=\"unqualified\" elementFormDefault=\"qualified\">\n"
      +
      "  <xsd:element name=\"http\" type=\"HTTP\" />\n" +
      "</xs:schema>";
  private static final String HTTP_XSD_MODIFIED_CONTENT = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" +
      "\n" +
      "<xs:schema xmlns:mule=\"http://www.mulesoft.org/schema/mule/core\" xmlns:xs=\"http://www.w3.org/2001/XMLSchema\" targetNamespace=\"http://www.mulesoft.org/schema/mule/http\" attributeFormDefault=\"unqualified\" elementFormDefault=\"qualified\">\n"
      +
      "  <xsd:element name=\"http\" type=\"HTTP Type\" />\n" +
      "</xs:schema>";

  // Just a version value (no matter if it is hardcoded) as this is used by the ExtensionModelCache
  // to create a folder for the data with this version value.
  private static final String TOOLING_VERSION = "4.2.0";

  private static final String MIN_MULE_VERSION = "4.1.0";

  private static final String TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_DISK_STORE_PATH =
      "tooling.client.ExtensionModelServiceCache.diskStore.path";

  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  @Mock
  private InternalExtensionModelService mockExtensionModelService;

  @Test
  public void removePreviousDataWhenSnapshotChangesUsingTimestampedVersions() {
    withSystemProperty(temporaryFolder.getRoot(), TOOLING_VERSION, cachedExtensionModelService -> {
      String baseVersion = "1.5.0-SNAPSHOT";

      // Version 1
      String versionV1 = "1.5.0-20181228.162716-230";
      File pluginFileV1 = temporaryFolder.newFile();
      ExtensionModel extensionModel = createExtensionModel(HTTP, versionV1, HTTP);
      LoadedExtensionInformation loadedExtensionInformation =
          new LoadedExtensionInformation(extensionModel, new LazyValue<>(HTTP_XSD_CONTENT), MIN_MULE_VERSION);

      when(mockExtensionModelService.loadExtensionData(eq(createBundleDescriptor(versionV1, baseVersion))))
          .thenReturn(Optional.of(loadedExtensionInformation));
      Optional<LoadedExtensionInformation> fetchedExtensionInformation = cachedExtensionModelService
          .loadExtensionInformation(createBundleDependency(pluginFileV1, versionV1, baseVersion), mockExtensionModelService);

      assertThat(fetchedExtensionInformation, not(empty()));
      assertThat(fetchedExtensionInformation.get().getExtensionModel().getVersion(), equalTo(versionV1));

      File resourcesDirectoryRootFolder =
          temporaryFolder.getRoot().toPath().resolve(Paths.get(TOOLING_VERSION, ORG, MULE, CONNECTORS, ARTIFACT_ID)).toFile();
      assertThat(resourcesDirectoryRootFolder.exists(), is(true));
      assertThat(resourcesDirectoryRootFolder.list(), arrayWithSize(1));

      File resourcesDirectoryV1 = resourcesDirectoryRootFolder.toPath().resolve(Paths.get(versionV1)).toFile();
      assertThat(resourcesDirectoryV1.exists(), is(true));
      assertThat(resourcesDirectoryV1.list(), arrayWithSize(3));

      CacheStats cacheStats = cachedExtensionModelService.getCacheStats();
      assertThat(cacheStats.missCount(), is(1l));
      assertThat(cacheStats.loadCount(), is(1l));
      assertThat(cacheStats.evictionCount(), is(0l));

      // Version 2
      String versionV2 = "1.5.0-20181228.183748-231";
      File pluginFileV2 = temporaryFolder.newFile();
      extensionModel = createExtensionModel(HTTP, versionV2, HTTP);
      loadedExtensionInformation =
          new LoadedExtensionInformation(extensionModel, new LazyValue<>(HTTP_XSD_CONTENT), MIN_MULE_VERSION);

      when(mockExtensionModelService.loadExtensionData(eq(createBundleDescriptor(versionV2, baseVersion))))
          .thenReturn(Optional.of(loadedExtensionInformation));
      fetchedExtensionInformation = cachedExtensionModelService
          .loadExtensionInformation(createBundleDependency(pluginFileV2, versionV2, baseVersion), mockExtensionModelService);

      assertThat(fetchedExtensionInformation, not(empty()));
      assertThat(fetchedExtensionInformation.get().getExtensionModel().getVersion(), equalTo(versionV2));

      resourcesDirectoryRootFolder =
          temporaryFolder.getRoot().toPath().resolve(Paths.get(TOOLING_VERSION, ORG, MULE, CONNECTORS, ARTIFACT_ID)).toFile();
      assertThat(resourcesDirectoryRootFolder.exists(), is(true));
      assertThat(resourcesDirectoryRootFolder.list(), arrayWithSize(1));

      File resourcesDirectoryV2 = resourcesDirectoryRootFolder.toPath().resolve(Paths.get(versionV2)).toFile();
      assertThat(resourcesDirectoryV2.exists(), is(true));
      assertThat(resourcesDirectoryV2.list(), arrayWithSize(3));

      cacheStats = cachedExtensionModelService.getCacheStats();
      assertThat(cacheStats.missCount(), is(2l));
      assertThat(cacheStats.loadCount(), is(2l));

      // previous version should not exists
      assertThat(resourcesDirectoryV1.exists(), is(false));
    });
  }

  @Test
  public void removePreviousDataWhenSnapshotChangesUsingNormalizedSnapshotVersions() {
    withSystemProperty(temporaryFolder.getRoot(), TOOLING_VERSION, cachedExtensionModelService -> {
      String baseVersion = "1.5.0-SNAPSHOT";

      // Version (same in both cases)
      String version = "1.5.0-SNAPSHOT";
      File pluginFile = temporaryFolder.newFile();
      BundleDescriptor bundleDescriptor = createBundleDescriptor(version, baseVersion);

      ExtensionModel extensionModel = createExtensionModel(HTTP, version, HTTP);
      LoadedExtensionInformation loadedExtensionInformation =
          new LoadedExtensionInformation(extensionModel, new LazyValue<>(HTTP_XSD_CONTENT), MIN_MULE_VERSION);

      when(mockExtensionModelService.loadExtensionData(bundleDescriptor)).thenReturn(Optional.of(loadedExtensionInformation));
      Optional<LoadedExtensionInformation> fetchedExtensionInformation = cachedExtensionModelService
          .loadExtensionInformation(createBundleDependency(pluginFile, version, baseVersion), mockExtensionModelService);

      assertThat(fetchedExtensionInformation, not(empty()));
      assertThat(fetchedExtensionInformation.get().getExtensionModel().getVersion(), equalTo(version));

      File resourcesDirectoryRootFolder =
          temporaryFolder.getRoot().toPath().resolve(Paths.get(TOOLING_VERSION, ORG, MULE, CONNECTORS, ARTIFACT_ID)).toFile();
      assertThat(resourcesDirectoryRootFolder.exists(), is(true));
      assertThat(resourcesDirectoryRootFolder.list(), arrayWithSize(1));

      File resourcesDirectorySnapshot = resourcesDirectoryRootFolder.listFiles()[0];
      assertThat(resourcesDirectorySnapshot.exists(), is(true));
      assertThat(resourcesDirectorySnapshot.list(), arrayWithSize(3));

      CacheStats cacheStats = cachedExtensionModelService.getCacheStats();
      assertThat(cacheStats.missCount(), is(1l));
      assertThat(cacheStats.loadCount(), is(1l));
      assertThat(cacheStats.evictionCount(), is(0l));

      // Version 2
      extensionModel = createExtensionModel(HTTP, version, HTTP_MODIFIED_DESCRIPTION);
      // Force to change lastModified on the same SNAPSHOT file (mocks a mvn install locally)
      touch(pluginFile.toPath());
      loadedExtensionInformation =
          new LoadedExtensionInformation(extensionModel, new LazyValue<>(HTTP_XSD_CONTENT), MIN_MULE_VERSION);

      when(mockExtensionModelService.loadExtensionData(bundleDescriptor)).thenReturn(Optional.of(loadedExtensionInformation));
      fetchedExtensionInformation = cachedExtensionModelService
          .loadExtensionInformation(createBundleDependency(pluginFile, version, baseVersion), mockExtensionModelService);

      assertThat(fetchedExtensionInformation, not(empty()));
      assertThat(fetchedExtensionInformation.get().getExtensionModel().getDescription(), equalTo(HTTP_MODIFIED_DESCRIPTION));

      File resourcesDirectorySecondTime = resourcesDirectoryRootFolder.listFiles()[0];
      assertThat(resourcesDirectorySecondTime.exists(), is(true));
      assertThat(resourcesDirectorySecondTime.list(), arrayWithSize(3));

      assertThat(resourcesDirectorySnapshot.exists(), is(false));
      assertThat(resourcesDirectorySnapshot.getName(), not(equalTo(resourcesDirectorySecondTime)));

      cacheStats = cachedExtensionModelService.getCacheStats();
      assertThat(cacheStats.missCount(), is(2l));
      assertThat(cacheStats.loadCount(), is(2l));
      verify(mockExtensionModelService, times(2)).loadExtensionData(bundleDescriptor);
    });
  }

  @Test
  public void keepPreviousVersionsWhenUsingReleases() {
    withSystemProperty(temporaryFolder.getRoot(), TOOLING_VERSION, cachedExtensionModelService -> {
      String baseVersion = "1.5.0";

      // Version 1
      String versionV1 = "1.5.0";
      File pluginFileV1 = temporaryFolder.newFile();
      ExtensionModel extensionModel = createExtensionModel(HTTP, versionV1, HTTP);
      LoadedExtensionInformation loadedExtensionInformation =
          new LoadedExtensionInformation(extensionModel, new LazyValue<>(HTTP_XSD_CONTENT), MIN_MULE_VERSION);

      when(mockExtensionModelService.loadExtensionData(createBundleDescriptor(versionV1, baseVersion)))
          .thenReturn(Optional.of(loadedExtensionInformation));
      Optional<LoadedExtensionInformation> fetchedExtensionInformation = cachedExtensionModelService
          .loadExtensionInformation(createBundleDependency(pluginFileV1, versionV1, baseVersion), mockExtensionModelService);

      assertThat(fetchedExtensionInformation, not(empty()));
      assertThat(fetchedExtensionInformation.get().getExtensionModel().getVersion(), equalTo(versionV1));

      File resourcesDirectoryRootFolder =
          temporaryFolder.getRoot().toPath().resolve(Paths.get(TOOLING_VERSION, ORG, MULE, CONNECTORS, ARTIFACT_ID)).toFile();
      assertThat(resourcesDirectoryRootFolder.exists(), is(true));
      assertThat(resourcesDirectoryRootFolder.list(), arrayWithSize(1));

      File resourcesDirectoryV1 = resourcesDirectoryRootFolder.toPath().resolve(Paths.get(versionV1)).toFile();
      assertThat(resourcesDirectoryV1.exists(), is(true));
      assertThat(resourcesDirectoryV1.list(), arrayWithSize(2));

      CacheStats cacheStats = cachedExtensionModelService.getCacheStats();
      assertThat(cacheStats.missCount(), is(1l));
      assertThat(cacheStats.loadCount(), is(1l));
      assertThat(cacheStats.evictionCount(), is(0l));

      // Version 2
      String versionV2 = "1.5.1";
      File pluginFileV2 = temporaryFolder.newFile();
      extensionModel = createExtensionModel(HTTP, versionV2, HTTP);
      loadedExtensionInformation =
          new LoadedExtensionInformation(extensionModel, new LazyValue<>(HTTP_XSD_CONTENT), MIN_MULE_VERSION);

      when(mockExtensionModelService.loadExtensionData(createBundleDescriptor(versionV2, baseVersion)))
          .thenReturn(Optional.of(loadedExtensionInformation));
      fetchedExtensionInformation = cachedExtensionModelService
          .loadExtensionInformation(createBundleDependency(pluginFileV2, versionV2, baseVersion), mockExtensionModelService);

      assertThat(fetchedExtensionInformation, not(empty()));
      assertThat(fetchedExtensionInformation.get().getExtensionModel().getVersion(), equalTo(versionV2));

      File resourcesDirectoryV2 = resourcesDirectoryRootFolder.toPath().resolve(Paths.get(versionV2)).toFile();
      assertThat(resourcesDirectoryV2.exists(), is(true));
      assertThat(resourcesDirectoryV2.list(), arrayWithSize(2));

      cacheStats = cachedExtensionModelService.getCacheStats();
      assertThat(cacheStats.missCount(), is(2l));
      assertThat(cacheStats.loadCount(), is(2l));

      // Previous version should exists too
      resourcesDirectoryV1 = resourcesDirectoryRootFolder.toPath().resolve(Paths.get(versionV1)).toFile();
      assertThat(resourcesDirectoryV1.exists(), is(true));
      assertThat(resourcesDirectoryV1.list(), arrayWithSize(2));
    });
  }

  @Test
  public void loadDataFromFileSystem() {
    Reference<Long> lastModifiedExtensionModel = new Reference<>();
    withSystemProperty(temporaryFolder.getRoot(), TOOLING_VERSION, cachedExtensionModelService -> {
      // First time the cache entry should be generated and persisted
      lastModifiedExtensionModel.set(doTestLoadDataFomFileSystem(cachedExtensionModelService));
      // Second time response should come from memory
      assertThat(lastModifiedExtensionModel.get(), equalTo(doTestLoadDataFomFileSystem(cachedExtensionModelService)));

      CacheStats cacheStats = cachedExtensionModelService.getCacheStats();
      assertThat(cacheStats.missCount(), is(1l));
      assertThat(cacheStats.loadCount(), is(1l));
      assertThat(cacheStats.hitCount(), is(1l));
      assertThat(cacheStats.evictionCount(), is(0l));
    });

    withSystemProperty(temporaryFolder.getRoot(), TOOLING_VERSION, cachedExtensionModelService -> {
      assertThat(lastModifiedExtensionModel.get(), equalTo(doTestLoadDataFomFileSystem(cachedExtensionModelService)));

      CacheStats cacheStats = cachedExtensionModelService.getCacheStats();
      assertThat(cacheStats.missCount(), is(1l));
      assertThat(cacheStats.loadCount(), is(1l));
      assertThat(cacheStats.hitCount(), is(0l));
      assertThat(cacheStats.evictionCount(), is(0l));
    });
  }

  @Test
  public void schemaShouldBeLoadedLazily() {
    Reference<Long> lastModifiedExtensionModel = new Reference<>();
    withSystemProperty(temporaryFolder.getRoot(), TOOLING_VERSION, cachedExtensionModelService -> {
      // First time the cache entry should be generated and persisted
      lastModifiedExtensionModel.set(doTestLoadDataFomFileSystem(cachedExtensionModelService));
    });
    withSystemProperty(temporaryFolder.getRoot(), TOOLING_VERSION, cachedExtensionModelService -> {
      // Second time response should come from memory
      assertThat(lastModifiedExtensionModel.get(),
                 equalTo(doTestLoadDataFomFileSystem(cachedExtensionModelService, of(optionalLoadedExtensionModelInformation -> {
                   assertThat(optionalLoadedExtensionModelInformation, not(empty()));

                   // now just modify the xsd file from cache persistent location as the load of the xsd should be done lazily
                   File resourcesDirectoryRootFolder =
                       temporaryFolder.getRoot().toPath().resolve(Paths.get(TOOLING_VERSION, ORG, MULE, CONNECTORS, ARTIFACT_ID))
                           .toFile();
                   assertThat(resourcesDirectoryRootFolder.exists(), is(true));

                   File[] files =
                       resourcesDirectoryRootFolder.listFiles()[0].listFiles(pathname -> pathname.getName().endsWith(".xsd"));
                   assertThat(files, arrayWithSize(1));
                   try {
                     FileUtils.writeStringToFile(files[0], HTTP_XSD_MODIFIED_CONTENT, false);
                   } catch (IOException e) {
                     throw new UncheckedIOException(e);
                   }

                   assertThat(optionalLoadedExtensionModelInformation.get().getSchema().get(),
                              equalTo(HTTP_XSD_MODIFIED_CONTENT));
                 }))));
    });
  }

  private long doTestLoadDataFomFileSystem(ExtensionModelServiceCache cachedExtensionModelService) throws IOException {
    return doTestLoadDataFomFileSystem(cachedExtensionModelService, empty());
  }

  private long doTestLoadDataFomFileSystem(ExtensionModelServiceCache cachedExtensionModelService,
                                           Optional<Consumer<Optional<LoadedExtensionInformation>>> optionalConsumer)
      throws IOException {
    String baseVersion = "1.5.0";

    // Version 1
    String versionV1 = "1.5.0";
    File pluginFileV1 = temporaryFolder.newFile();
    ExtensionModel extensionModel = createExtensionModel(HTTP, versionV1, HTTP);
    LoadedExtensionInformation loadedExtensionInformation =
        new LoadedExtensionInformation(extensionModel, new LazyValue<>(HTTP_XSD_CONTENT), MIN_MULE_VERSION);

    when(mockExtensionModelService.loadExtensionData(createBundleDescriptor(versionV1, baseVersion)))
        .thenReturn(Optional.of(loadedExtensionInformation));
    Optional<LoadedExtensionInformation> fetchedExtensionInformation = cachedExtensionModelService
        .loadExtensionInformation(createBundleDependency(pluginFileV1, versionV1, baseVersion), mockExtensionModelService);

    assertThat(fetchedExtensionInformation, not(empty()));
    assertThat(fetchedExtensionInformation.get().getExtensionModel().getVersion(), equalTo(versionV1));

    optionalConsumer.ifPresent(consumer -> consumer.accept(fetchedExtensionInformation));

    File resourcesDirectoryRootFolder =
        temporaryFolder.getRoot().toPath().resolve(Paths.get(TOOLING_VERSION, ORG, MULE, CONNECTORS, ARTIFACT_ID)).toFile();
    assertThat(resourcesDirectoryRootFolder.exists(), is(true));
    assertThat(resourcesDirectoryRootFolder.list(), arrayWithSize(1));

    File resourcesDirectory = resourcesDirectoryRootFolder.toPath().resolve(Paths.get(versionV1)).toFile();
    assertThat(resourcesDirectory.exists(), is(true));
    assertThat(resourcesDirectory.list(), arrayWithSize(2));

    CacheStats cacheStats = cachedExtensionModelService.getCacheStats();
    assertThat(cacheStats.missCount(), is(1l));
    assertThat(cacheStats.loadCount(), is(1l));
    assertThat(cacheStats.evictionCount(), is(0l));

    return resourcesDirectory.listFiles(pathname -> pathname.getName().endsWith(".json"))[0].lastModified();
  }

  public static void touch(final Path path) throws IOException {
    requireNonNull(path, "path is null");
    if (Files.exists(path)) {
      Files.setLastModifiedTime(path, FileTime.from(Instant.now().plus(Duration.ofMinutes(1))));
    } else {
      Files.createFile(path);
    }
  }

  private BundleDependency createBundleDependency(File pluginFile, String version, String baseVersion) {
    return new BundleDependency.Builder()
        .setDescriptor(createBundleDescriptor(version, baseVersion))
        .setBundleUri(pluginFile.toURI())
        .build();
  }

  private BundleDescriptor createBundleDescriptor(String version, String baseVersion) {
    return new BundleDescriptor.Builder()
        .setGroupId(GROUP_ID)
        .setArtifactId(ARTIFACT_ID)
        .setVersion(version)
        .setBaseVersion(baseVersion)
        .setType(JAR)
        .setClassifier(MULE_PLUGIN)
        .build();
  }

  private ExtensionModel createExtensionModel(String name, String version, String description) {
    return new ImmutableExtensionModel(name,
                                       description,
                                       version,
                                       "Mule",
                                       Category.PREMIUM,
                                       emptyList(),
                                       emptyList(),
                                       emptyList(),
                                       emptyList(),
                                       emptyList(),
                                       emptyList(),
                                       DisplayModel.builder().displayName(name).build(),
                                       XmlDslModel.builder().setNamespace(name).build(),
                                       emptySet(),
                                       emptySet(),
                                       emptySet(),
                                       emptySet(),
                                       emptySet(),
                                       emptySet(),
                                       emptySet(),
                                       emptySet(),
                                       emptySet(),
                                       null);
  }


  private void withSystemProperty(File persistenceFolder, String toolingVersion,
                                  CheckedConsumer<ExtensionModelServiceCache> consumer) {
    final String persistenceCacheSystemPropertyName = TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_DISK_STORE_PATH;
    final String previous = setProperty(persistenceCacheSystemPropertyName, persistenceFolder.getAbsolutePath());
    ExtensionModelServiceCache extensionModelServiceCache = new ExtensionModelServiceCache(toolingVersion, true);
    try {
      consumer.accept(extensionModelServiceCache);
    } finally {
      if (previous != null) {
        setProperty(persistenceCacheSystemPropertyName, persistenceFolder.getAbsolutePath());
      } else {
        clearProperty(persistenceCacheSystemPropertyName);
      }
      extensionModelServiceCache.dispose();
    }
  }

}
