/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.instanceOf;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.core.IsNot.not;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.doAnswer;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mule.runtime.api.meta.Category.COMMUNITY;
import static org.mule.runtime.api.meta.model.connection.ConnectionManagementType.NONE;

import org.mule.maven.client.api.model.BundleDependency;
import org.mule.runtime.api.meta.MuleVersion;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.XmlDslModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.construct.ConstructModel;
import org.mule.runtime.api.meta.model.data.sample.SampleDataProviderModel;
import org.mule.runtime.api.meta.model.nested.NestableElementModel;
import org.mule.runtime.api.meta.model.nested.NestableElementModelVisitor;
import org.mule.runtime.api.meta.model.nested.NestedChainModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.parameter.ValueProviderModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.extension.api.property.SinceMuleVersionModelProperty;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Before;
import org.junit.Test;

@Feature("DTOs")
@Story("Metadata Model mediator to filter out model properties that are not supported by the targetRuntimeVersion")
public class ExtensionModelTargetVersionMediatorTestCase implements MuleRuntimeExtensionModelProvider {

  private static final String NAME = "name";
  private static final String VERSION = "1.0";
  private static final String VENDOR = "Mule";

  private static final String PARAMETER_GROUP_MODEL_NAME = "parameterGroupModelName";
  private static final String PARAMETER_MODEL_NAME = "parameterModelName";

  private ArtifactDescriptor artifactDescriptor;
  private ExtensionModel extensionModel;
  private String minMuleVersion = "4.1.0";

  @Before
  public void setupExtensionModel() {
    artifactDescriptor = ArtifactDescriptor.newBuilder()
        .withGroupId("groupId")
        .withArtifactId("artifactId")
        .withVersion("1.0")
        .build();
    extensionModel = mock(ExtensionModel.class);
    when(extensionModel.getName()).thenReturn(NAME);
    when(extensionModel.getVersion()).thenReturn(VERSION);
    when(extensionModel.getCategory()).thenReturn(COMMUNITY);
    when(extensionModel.getVendor()).thenReturn(VENDOR);
    when(extensionModel.getXmlDslModel()).thenReturn(XmlDslModel.builder().build());
  }

  @Test
  public void filterByMajorMinorVersion() {
    doFilterTest("4.2.0", "4.1.0");
  }

  @Test
  public void filterByMajorMinorPatchVersion() {
    doFilterTest("4.1.5", "4.1.4");
  }

  @Test
  public void nestedChainModelsAreNotTransformed() {
    ConstructModel constructModel = mock(ConstructModel.class);
    when(constructModel.getName()).thenReturn("construct");
    NestedChainModel nestableElementModel = mock(NestedChainModel.class);
    doAnswer(i -> {
      ((NestableElementModelVisitor) i.getArgument(0)).visit(nestableElementModel);
      return null;
    }).when(nestableElementModel).accept(any(NestableElementModelVisitor.class));
    when(nestableElementModel.getName()).thenReturn("nestedChain");
    doReturn(asList(nestableElementModel)).when(constructModel).getNestedComponents();
    when(extensionModel.getConstructModels()).thenReturn(asList(constructModel));


    MuleRuntimeExtensionModelProvider mediator = new ExtensionModelTargetVersionMediator(new MuleVersion("4.3.0"), this);
    ExtensionModel extensionModel = mediator.getExtensionModel(artifactDescriptor).get();

    assertThat(extensionModel.getConstructModels(), hasSize(1));
    ConstructModel filteredConstructModel = extensionModel.getConstructModels().get(0);
    assertThat(filteredConstructModel.getNestedComponents(), hasSize(1));
    NestableElementModel filteredNestedComponent = filteredConstructModel.getNestedComponents().get(0);
    assertThat(filteredNestedComponent, instanceOf(NestedChainModel.class));
  }

  @Test
  public void valueProviderModelWithSinceModelProperty() {
    ValueProviderModel valueProviderModel = mock(ValueProviderModel.class);
    SinceMuleVersionModelProperty sinceMuleVersionModelProperty = new SinceMuleVersionModelProperty("4.4.0");
    when(valueProviderModel.getModelProperty(SinceMuleVersionModelProperty.class)).thenReturn(of(sinceMuleVersionModelProperty));

    final String name = "dummy";

    ParameterModel dummyParameter = mock(ParameterModel.class);
    when(dummyParameter.getName()).thenReturn(name);
    when(dummyParameter.getValueProviderModel()).thenReturn(of(valueProviderModel));

    ParameterGroupModel dummyGroup = mock(ParameterGroupModel.class);
    when(dummyGroup.getName()).thenReturn(name);
    when(dummyGroup.getParameterModels()).thenReturn(singletonList(dummyParameter));

    OperationModel dummyOperation = mock(OperationModel.class);
    when(dummyOperation.getName()).thenReturn(name);
    when(dummyOperation.getParameterGroupModels()).thenReturn(singletonList(dummyGroup));

    when(extensionModel.getOperationModels()).thenReturn(singletonList(dummyOperation));

    final String oldTargetRuntime = "4.1.0";
    final String newTargetRuntime = "4.4.0";

    ExtensionModelTargetVersionMediator mediator =
        new ExtensionModelTargetVersionMediator(new MuleVersion(newTargetRuntime), this);
    Optional<ExtensionModel> extensionModel = mediator.getExtensionModel(artifactDescriptor);

    OperationModel operationModel = extensionModel.get().getOperationModel(name).get();
    ParameterModel parameterModel = operationModel.getAllParameterModels().get(0);
    assertThat(parameterModel.getValueProviderModel().get(), is(valueProviderModel));

    mediator = new ExtensionModelTargetVersionMediator(new MuleVersion(oldTargetRuntime), this);
    extensionModel = mediator.getExtensionModel(artifactDescriptor);

    operationModel = extensionModel.get().getOperationModel(name).get();
    parameterModel = operationModel.getAllParameterModels().get(0);
    assertThat(parameterModel.getValueProviderModel(), is(empty()));
  }

  @Test
  public void sampleDataProviderModelIsNotFilteredForSameTargetVersion() {
    SampleDataProviderModel sampleDataProviderModel = mock(SampleDataProviderModel.class);

    final String name = "dummy";

    OperationModel dummyOperation = mock(OperationModel.class);
    when(dummyOperation.getName()).thenReturn(name);
    when(dummyOperation.getSampleDataProviderModel()).thenReturn(of(sampleDataProviderModel));

    SourceModel dummySource = mock(SourceModel.class);
    when(dummySource.getName()).thenReturn(name);
    when(dummySource.getSampleDataProviderModel()).thenReturn(of(sampleDataProviderModel));

    when(extensionModel.getOperationModels()).thenReturn(singletonList(dummyOperation));
    when(extensionModel.getSourceModels()).thenReturn(singletonList(dummySource));

    String newVersion = "4.4.0";

    ExtensionModelTargetVersionMediator mediator =
        new ExtensionModelTargetVersionMediator(new MuleVersion(newVersion), this);
    Optional<ExtensionModel> extensionModel = mediator.getExtensionModel(artifactDescriptor);

    OperationModel operationModel = extensionModel.get().getOperationModel(name).get();
    assertThat(operationModel.getName(), equalTo(name));
    assertThat(operationModel.getSampleDataProviderModel(), is(not(empty())));
    assertThat(operationModel.getSampleDataProviderModel().get(), is(sampleDataProviderModel));

    SourceModel sourceModel = extensionModel.get().getSourceModel(name).get();
    assertThat(sourceModel.getName(), equalTo(name));
    assertThat(sourceModel.getSampleDataProviderModel(), is(not(empty())));
    assertThat(sourceModel.getSampleDataProviderModel().get(), is(sampleDataProviderModel));
  }

  @Test
  public void semanticTermsAreNotFilteredForSameTargetVersion() {
    final String name = "dummy";
    final Set<String> semanticTerms = ImmutableSet.of("some", "semantic", "term");

    OperationModel dummyOperation = mock(OperationModel.class);
    when(dummyOperation.getName()).thenReturn(name);
    when(dummyOperation.getSemanticTerms()).thenReturn(semanticTerms);

    SourceModel dummySource = mock(SourceModel.class);
    when(dummySource.getName()).thenReturn(name);
    when(dummySource.getSemanticTerms()).thenReturn(semanticTerms);

    ConnectionProviderModel dummyConnectionProvider = mock(ConnectionProviderModel.class);
    when(dummyConnectionProvider.getName()).thenReturn(name);
    when(dummyConnectionProvider.getSemanticTerms()).thenReturn(semanticTerms);
    when(dummyConnectionProvider.getConnectionManagementType()).thenReturn(NONE);

    ConstructModel dummyConstructModel = mock(ConstructModel.class);
    when(dummyConstructModel.getName()).thenReturn(name);
    when(dummyConstructModel.getSemanticTerms()).thenReturn(semanticTerms);

    when(extensionModel.getOperationModels()).thenReturn(singletonList(dummyOperation));
    when(extensionModel.getSourceModels()).thenReturn(singletonList(dummySource));
    when(extensionModel.getConstructModels()).thenReturn(singletonList(dummyConstructModel));
    when(extensionModel.getConnectionProviders()).thenReturn(singletonList(dummyConnectionProvider));

    String newVersion = "4.4.0";

    ExtensionModelTargetVersionMediator mediator =
        new ExtensionModelTargetVersionMediator(new MuleVersion(newVersion), this);
    Optional<ExtensionModel> extensionModel = mediator.getExtensionModel(artifactDescriptor);

    OperationModel operationModel = extensionModel.get().getOperationModel(name).get();
    assertThat(operationModel.getName(), equalTo(name));
    assertThat(operationModel.getSemanticTerms(), equalTo(semanticTerms));

    SourceModel sourceModel = extensionModel.get().getSourceModel(name).get();
    assertThat(sourceModel.getName(), equalTo(name));
    assertThat(sourceModel.getSemanticTerms(), equalTo(semanticTerms));

    ConstructModel constructModel = extensionModel.get().getConstructModel(name).get();
    assertThat(constructModel.getName(), equalTo(name));
    assertThat(constructModel.getSemanticTerms(), equalTo(semanticTerms));

    ConnectionProviderModel connectionProviderModel = extensionModel.get().getConnectionProviderModel(name).get();
    assertThat(connectionProviderModel.getName(), equalTo(name));
    assertThat(connectionProviderModel.getSemanticTerms(), equalTo(semanticTerms));
  }


  private void doFilterTest(String sinceVersion, String targetRuntimeVersion) {
    // Configuration Models (includes ParameterGroupModels and ParameterModels)
    List<ConfigurationModel> configurationModels = new ArrayList<>();

    ConfigurationModel nonFilteredConfigurationModel = mock(ConfigurationModel.class);
    when(nonFilteredConfigurationModel.getModelProperty(SinceMuleVersionModelProperty.class))
        .thenReturn(of(new SinceMuleVersionModelProperty(targetRuntimeVersion)));
    String configurationModelName = "configurationModel1";
    when(nonFilteredConfigurationModel.getName()).thenReturn(configurationModelName);
    setParameterGroupModels(sinceVersion, targetRuntimeVersion, nonFilteredConfigurationModel);
    configurationModels.add(nonFilteredConfigurationModel);

    ConfigurationModel filteredConfigurationModel = mock(ConfigurationModel.class);
    when(filteredConfigurationModel.getModelProperty(SinceMuleVersionModelProperty.class))
        .thenReturn(of(new SinceMuleVersionModelProperty(sinceVersion)));
    configurationModels.add(filteredConfigurationModel);

    when(extensionModel.getConfigurationModels()).thenReturn(configurationModels);
    // Operation Models
    List<OperationModel> operationModels = new ArrayList<>();

    OperationModel nonFilteredOperationModel = mock(OperationModel.class);
    when(nonFilteredOperationModel.getModelProperty(SinceMuleVersionModelProperty.class))
        .thenReturn(of(new SinceMuleVersionModelProperty(targetRuntimeVersion)));
    String operationModelName = "operation1";
    when(nonFilteredOperationModel.getName()).thenReturn(operationModelName);
    setParameterGroupModels(sinceVersion, targetRuntimeVersion, nonFilteredOperationModel);
    operationModels.add(nonFilteredOperationModel);

    OperationModel filteredOperationModel = mock(OperationModel.class);
    when(filteredOperationModel.getModelProperty(SinceMuleVersionModelProperty.class))
        .thenReturn(of(new SinceMuleVersionModelProperty(sinceVersion)));
    operationModels.add(filteredOperationModel);

    when(extensionModel.getOperationModels()).thenReturn(operationModels);

    // Filter and assertions
    ExtensionModelTargetVersionMediator mediator =
        new ExtensionModelTargetVersionMediator(new MuleVersion(targetRuntimeVersion), this);
    Optional<ExtensionModel> extensionModel = mediator.getExtensionModel(artifactDescriptor);

    // Configuration Models
    assertThat(extensionModel, is(notNullValue()));
    assertThat(extensionModel.get().getConfigurationModels(), hasSize(1));
    ConfigurationModel configurationModel = extensionModel.get().getConfigurationModels().get(0);
    assertThat(configurationModel.getName(), is(configurationModelName));
    assertThat(configurationModel.getParameterGroupModels(), hasSize(1));
    ParameterGroupModel parameterGroupModel = configurationModel.getParameterGroupModels().get(0);
    assertThat(parameterGroupModel.getName(), is(PARAMETER_GROUP_MODEL_NAME));
    assertThat(parameterGroupModel.getParameterModels(), hasSize(1));
    assertThat(parameterGroupModel.getParameterModels().get(0).getName(), is(PARAMETER_MODEL_NAME));

    // Operations Models
    assertThat(extensionModel, is(notNullValue()));
    assertThat(extensionModel.get().getOperationModels(), hasSize(1));
    OperationModel operationModel = extensionModel.get().getOperationModels().get(0);
    assertThat(operationModel.getName(), is(operationModelName));
    assertThat(operationModel.getParameterGroupModels(), hasSize(1));
    parameterGroupModel = configurationModel.getParameterGroupModels().get(0);
    assertThat(parameterGroupModel.getName(), is(PARAMETER_GROUP_MODEL_NAME));
    assertThat(parameterGroupModel.getParameterModels(), hasSize(1));
    assertThat(parameterGroupModel.getParameterModels().get(0).getName(), is(PARAMETER_MODEL_NAME));
  }

  private void setParameterGroupModels(String sinceVersion, String targetRuntimeVersion, ParameterizedModel parameterizedModel) {
    List<ParameterGroupModel> parameterGroupModels = new ArrayList<>();
    ParameterGroupModel filteredParameterGroupModel = mock(ParameterGroupModel.class);
    when(filteredParameterGroupModel.getModelProperty(SinceMuleVersionModelProperty.class))
        .thenReturn(of(new SinceMuleVersionModelProperty(sinceVersion)));
    parameterGroupModels.add(filteredParameterGroupModel);

    ParameterGroupModel nonFilteredParameterGroupModel = mock(ParameterGroupModel.class);
    when(nonFilteredParameterGroupModel.getModelProperty(SinceMuleVersionModelProperty.class))
        .thenReturn(of(new SinceMuleVersionModelProperty(targetRuntimeVersion)));
    when(nonFilteredParameterGroupModel.getName()).thenReturn(PARAMETER_GROUP_MODEL_NAME);
    setParameterModels(sinceVersion, targetRuntimeVersion, nonFilteredParameterGroupModel);
    parameterGroupModels.add(nonFilteredParameterGroupModel);

    when(parameterizedModel.getParameterGroupModels()).thenReturn(parameterGroupModels);
  }

  private void setParameterModels(String sinceVersion, String targetRuntimeVersion, ParameterGroupModel parameterGroupModel) {
    List<ParameterModel> parameterModels = new ArrayList<>();
    ParameterModel filteredParameterModel = mock(ParameterModel.class);
    when(filteredParameterModel.getModelProperty(SinceMuleVersionModelProperty.class))
        .thenReturn(of(new SinceMuleVersionModelProperty(sinceVersion)));
    parameterModels.add(filteredParameterModel);

    ParameterModel nonFilteredParameterModel = mock(ParameterModel.class);
    when(nonFilteredParameterModel.getModelProperty(SinceMuleVersionModelProperty.class))
        .thenReturn(of(new SinceMuleVersionModelProperty(targetRuntimeVersion)));
    when(nonFilteredParameterModel.getName()).thenReturn(PARAMETER_MODEL_NAME);
    parameterModels.add(nonFilteredParameterModel);

    when(parameterGroupModel.getParameterModels()).thenReturn(parameterModels);

  }

  @Override
  public Optional<ExtensionModel> getExtensionModel(ArtifactDescriptor pluginDescriptor) {
    return of(extensionModel);
  }

  @Override
  public Optional<String> getMinMuleVersion(ArtifactDescriptor pluginDescriptor) {
    return of(minMuleVersion);
  }

  @Override
  public Optional<LoadedExtensionInformation> getExtensionModel(File plugin) {
    return of(new LoadedExtensionInformation(extensionModel, new LazyValue<>(""), minMuleVersion));
  }

  @Override
  public Optional<ExtensionModel> getExtensionModel(BundleDependency bundleDependency) {
    return of(extensionModel);
  }

  @Override
  public Optional<String> getMinMuleVersion(File plugin) {
    return of(minMuleVersion);
  }

  @Override
  public Set<ExtensionModel> loadExtensionModels(List<ArtifactClassLoader> artifactPluginClassLoaders) {
    return Sets.newHashSet(extensionModel);
  }

  @Override
  public Optional<String> getExtensionSchema(File plugin) {
    return empty();
  }

  @Override
  public Optional<String> getExtensionSchema(ArtifactDescriptor pluginDescriptor) {
    return empty();
  }

  @Override
  public List<ExtensionModel> getRuntimeExtensionModels() {
    return emptyList();
  }
}
