/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.lang.System.clearProperty;
import static java.lang.System.setProperty;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.junit.Assert.assertThat;
import static org.mockito.Mockito.mock;

import org.mule.runtime.api.component.ConfigurationProperties;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.tooling.client.internal.application.ToolingApplicationModel;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Optional;
import java.util.UUID;

import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.mockito.Mock;
import org.mockito.junit.MockitoJUnit;
import org.mockito.junit.MockitoRule;

public class ToolingApplicationModelTestCase {

  private static final String TOOLING_CLIENT_ALLOW_ACCESS_EXTERNAL_RESOURCES =
      "tooling.client.configuration.allowAccessExternalResources";

  @Rule
  public MockitoRule rule = MockitoJUnit.rule();

  @Mock
  private ToolingApplicationModel parent;
  @Mock
  private ArtifactAst muleApplicationModel;
  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  @Test
  public void classLoaderResource() throws IOException {
    File tmpResource = temporaryFolder.newFile();
    ClassLoader classLoader = new URLClassLoader(new URL[] {tmpResource.getParentFile().toURI().toURL()});

    Optional<URI> resource =
        new ToolingApplicationModel(parent, muleApplicationModel, mock(ConfigurationProperties.class), classLoader)
            .findResource(tmpResource.getName());
    assertThat(resource, equalTo(of(tmpResource.toURI())));
  }

  @Test
  public void nonExistingClassLoaderResource() throws IOException {
    File tmpResource = temporaryFolder.newFile();
    ClassLoader classLoader = new URLClassLoader(new URL[] {tmpResource.getParentFile().toURI().toURL()});

    Optional<URI> resource =
        new ToolingApplicationModel(parent, muleApplicationModel, mock(ConfigurationProperties.class), classLoader)
            .findResource(UUID.randomUUID().toString());
    assertThat(resource, equalTo(empty()));
  }

  @Test
  public void absoluteFilePathResource() throws Exception {
    File tmpResource = temporaryFolder.newFile();
    ClassLoader classLoader = new URLClassLoader(new URL[0]);

    withRemoteAccessEnabled(() -> {
      Optional<URI> resource =
          new ToolingApplicationModel(parent, muleApplicationModel, mock(ConfigurationProperties.class), classLoader)
              .findResource(tmpResource.toURI().toString());
      assertThat(resource, equalTo(of(tmpResource.toURI())));
    });
  }

  @Test
  public void absoluteFilePathResourceRemoteAccessDisable() throws IOException {
    File tmpResource = temporaryFolder.newFile();
    ClassLoader classLoader = new URLClassLoader(new URL[0]);

    Optional<URI> resource =
        new ToolingApplicationModel(parent, muleApplicationModel, mock(ConfigurationProperties.class), classLoader)
            .findResource(tmpResource.toURI().toString());
    assertThat(resource, equalTo(empty()));
  }

  @Test
  public void absoluteFilePathResourceNoSchema() throws IOException {
    File tmpResource = temporaryFolder.newFile();
    ClassLoader classLoader = new URLClassLoader(new URL[0]);

    Optional<URI> resource =
        new ToolingApplicationModel(parent, muleApplicationModel, mock(ConfigurationProperties.class), classLoader)
            .findResource(tmpResource.toURI().toString());
    assertThat(resource, equalTo(empty()));
  }

  @Test
  public void remoteResource() throws Exception {
    ClassLoader classLoader = new URLClassLoader(new URL[0]);

    final String wsdl = "http://tshirt-service.cloudhub.io/?wsdl";
    withRemoteAccessEnabled(() -> {
      Optional<URI> resource =
          new ToolingApplicationModel(parent, muleApplicationModel, mock(ConfigurationProperties.class), classLoader)
              .findResource(wsdl);
      assertThat(resource, equalTo(of(URI.create(wsdl))));
    });
  }

  private void withRemoteAccessEnabled(TestCallback callable) throws Exception {
    String previousValue = setProperty(TOOLING_CLIENT_ALLOW_ACCESS_EXTERNAL_RESOURCES, "true");
    try {
      callable.run();
    } finally {
      if (previousValue != null) {
        setProperty(TOOLING_CLIENT_ALLOW_ACCESS_EXTERNAL_RESOURCES, previousValue);
      } else {
        clearProperty(TOOLING_CLIENT_ALLOW_ACCESS_EXTERNAL_RESOURCES);
      }
    }
  }

  public interface TestCallback {

    void run() throws Exception;
  }


  @Test
  public void remoteResourceRemoteAccessDisable() throws IOException {
    ClassLoader classLoader = new URLClassLoader(new URL[0]);

    final String wsdl = "http://tshirt-service.cloudhub.io/?wsdl";
    Optional<URI> resource =
        new ToolingApplicationModel(parent, muleApplicationModel, mock(ConfigurationProperties.class), classLoader)
            .findResource(wsdl);
    assertThat(resource, equalTo(empty()));
  }
}
