/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.hamcrest;

import static java.util.Optional.empty;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.collection.IsIterableContainingInOrder.contains;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.hamcrest.core.IsNull.nullValue;
import static org.mule.tooling.client.internal.hamcrest.HamcrestUtils.validateThat;

import org.mule.tooling.client.api.extension.model.connection.ConnectionProviderModel;

import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeDiagnosingMatcher;

public class ConnectionProviderMatcher extends TypeSafeDiagnosingMatcher<ConnectionProviderModel> {

  public static Matcher<ConnectionProviderModel> from(org.mule.runtime.api.meta.model.connection.ConnectionProviderModel runtimeConnectionProvider) {
    if (runtimeConnectionProvider == null) {
      return nullValue(ConnectionProviderModel.class);
    }
    return new ConnectionProviderMatcher(runtimeConnectionProvider);
  }

  private final org.mule.runtime.api.meta.model.connection.ConnectionProviderModel runtimeConnectionProvider;

  private ConnectionProviderMatcher(org.mule.runtime.api.meta.model.connection.ConnectionProviderModel runtimeConnectionProvider) {
    this.runtimeConnectionProvider = runtimeConnectionProvider;
  }

  @Override
  protected boolean matchesSafely(ConnectionProviderModel connectionProviderModel, Description description) {
    return validateThat("name", connectionProviderModel.getName(), equalTo(runtimeConnectionProvider.getName()), description) &&
        validateThat("description", connectionProviderModel.getDescription(), equalTo(runtimeConnectionProvider.getDescription()),
                     description)
        &&
        validateThat("supportsConnectivityTesting", connectionProviderModel.supportsConnectivityTesting(),
                     equalTo(runtimeConnectionProvider.supportsConnectivityTesting()), description)
        &&

        runtimeConnectionProvider
            .getDisplayModel()
            .map(rdm -> connectionProviderModel.getDisplayModel()
                .map(dm -> validateThat("displayModel", dm, DisplayModelMatcher.from(rdm), description))
                .orElse(false))
            .orElseGet(() -> validateThat("displayModel", connectionProviderModel.getDisplayModel(), equalTo(empty()),
                                          description))
        &&

        validateThat("connectionManagementType", connectionProviderModel.getConnectionManagementType(),
                     ConnectionManagementTypeMatcher.from(runtimeConnectionProvider.getConnectionManagementType()), description)

        &&

        (runtimeConnectionProvider.getParameterGroupModels().isEmpty()
            ? validateThat("parameterGroupModels", connectionProviderModel.getParameterGroupModels(), hasSize(0), description)
            : validateThat("parameterGroupModels", connectionProviderModel.getParameterGroupModels(),
                           contains(ParameterGroupMatcher.sFrom(runtimeConnectionProvider.getParameterGroupModels())),
                           description))

        &&

        (runtimeConnectionProvider.getExternalLibraryModels().isEmpty()
            ? validateThat("externalLibraryModels", connectionProviderModel.getExternalLibraryModels(), hasSize(0), description)
            : validateThat("externalLibraryModels", connectionProviderModel.getExternalLibraryModels(),
                           contains(ExternalLibraryModelMatcher.sFrom(runtimeConnectionProvider.getExternalLibraryModels())),
                           description))
        &&

        (runtimeConnectionProvider.getStereotype() == null
            ? validateThat("stereotype", connectionProviderModel.getStereotype(), nullValue(), description)
            : validateThat("stereotype", connectionProviderModel.getStereotype(),
                           StereotypeMatcher.from(runtimeConnectionProvider.getStereotype()), description))

        &&

        (!connectionProviderModel.getDeprecationModel().isEnabled() ||
            runtimeConnectionProvider.getDeprecationModel()
                .map(rdm -> validateThat("deprecationModel", connectionProviderModel.getDeprecationModel().get(),
                                         DeprecationModelMatcher.from(rdm),
                                         description))
                .orElseGet(() -> validateThat("deprecationModel", connectionProviderModel.getDeprecationModel().get(),
                                              nullValue(),
                                              description)))
        &&

        validateThat("semanticTerms", connectionProviderModel.getSemanticTerms(),
                     equalTo(runtimeConnectionProvider.getSemanticTerms()), description);
  }

  @Override
  public void describeTo(Description description) {
    description.appendText("ConnectionProviderModel: ").appendValue(runtimeConnectionProvider);
  }

}
