/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.hamcrest;

import static java.util.Optional.empty;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.collection.IsIterableContainingInOrder.contains;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.hamcrest.core.IsNull.nullValue;
import static org.mule.tooling.client.internal.hamcrest.HamcrestUtils.validateThat;

import org.mule.runtime.extension.api.property.MetadataKeyIdModelProperty;
import org.mule.runtime.extension.api.property.TypeResolversInformationModelProperty;
import org.mule.tooling.client.api.extension.model.source.SourceModel;

import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeDiagnosingMatcher;

public class SourceModelMatcher extends TypeSafeDiagnosingMatcher<SourceModel> {

  public static Matcher<SourceModel> from(org.mule.runtime.api.meta.model.source.SourceModel runtimeSourceModel) {
    if (runtimeSourceModel == null) {
      return nullValue(SourceModel.class);
    }
    return new SourceModelMatcher(runtimeSourceModel);
  }

  private final org.mule.runtime.api.meta.model.source.SourceModel runtimeSourceModel;

  private SourceModelMatcher(org.mule.runtime.api.meta.model.source.SourceModel runtimeSourceModel) {
    this.runtimeSourceModel = runtimeSourceModel;
  }

  @Override
  protected boolean matchesSafely(SourceModel item, Description description) {
    return validateThat("name", item.getName(), equalTo(runtimeSourceModel.getName()), description) &&
        validateThat("description", item.getDescription(), equalTo(runtimeSourceModel.getDescription()), description) &&


        (runtimeSourceModel.getParameterGroupModels().isEmpty()
            ? validateThat("parameterGroupModels", item.getParameterGroupModels(), hasSize(0), description)
            : validateThat("parameterGroupModels", item.getParameterGroupModels(),
                           contains(ParameterGroupMatcher.sFrom(runtimeSourceModel.getParameterGroupModels())), description))

        &&

        runtimeSourceModel
            .getDisplayModel()
            .map(rdm -> item.getDisplayModel()
                .map(dm -> validateThat("displayModel", dm, DisplayModelMatcher.from(rdm), description))
                .orElse(false))
            .orElseGet(() -> validateThat("displayModel", item.getDisplayModel(), nullValue(), description))

        &&

        validateThat("output", item.getOutput(), OutputModelMatcher.from(runtimeSourceModel.getOutput()), description)

        &&

        validateThat("outputAttributes", item.getOutputAttributes(),
                     OutputModelMatcher.from(runtimeSourceModel.getOutputAttributes()), description)

        &&

        validateThat("isTransactional", item.isTransactional(), equalTo(runtimeSourceModel.isTransactional()), description)

        &&

        validateThat("requiresConnection", item.requiresConnection(), equalTo(runtimeSourceModel.requiresConnection()),
                     description)

        &&

        validateThat("supportsStreaming", item.supportsStreaming(), equalTo(runtimeSourceModel.supportsStreaming()), description)

        &&

        (runtimeSourceModel.getStereotype() == null
            ? validateThat("stereotype", item.getStereotype(), nullValue(), description)
            : validateThat("stereotype", item.getStereotype(), StereotypeMatcher.from(runtimeSourceModel.getStereotype()),
                           description))

        &&

        validateThat("hasResponse", item.hasResponse(), equalTo(runtimeSourceModel.hasResponse()), description)

        &&

        runtimeSourceModel
            .getSuccessCallback()
            .map(rsc -> item
                .getSuccessCallback()
                .map(sc -> validateThat("successCallback", sc, SourceCallbackMatcher.from(rsc), description))
                .orElse(false))
            .orElseGet(() -> validateThat("successCallback", item.getSuccessCallback(), equalTo(empty()), description))

        &&

        runtimeSourceModel
            .getErrorCallback()
            .map(rsc -> item
                .getErrorCallback()
                .map(sc -> validateThat("errorCallback", sc, SourceCallbackMatcher.from(rsc), description))
                .orElse(false))
            .orElseGet(() -> validateThat("errorCallback", item.getErrorCallback(), equalTo(empty()), description))

        &&

        runtimeSourceModel
            .getTerminateCallback()
            .map(rsc -> item
                .getTerminateCallback()
                .map(sc -> validateThat("terminateCallback", sc, SourceCallbackMatcher.from(rsc), description))
                .orElse(false))
            .orElseGet(() -> validateThat("terminateCallback", item.getTerminateCallback(), equalTo(empty()), description))

        &&

        (runtimeSourceModel.getErrorModels().isEmpty()
            ? validateThat("errorModels", item.getErrorModels(), hasSize(0), description)
            : validateThat("errorModels", item.getErrorModels(),
                           contains(ErrorModelMatcher.sFrom(runtimeSourceModel.getErrorModels())), description))

        &&

        (runtimeSourceModel.getNestedComponents().isEmpty()
            ? validateThat("nestedComponents", item.getNestedComponents(), hasSize(0), description)
            : validateThat("nestedComponents", item.getNestedComponents(),
                           contains(NestableElementMatcher.sFrom(runtimeSourceModel.getNestedComponents())), description))

        &&

        (!item.getDeprecationModel().isEnabled() ||
            runtimeSourceModel
                .getDeprecationModel()
                .map(rdp -> validateThat("deprecationModel", item.getDeprecationModel().get(),
                                         DeprecationModelMatcher.from(rdp), description))
                .orElseGet(() -> validateThat("deprecationModel", item.getDeprecationModel().get(), nullValue(),
                                              description)))
        &&

        (!item.getSampleDataProviderModel().isEnabled() ||
            runtimeSourceModel
                .getSampleDataProviderModel()
                .map(rsdm -> validateThat("sampleDataProvider", item.getSampleDataProviderModel().get(),
                                          SampleDataMatcher.from(rsdm), description))
                .orElseGet(() -> validateThat("sampleDataProvider", item.getSampleDataProviderModel().get(), nullValue(),
                                              description)))

        &&

        (!item.getRunsOnPrimaryNodeOnly().isEnabled()
            || validateThat("runsOnPrimaryNodeOnly", item.getRunsOnPrimaryNodeOnly().get(),
                            equalTo(runtimeSourceModel.runsOnPrimaryNodeOnly()), description))

        &&

        validateThat("semanticTerms", item.getSemanticTerms(), equalTo(runtimeSourceModel.getSemanticTerms()), description)

        &&

        runtimeSourceModel
            .getModelProperty(MetadataKeyIdModelProperty.class)
            .map(rmki -> item.getMetadataKeyIdModel()
                .map(mki -> validateThat("metadataKeyIdModel", mki, MetadataKeyIdMatcher.from(rmki), description))
                .orElse(false))
            .orElseGet(() -> validateThat("metadataKeyIdModel", item.getMetadataKeyIdModel(), equalTo(empty()), description))
        &&

        (!item.getTypeResolversInformationModel().isEnabled() ||
            runtimeSourceModel
                .getModelProperty(TypeResolversInformationModelProperty.class)
                .map(rmki -> validateThat("typeResolvers", item.getTypeResolversInformationModel().get(),
                                          TypeResolverInformationModelMatcher.from(rmki), description))
                .orElseGet(() -> validateThat("typeResolvers", item.getTypeResolversInformationModel().get(), nullValue(),
                                              description)));

  }

  @Override
  public void describeTo(Description description) {
    description.appendText("SourceModel: ").appendValue(runtimeSourceModel);
  }

}
