/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal;

import static java.lang.String.format;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static org.mule.runtime.api.component.location.Location.builderFromStringRepresentation;
import static org.mule.runtime.api.util.Preconditions.checkState;
import static org.mule.runtime.api.value.ResolvingFailure.Builder.newFailure;
import static org.mule.runtime.api.value.ValueResult.resultFrom;
import static org.mule.runtime.extension.api.values.ValueResolvingException.INVALID_LOCATION;
import static org.mule.runtime.module.extension.internal.value.ValueProviderUtils.cloneAndEnrichValue;
import static org.mule.tooling.client.api.feature.Feature.enabled;
import static org.mule.tooling.client.internal.Command.methodNotFound;
import static org.mule.tooling.client.internal.utils.ValueUtils.toFailureValueResultDTO;
import static org.mule.tooling.client.internal.utils.ValueUtils.toValueResultDTO;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.metadata.api.locator.ComponentLocator;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.feature.Action;
import org.mule.tooling.client.api.feature.Feature;
import org.mule.tooling.client.api.value.Value;
import org.mule.tooling.client.api.value.ValueResult;
import org.mule.tooling.client.api.value.provider.ValueProviderDisposeRequest;
import org.mule.tooling.client.api.value.provider.ValueProviderRequest;
import org.mule.tooling.client.api.value.provider.ValueProviderService;
import org.mule.tooling.client.internal.action.DispacheableAction;
import org.mule.tooling.client.internal.serialization.Serializer;
import org.mule.tooling.client.internal.values.ValueProviderCache;

import java.util.List;
import java.util.function.Supplier;

/**
 * Provides the capability of resolve the {@link Value values} for any Value Provider capable element in a Mule Domain or
 * Application.
 *
 * @since 1.0
 */
public abstract class AbstractValueProviderService implements ValueProviderService, Command {

  private LazyValue<ValueProviderCache> valueProviderCache;
  private LazyValue<? extends ComponentLocator<ComponentAst>> locator;
  private Serializer serializer;
  private Supplier<List<ExtensionModel>> extensionModelsSupplier;

  protected AbstractValueProviderService(Supplier<List<ExtensionModel>> extensionModelsSupplier,
                                         Serializer serializer,
                                         LazyValue<ValueProviderCache> valueProviderCache,
                                         LazyValue<? extends ComponentLocator<ComponentAst>> locator) {
    requireNonNull(extensionModelsSupplier, "extensionModelsSupplier should not be null");
    requireNonNull(serializer, "serializer cannot be null");
    requireNonNull(valueProviderCache, "value providers cache cannot be null");
    requireNonNull(locator, "component locator should not be null");

    this.serializer = serializer;
    this.valueProviderCache = valueProviderCache;
    this.locator = locator;
    this.extensionModelsSupplier = extensionModelsSupplier;
  }

  private List<ExtensionModel> getExtensionModels() {
    return extensionModelsSupplier.get();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ValueResult getValues(ValueProviderRequest valueProviderRequest) {
    try {
      return toValueResultDTO(
                              locator
                                  .get()
                                  .get(builderFromStringRepresentation(valueProviderRequest.getLocation()).build())
                                  .map(componentAst -> cloneAndEnrichValues(valueProviderCache.get().getValues(componentAst,
                                                                                                               valueProviderRequest
                                                                                                                   .getProviderName(),
                                                                                                               () -> doGetValue(valueProviderRequest)),
                                                                            componentAst,
                                                                            valueProviderRequest.getProviderName()))
                                  .orElse(resultFrom(newFailure().withMessage("Could not find component with location: "
                                      + valueProviderRequest.getLocation()
                                      + " for resolving values").withFailureCode(INVALID_LOCATION).build())));
    } catch (ToolingException e) {
      throw e;
    } catch (Exception e) {
      return toFailureValueResultDTO(e);
    }
  }

  private org.mule.runtime.api.value.ValueResult cloneAndEnrichValues(org.mule.runtime.api.value.ValueResult valueResult,
                                                                      ComponentAst componentAst, String providerName) {
    if (!valueResult.isSuccess()) {
      return valueResult;
    }

    return org.mule.runtime.api.value.ValueResult
        .resultFrom(valueResult.getValues().stream()
            .map(option -> cloneAndEnrichValue(option, componentAst.getModel(ParameterizedModel.class)
                .map(parameterizedModel -> getParameters(parameterizedModel, providerName))
                .orElseThrow(() -> new IllegalStateException("Component: '%s' is not a parameterized model: "
                    + componentAst.getLocation()))).build())
            .collect(toSet()));
  }

  /**
   * Given a parameter or parameter group name, this method will look for the correspondent {@link ParameterModel} or
   * {@link ParameterGroupModel}
   *
   *
   * @param parameterizedModel container model
   * @param valueName          name of value provider
   * @return the correspondent parameter
   */
  private List<ParameterModel> getParameters(ParameterizedModel parameterizedModel, String valueName) {
    return parameterizedModel.getAllParameterModels()
        .stream()
        .filter(parameterModel -> parameterModel.getValueProviderModel()
            .map(provider -> provider.getProviderName().equals(valueName))
            .orElse(false))
        .collect(toList());
  }


  @Override
  public Feature<Action<ValueProviderDisposeRequest>> disposeCachedValues() {
    return enabled(
                   new DispacheableAction<>(
                                            request -> locator
                                                .get()
                                                .get(builderFromStringRepresentation(request.getLocation().toString()).build())
                                                .ifPresent(componentAst -> valueProviderCache.get()
                                                    .dispose(componentAst, request.getParameterNames())),
                                            ValueProviderDisposeRequest.class,
                                            serializer));
  }

  protected abstract org.mule.runtime.api.value.ValueResult doGetValue(ValueProviderRequest valueProviderRequest);

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "getValues": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(ValueProviderRequest.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        ValueProviderRequest valueProviderRequest = serializer.deserialize(arguments[0]);
        return serializer.serialize(getValues(valueProviderRequest));
      }
      case "disposeCachedValues": {
        return disposeCachedValues();
      }
    }
    throw methodNotFound(this.getClass(), methodName);
  }
}
