/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal;

import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;
import static org.mule.tooling.client.internal.ComponentLocationFactory.toLocationPartsDTO;
import static org.mule.tooling.client.internal.ComponentLocationFactory.toSourceCodeLocationDTO;
import static org.mule.tooling.client.internal.ComponentLocationFactory.toTypedComponentIdentifierDTO;
import static org.mule.tooling.client.internal.LocationFactory.toLocationDTO;
import static org.mule.tooling.client.internal.MetadataPartsFactory.failureCodeToDTO;
import static org.mule.tooling.client.internal.MetadataPartsFactory.toMetadataComponentDTO;
import org.mule.datasense.api.DataSenseComponentInfo;
import org.mule.datasense.api.DataSenseElementInfo;
import org.mule.datasense.api.notifications.DataSenseNotification;
import org.mule.datasense.api.notifications.DataSenseNotificationType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.i18n.I18nMessage;
import org.mule.tooling.client.api.component.location.Location;
import org.mule.tooling.client.api.datasense.DataSenseInfo;
import org.mule.tooling.client.api.metadata.FailureCode;
import org.mule.tooling.client.api.metadata.MetadataComponent;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

public class DataSensePartsFactory {

  private ExtensionModelPartsFactory extensionModelPartsFactory;

  public DataSensePartsFactory() {
    this.extensionModelPartsFactory = new ExtensionModelPartsFactory();
  }

  private org.mule.tooling.client.api.datasense.DataSenseNotificationType toDataSenseNotificationTypeDTO(DataSenseNotificationType type) {
    if (type == DataSenseNotificationType.ERROR) {
      return org.mule.tooling.client.api.datasense.DataSenseNotificationType.errorDataSenseNotificationType(type.name());
    }
    if (type == DataSenseNotificationType.FATAL_ERROR) {
      return org.mule.tooling.client.api.datasense.DataSenseNotificationType.fatalErrorDataSenseNotificationType(type.name());
    }
    if (type == DataSenseNotificationType.INFO) {
      return org.mule.tooling.client.api.datasense.DataSenseNotificationType.infoDataSenseNotificationType(type.name());
    }
    if (type == DataSenseNotificationType.WARNING) {
      return org.mule.tooling.client.api.datasense.DataSenseNotificationType.warningDataSenseNotificationType(type.name());
    }
    return new org.mule.tooling.client.api.datasense.DataSenseNotificationType(type.name());
  }

  private org.mule.tooling.client.api.datasense.DataSenseNotification toDataSenseNotificationDTO(DataSenseNotification notification) {

    MetadataComponent component = notification.getMetadataComponent().isPresent()
        ? toMetadataComponentDTO(notification.getMetadataComponent().get())
        : null;

    I18nMessage reason = notification.getReason().orElse(null);
    org.mule.tooling.client.api.datasense.I18nMessage reasonDTO = notification.getReason().isPresent()
        ? new org.mule.tooling.client.api.datasense.I18nMessage(reason.getMessage(), reason.getCode(), reason.getArgs())
        : null;

    FailureCode failureCode = notification.getFailureCode().isPresent()
        ? failureCodeToDTO(notification.getFailureCode().get())
        : null;

    org.mule.tooling.client.api.component.location.ComponentLocation componentLocationDTO = null;
    if (notification.getComponentLocation().isPresent()) {
      ComponentLocation componentLocation = notification.getComponentLocation().get();

      componentLocationDTO = new org.mule.tooling.client.api.component.location.ComponentLocation(
                                                                                                  componentLocation.getLocation(),
                                                                                                  toTypedComponentIdentifierDTO(componentLocation
                                                                                                      .getComponentIdentifier()),
                                                                                                  toLocationPartsDTO(componentLocation
                                                                                                      .getParts()),
                                                                                                  toSourceCodeLocationDTO(componentLocation));

    }

    return new org.mule.tooling.client.api.datasense.DataSenseNotification(
                                                                           toDataSenseNotificationTypeDTO(notification
                                                                               .getNotificationType()),
                                                                           component,
                                                                           notification.getFailingElement().orElse(null),
                                                                           failureCode,
                                                                           new org.mule.tooling.client.api.datasense.I18nMessage(notification
                                                                               .getMessage().getMessage(),
                                                                                                                                 notification
                                                                                                                                     .getMessage()
                                                                                                                                     .getCode(),
                                                                                                                                 notification
                                                                                                                                     .getMessage()
                                                                                                                                     .getArgs()),
                                                                           reasonDTO,
                                                                           componentLocationDTO);
  }

  private List<org.mule.tooling.client.api.datasense.DataSenseNotification> toDataSenseMNotificationsDTO(List<DataSenseNotification> notifications) {
    return notifications.stream().map(dataSenseNotification -> toDataSenseNotificationDTO(dataSenseNotification))
        .collect(toList());
  }

  private org.mule.tooling.client.api.datasense.DataSenseElementInfo toDataSenseElementInfoDTO(DataSenseElementInfo dataSenseElementInfo) {
    return org.mule.tooling.client.api.datasense.DataSenseElementInfo.newBuilder()
        .withLocation(toLocationDTO(dataSenseElementInfo.getLocation()))
        .withExpected(dataSenseElementInfo.getExpected().orElse(null))
        .withExpectedOutput(dataSenseElementInfo.getExpectedOutput().orElse(null))
        .withActualOutput(dataSenseElementInfo.getActualOutput().orElse(null))
        .withInput(dataSenseElementInfo.getInput().orElse(null))
        .withOutput(dataSenseElementInfo.getOutput().orElse(null))
        .withIncoming(dataSenseElementInfo.getIncoming().orElse(null))
        .withResult(dataSenseElementInfo.getResult().orElse(null))
        .withMessages(dataSenseElementInfo.getMessages())
        .withExpectedInput(dataSenseElementInfo.getExpectedInput().orElse(null))
        .withActualInput(dataSenseElementInfo.getActualInput().orElse(null))
        .build();
  }

  public Optional<DataSenseInfo> toDataSenseInfoDTO(Optional<org.mule.datasense.api.DataSenseInfo> dataSenseInfoOptional) {
    if (!dataSenseInfoOptional.isPresent()) {
      return Optional.empty();
    }

    org.mule.datasense.api.DataSenseInfo dataSenseInfo = dataSenseInfoOptional.get();

    Map<Location, org.mule.tooling.client.api.datasense.DataSenseElementInfo> componentInfoByComponentPath = new HashMap<>();
    if (dataSenseInfo.getComponentInfoByComponentPath().isPresent()) {
      dataSenseInfo.getComponentInfoByComponentPath().get().entrySet().stream().forEach(entry -> componentInfoByComponentPath
          .put(toLocationDTO(entry.getKey()), toDataSenseElementInfoDTO(entry.getValue())));
    }

    final DataSenseInfo.Builder builder = DataSenseInfo.newBuilder()
        .withLocation(toLocationDTO(dataSenseInfo.getLocation()))
        .withActualInput(dataSenseInfo.getActualInput().orElse(null))
        .withActualOutput(dataSenseInfo.getActualOutput().orElse(null))
        .withExpected(dataSenseInfo.getExpected().orElse(null))
        .withExpectedInput(dataSenseInfo.getExpectedInput().orElse(null))
        .withExpectedOutput(dataSenseInfo.getExpectedOutput().orElse(null))
        .withIncoming(dataSenseInfo.getIncoming().orElse(null))
        .withInput(dataSenseInfo.getInput().orElse(null))
        .withOutput(dataSenseInfo.getOutput().orElse(null))
        .withResult(dataSenseInfo.getResult().orElse(null))
        .withOperationModel(extensionModelPartsFactory.toOperationModelDTO(dataSenseInfo.getOperationModel().orElse(null)))
        .withSourceModel(extensionModelPartsFactory.toSourceModelDTO(dataSenseInfo.getSourceModel().orElse(null)))
        .withMessages(dataSenseInfo.getMessages())
        .withComponentInfoByComponentPath(componentInfoByComponentPath)
        .withNotifications(toDataSenseMNotificationsDTO(dataSenseInfo.getDataSenseNotifications()));

    Map<String, MetadataType> globalBindings = new HashMap<>();
    if (dataSenseInfo.getGlobalBindings() != null) {
      dataSenseInfo.getGlobalBindings().entrySet().stream()
          .forEach(entry -> globalBindings.put(entry.getKey(), entry.getValue()));
    }
    if (!globalBindings.isEmpty()) {
      builder.withGlobalBindings(globalBindings);
    }

    Map<String, MetadataType> functionBindings = new HashMap<>();
    if (dataSenseInfo.getFunctionBindings() != null) {
      dataSenseInfo.getFunctionBindings().entrySet().stream()
          .forEach(entry -> functionBindings.put(entry.getKey(), entry.getValue()));
    }
    if (!functionBindings.isEmpty()) {
      builder.withFunctionBindings(functionBindings);
    }

    return of(builder.build());
  }

  public Optional<org.mule.tooling.client.api.datasense.DataSenseComponentInfo> toDataSenseComponentInfoDTO(Optional<DataSenseComponentInfo> dataSenseComponentInfoOptional) {
    if (!dataSenseComponentInfoOptional.isPresent()) {
      return Optional.empty();
    }

    DataSenseComponentInfo dataSenseComponentInfo = dataSenseComponentInfoOptional.get();
    return of(new org.mule.tooling.client.api.datasense.DataSenseComponentInfo(
                                                                               extensionModelPartsFactory
                                                                                   .toOperationModelDTO(dataSenseComponentInfo
                                                                                       .getOperationModel().orElse(null)),
                                                                               extensionModelPartsFactory
                                                                                   .toSourceModelDTO(dataSenseComponentInfo
                                                                                       .getSourceModel().orElse(null)),
                                                                               dataSenseComponentInfo.getMessages(),
                                                                               toDataSenseMNotificationsDTO(dataSenseComponentInfo
                                                                                   .getDataSenseNotifications())));
  }
}
