/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.application;

import static com.google.common.io.Files.createTempDir;
import static java.lang.String.format;
import static java.nio.file.Files.createTempDirectory;
import static java.util.Objects.requireNonNull;
import static org.apache.commons.io.FileUtils.deleteQuietly;
import static org.apache.commons.io.FileUtils.toFile;
import static org.mule.runtime.core.api.util.FileUtils.unzip;
import static org.mule.runtime.deployment.model.api.application.ApplicationDescriptor.MULE_APPLICATION_CLASSIFIER;
import static org.mule.runtime.deployment.model.api.domain.DomainDescriptor.MULE_DOMAIN_CLASSIFIER;
import static org.mule.tooling.client.internal.utils.IOUtils.readContentFromUrl;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.maven.client.api.BundleDescriptorCreationException;
import org.mule.maven.client.internal.util.MavenUtils;
import org.mule.runtime.core.api.config.bootstrap.ArtifactType;
import org.mule.tooling.client.api.Disposable;
import org.mule.tooling.client.api.exception.ToolingException;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;

import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.maven.model.Model;
import org.slf4j.Logger;

/**
 * Defines the type of artifact and the resources associated.
 *
 * @since 4.1
 */
public class ArtifactResources implements Disposable {

  private static final int CONNECT_TIMEOUT = 5000;
  private static final int READ_TIMEOUT = 5000;

  private static final String FILE_PROTOCOL = "file";

  private static final Logger LOGGER = getLogger(DefaultApplication.class);

  private final String artifactId;
  private final URL artifactUrlContent;
  private File rootArtifactFile;
  private File workingDirectory;
  private final ArtifactType artifactType;

  /**
   * Creates an instance of the resources for the given {@link URL}. If the
   *
   * @param artifactUrlContent
   */
  public ArtifactResources(String artifactId, URL artifactUrlContent) {
    requireNonNull(artifactUrlContent, "artifactUrlContent cannot be null");

    this.artifactUrlContent = artifactUrlContent;
    this.artifactId = artifactId;
    explode(artifactUrlContent);
    this.artifactType = discoverArtifactType(rootArtifactFile);
  }

  public String getArtifactId() {
    return artifactId;
  }

  /**
   * @return {@link URL} from which this artifact resources where created.
   */
  public URL getArtifactUrlContent() {
    return artifactUrlContent;
  }

  /**
   * @return {@link File} to the root artifact exploded location.
   */
  public File getRootArtifactFile() {
    return rootArtifactFile;
  }

  /**
   * @return {@link File} to a working directory created for this artifact.
   */
  public File getWorkingDirectory() {
    return workingDirectory;
  }

  /**
   * @return {@link ArtifactType} the type of this artifact.
   */
  public ArtifactType getArtifactType() {
    return artifactType;
  }

  /**
   * @return {@code true} if the {@link #getArtifactUrlContent()} references to a remote jar file.
   */
  // TODO change this to check if agent is on same machine
  public boolean isRemote() {
    return !isFileProtocol(artifactUrlContent) && !isArtifactExploded(artifactUrlContent);
  }

  @Override
  public void dispose() {
    if (!isArtifactExploded(artifactUrlContent)) {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Deleting temporary file used for artifact");
      }
      boolean fileDeleted = deleteQuietly(rootArtifactFile);
      if (!fileDeleted) {
        if (LOGGER.isWarnEnabled()) {
          LOGGER.warn("Couldn't delete temporary artifact file: {}", rootArtifactFile.getAbsoluteFile());
        }
      }
    }
    if (workingDirectory != null && workingDirectory.exists()) {
      boolean fileDeleted = deleteQuietly(workingDirectory);
      if (!fileDeleted) {
        if (LOGGER.isWarnEnabled()) {
          LOGGER.warn("Couldn't delete temporary working artifact file: {}", workingDirectory.getAbsoluteFile());
        }
      }
    }
  }

  @Override
  public String toString() {
    return ToStringBuilder.reflectionToString(this);
  }

  private void explode(URL artifactUrlContent) {
    workingDirectory = createTempDir();
    try {
      if (isArtifactExploded(artifactUrlContent)) {
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug("Artifact URL already references to a file location with the content exploded");
        }
        rootArtifactFile = toFile(artifactUrlContent);
      } else {
        if (LOGGER.isDebugEnabled()) {
          LOGGER
              .debug(
                     "Artifact URL either references to a remote location or a file location with packaged jar, copying to a temporary folder");
        }
        rootArtifactFile = createTempDirectory("artifact").toFile();
        // Just in case if an error happens and it is not disposed the artifact
        rootArtifactFile.deleteOnExit();

        if (!isFileProtocol(artifactUrlContent)) {
          // TODO this should be configured in Builder for tooling runtime client (TLS may be needed too)
          try (InputStream fileContent = readContentFromUrl(artifactUrlContent, CONNECT_TIMEOUT, READ_TIMEOUT)) {
            unzip(fileContent, rootArtifactFile);
          }
        } else {
          unzip(toFile(artifactUrlContent), rootArtifactFile);
        }
      }
    } catch (IOException e) {
      throw new ToolingException(format("Couldn't expand the artifact or content to a temporary folder from URL: %s",
                                        artifactUrlContent),
                                 e);
    }
  }

  private static ArtifactType discoverArtifactType(File rootArtifactFile) {
    Model pomModel;
    try {
      pomModel = MavenUtils.getPomModel(rootArtifactFile);
    } catch (BundleDescriptorCreationException e) {
      throw new ToolingException(format("Couldn't load pom model from artifact %s", rootArtifactFile.getAbsolutePath()), e);
    }
    String packaging = pomModel.getPackaging();

    if (packaging.equals(MULE_DOMAIN_CLASSIFIER)) {
      return ArtifactType.DOMAIN;
    } else if (packaging.equals(MULE_APPLICATION_CLASSIFIER)) {
      return ArtifactType.APP;
    } else {
      throw new IllegalArgumentException(format(
                                                "Couldn't identify type of artifact, pom packaging '%s' is not supported, it should be one of '%s' or '%s'",
                                                packaging, MULE_DOMAIN_CLASSIFIER, MULE_APPLICATION_CLASSIFIER));
    }
  }

  private static boolean isFileProtocol(URL artifactUrlContent) {
    return artifactUrlContent.getProtocol().equals(FILE_PROTOCOL);
  }

  private static boolean isArtifactExploded(URL artifactUrlContent) {
    return isFileProtocol(artifactUrlContent) && toFile(artifactUrlContent).isDirectory();
  }


}
