/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.application;

import static org.mule.runtime.deployment.model.api.builder.DeployableArtifactClassLoaderFactoryProvider.domainClassLoaderFactory;

import static java.util.Collections.emptySet;
import static java.util.Objects.requireNonNull;

import org.mule.runtime.deployment.model.api.builder.RegionPluginClassLoadersFactory;
import org.mule.runtime.deployment.model.api.plugin.resolver.PluginDependenciesResolver;
import org.mule.runtime.deployment.model.internal.tooling.ToolingArtifactClassLoader;
import org.mule.runtime.deployment.model.internal.tooling.ToolingDomainClassLoaderBuilder;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.runtime.module.artifact.api.descriptor.ArtifactPluginDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.DomainDescriptor;

import java.io.File;
import java.util.List;

import com.google.common.collect.ImmutableList;

/**
 * Factory that knows how to create a {@link ArtifactClassLoader} for domains.
 *
 * @since 4.1
 */
public class DomainClassLoaderFactory {

  private final ArtifactClassLoader containerArtifactClassLoader;
  private final RegionPluginClassLoadersFactory regionPluginClassLoadersFactory;
  private final PluginDependenciesResolver pluginDependenciesResolver;

  public DomainClassLoaderFactory(ArtifactClassLoader containerArtifactClassLoader,
                                  RegionPluginClassLoadersFactory regionPluginClassLoadersFactory,
                                  PluginDependenciesResolver pluginDependenciesResolver) {
    requireNonNull(containerArtifactClassLoader, "containerArtifactClassLoader cannot be null");
    requireNonNull(regionPluginClassLoadersFactory, "regionPluginClassLoadersFactory cannot be null");
    requireNonNull(pluginDependenciesResolver, "pluginDependenciesResolver cannot be null");

    this.containerArtifactClassLoader = containerArtifactClassLoader;
    this.regionPluginClassLoadersFactory = regionPluginClassLoadersFactory;
    this.pluginDependenciesResolver = pluginDependenciesResolver;
  }

  public ToolingArtifactClassLoader createDomainClassLoader(DomainDescriptor domainDescriptor, File workingDirectory) {
    final ToolingDomainClassLoaderBuilder domainClassLoaderBuilder =
        new ToolingDomainClassLoaderBuilder(containerArtifactClassLoader,
                                            domainClassLoaderFactory(name -> new File(workingDirectory, name)),
                                            regionPluginClassLoadersFactory);
    domainClassLoaderBuilder.setArtifactDescriptor(domainDescriptor);

    List<ArtifactPluginDescriptor> resolvedArtifactPluginDescriptors =
        pluginDependenciesResolver
            .resolve(emptySet(),
                     ImmutableList.<ArtifactPluginDescriptor>builder().addAll(domainDescriptor.getPlugins()).build(), true);
    resolvedArtifactPluginDescriptors.stream().forEach(domainClassLoaderBuilder::addArtifactPluginDescriptors);

    return domainClassLoaderBuilder.build();
  }

}
