/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.metadata;

import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.runtime.ast.api.ComponentAst;

import java.util.concurrent.Callable;

/**
 * Internal interface that defines a cache for storing metadata.
 *
 * @since 4.2.2, 4.3.0
 */
public interface MetadataCache {

  /**
   * Gets the operations metadata for the given {@link ComponentAst}, if not present the {@link Callable} should be called and
   * result stored in order to resolve the metadata.
   *
   * @param componentAst of the operation for which to get the metadata
   * @param resolver     the callback that knows how to fetch the metadata on Mule Runtime.
   * @return The metadata for the operation.
   */
  MetadataResult<OperationModel> getOperationMetadata(ComponentAst componentAst,
                                                      Callable<MetadataResult<OperationModel>> resolver);

  /**
   * Gets the source metadata for the given {@link ComponentAst}, if not present the {@link Callable} should be called and result
   * stored in order to resolve the metadata.
   *
   * @param componentAst of the source for which to get the metadata
   * @param resolver     the callback that knows how to fetch the metadata on Mule Runtime.
   * @return the metadata for the source.
   */
  MetadataResult<SourceModel> getSourceMetadata(ComponentAst componentAst,
                                                Callable<MetadataResult<SourceModel>> resolver);

  /**
   * Gets the key metadata for the given {@link ComponentAst) , if not present the {@link Callable} should be called and result
   * stored in order to resolve the metadata.
   *
   * @param componentModel of the component for which to get the metadata
   * @param resolver       the callback that knows how to fetch the metadata on Mule Runtime.
   * @return the metadata for the key.
   */
  MetadataResult<MetadataKeysContainer> getMetadataKeys(ComponentAst componentAst,
                                                        Callable<MetadataResult<MetadataKeysContainer>> resolver);

  /**
   * Dispose the entries that are related to the received {@link ComponentAst}. If the {@link ComponentAst} corresponds to a
   * configuration, then all metadata from all the operations that relate to that configuration will also be disposed.
   *
   * @param componentAst
   */
  void dispose(ComponentAst componentAst);

  /**
   * Invalidate the metadata keys that are related to the received {@link ComponentAst}
   * <p/>
   * Unlike {@link MetadataCache#dispose(ComponentAst)}, if the {@link ComponentAst} corresponds to a configuration, only the
   * metadata key entries that are related to that configuration id will be invalidated
   *
   * @param componentAst
   */
  void invalidateMetadataKeysFor(ComponentAst componentAst);


}
