/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.session.cache;

import static java.lang.String.format;

import java.lang.reflect.Method;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;

import org.apache.commons.collections4.map.AbstractMapDecorator;
import org.apache.commons.collections4.set.AbstractSetDecorator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

/**
 * Map decorator that is in charge of propagating MDC context between class loaders.
 * 
 * @param <K>
 * @param <V>
 */
public class MDCMapDecorator<K, V> extends AbstractMapDecorator<K, V> {

  private static final Logger LOGGER = LoggerFactory.getLogger(MDCMapDecorator.class);

  private Optional<Method> setContextMapMethod;

  public MDCMapDecorator(Map<K, V> delegate) {
    super(delegate);
    final ClassLoader classLoader = delegate.getClass().getClassLoader();
    // At this point the MDC is correctly set but as the implementation to be invoked has
    // been loaded by another class loader (client's class loader) so we have to propagate the MDC
    // context using reflection. This class is loaded by an isolated class loader therefore this propagation has
    // to be made.
    try {
      setContextMapMethod = Optional.of(classLoader.loadClass(MDC.class.getName()).getMethod("setContextMap", Map.class));
    } catch (Exception e) {
      LOGGER
          .error(format("Could not get MDC class or method setContextMap from class loader: %s. MDC context won't be propagated when calling cache storage. Cause: %s",
                        classLoader, e.getMessage()));
    }
  }

  private void setMDCContextMap() {
    setContextMapMethod.ifPresent(method -> {
      try {
        method.invoke(null, MDC.getCopyOfContextMap());
      } catch (Exception e) {
        LOGGER
            .error(format("Error calling setContextMap method on MDC. MDC context won't be propageted for tracing on cache storage invocation. Cause: %s",
                          e.getMessage()));
      }
    });
  }

  @Override
  public V get(Object key) {
    setMDCContextMap();
    return super.get(key);
  }

  @Override
  public V put(K key, V value) {
    setMDCContextMap();
    return super.put(key, value);
  }

  @Override
  public boolean containsKey(Object key) {
    setMDCContextMap();
    return super.containsKey(key);
  }

  @Override
  public Set<K> keySet() {
    setMDCContextMap();
    return new MDCSetDecorator(super.keySet());
  }

  private class MDCSetDecorator extends AbstractSetDecorator<K> {

    public MDCSetDecorator(Set<K> delegate) {
      super(delegate);
    }

    @Override
    public Iterator<K> iterator() {
      Iterator<K> delegate = super.iterator();
      return new Iterator<K>() {

        @Override
        public boolean hasNext() {
          return delegate.hasNext();
        }

        @Override
        public K next() {
          return delegate.next();
        }

        @Override
        public void remove() {
          setMDCContextMap();
          delegate.remove();
        }
      };
    }

    @Override
    public boolean remove(Object object) {
      setMDCContextMap();
      return super.remove(object);
    }

    @Override
    public boolean removeIf(Predicate<? super K> filter) {
      setMDCContextMap();
      return decorated().removeIf(filter);
    }

    @Override
    public boolean removeAll(Collection<?> coll) {
      setMDCContextMap();
      return super.removeAll(coll);
    }
  }

}
