/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.session.mediator.resolver;

import static java.util.Collections.emptySet;
import static org.mule.tooling.client.internal.utils.ValueResolverUtils.toResolvedValues;
import static org.mule.tooling.client.internal.utils.ValueResolverUtils.toValueResolverResult;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.parameter.ValueProviderModel;
import org.mule.runtime.api.value.Value;
import org.mule.runtime.api.value.ValueResult;
import org.mule.runtime.app.declaration.api.ParameterizedElementDeclaration;
import org.mule.tooling.client.api.value.resolver.ValueResolverResult;
import org.mule.tooling.client.internal.session.filter.exception.InvalidLevelValueException;
import org.mule.tooling.client.internal.session.filter.exception.MissingLevelException;
import org.mule.tooling.client.internal.session.filter.ValueFilter;
import org.mule.tooling.client.internal.session.cache.DeclarationValueProviderCache;
import org.mule.tooling.client.internal.session.filter.exception.UnknownLevelValueException;

import java.util.Set;

import org.slf4j.Logger;

public class ValueProviderResolver extends SkeletalValueResolver<ValueResult> implements ValuesResolver {

  private static final Logger LOGGER = getLogger(ValueProviderResolver.class);

  private final ParameterizedElementDeclaration elementDeclaration;
  private final ValueProviderModel valueProviderModel;
  private final DeclarationValueProviderCache valueProviderCache;
  private final ValueFilter valueFilter;

  public ValueProviderResolver(SessionFunction<ValueResult> sessionExecutorConsumer,
                               boolean ignoreCache,
                               ParameterizedElementDeclaration elementDeclaration,
                               ValueProviderModel valueProviderModel,
                               DeclarationValueProviderCache valueProviderCache,
                               ValueFilter valueFilter) {
    super(sessionExecutorConsumer, ignoreCache);
    this.elementDeclaration = elementDeclaration;
    this.valueProviderModel = valueProviderModel;
    this.valueProviderCache = valueProviderCache;
    this.valueFilter = valueFilter;
  }

  @Override
  public ValueResolverResult resolve(ParameterizedModel parameterizedModel,
                                     ParameterModel parameterModel,
                                     String parameterName) {
    String providerName = valueProviderModel.getProviderName();
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug(
                   "Resolving values using ValueProvider mechanism for parameter: '{}' using providerName: '{}' on element: '{}' for declared for extension: '{}'",
                   parameterName, providerName, elementDeclaration.getName(), elementDeclaration.getDeclaringExtension());
    }

    org.mule.runtime.api.value.ValueResult values =
        valueProviderCache.getValues(
                                     elementDeclaration,
                                     parameterName,
                                     () -> sessionFunction.apply(
                                                                 (runtimeToolingService, id) -> runtimeToolingService
                                                                     .getValues(id, elementDeclaration, providerName)),
                                     ignoreCache);
    if (!values.isSuccess()) {
      return toValueResolverResult(values);
    }
    Set<Value> requestedLevelValues = emptySet();
    try {
      requestedLevelValues = valueFilter.filter(valueProviderModel,
                                                parameterizedModel,
                                                elementDeclaration,
                                                values.getValues());
    } catch (MissingLevelException e) {
      return missingActingParametersFailure(e.getMissingParameterName());
    } catch (InvalidLevelValueException e) {
      return invalidActingParameterFailure(e.getTargetSelector(), e.getMessage());
    } catch (UnknownLevelValueException e) {
      if (!valueProviderModel.isOpen() && valueProviderModel.getPartOrder() > 1) {
        return unknownActingParametersFailure(e.getParameterName(), e.getParameterValue());
      }
    }
    return ValueResolverResult.success(toResolvedValues(requestedLevelValues));
  }
}
