/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.test;

import static com.google.common.collect.Lists.newArrayList;
import static java.lang.Boolean.valueOf;
import static java.lang.System.getProperty;
import static java.net.InetAddress.getLocalHost;
import static java.util.Arrays.asList;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.collection.IsArrayWithSize.arrayWithSize;
import static org.junit.Assert.assertThat;
import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import static org.mule.tooling.client.api.configuration.agent.AgentConfiguration.newAgentConfigurationBuilder;
import static org.mule.tooling.client.test.utils.PortUtils.findFreePort;

import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.api.model.RemoteRepository;
import org.mule.maven.client.internal.AetherResolutionContext;
import org.mule.maven.client.internal.DefaultLocalRepositorySupplierFactory;
import org.mule.maven.client.internal.DefaultSettingsSupplierFactory;
import org.mule.maven.client.internal.MavenEnvironmentVariables;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;
import org.mule.tooling.client.test.utils.MuleStandaloneConfiguration;
import org.mule.tooling.client.test.utils.MuleStandaloneController;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.List;
import java.util.Optional;

import org.apache.commons.io.filefilter.WildcardFileFilter;
import org.apache.maven.model.Model;
import org.apache.maven.model.io.xpp3.MavenXpp3Reader;
import org.eclipse.aether.repository.Authentication;
import org.eclipse.aether.repository.AuthenticationSelector;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.rules.TemporaryFolder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractMuleRuntimeTestCase {

  private static final Logger LOGGER = LoggerFactory.getLogger(AbstractMuleRuntimeTestCase.class);

  protected static final int DEFAULT_START_TIMEOUT = 50000;
  protected static final int DEFAULT_START_POLL_INTERVAL = 500;
  protected static final int DEFAULT_START_POLL_DELAY = 300;
  protected static final int DEFAULT_CONTROLLER_OPERATION_TIMEOUT = 15000;
  public static final String HTTP = "http";

  private static final String MULESOFT_PUBLIC_REPOSITORY = "https://repository.mulesoft.org/nexus/content/repositories/public/";
  private static final String MULESOFT_PRIVATE_REPOSITORY = "https://repository.mulesoft.org/nexus/content/repositories/private/";

  public static final String RUNTIMES_FOLDER = "runtimes";
  public static final String REST_AGENT_TRANSPORT_PORT_SYS_PROP = "-M-Drest.agent.transport.port=";
  public static final String MULE_RUNTIME_TOOLING_CLIENT = "mule-runtime-tooling-client";
  public static final String MULE_TOOLING_API_VERSION = "mule.tooling.api.version";

  public static final String POM_XML = "pom.xml";

  private static String muleDir = getProperty("MULE_HOME");
  protected MuleStandaloneController muleStandaloneController;
  protected int agentPort = findFreePort();
  protected AgentConfiguration defaultAgentConfiguration;
  protected MavenConfiguration defaultMavenConfiguration;
  protected List<String> defaultMuleStartArguments;

  private static String muleVersion;

  @ClassRule
  public static TemporaryFolder temporaryFolder = new TemporaryFolder();

  @Before
  public final void before() throws Exception {
    if (muleDir == null) {
      File targetFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI()).getParentFile();
      File runtimesFolder = new File(targetFolder, RUNTIMES_FOLDER);
      final String[] runtimes = runtimesFolder.list(new WildcardFileFilter("mule*"));
      assertThat(runtimes, arrayWithSize(1));
      muleDir = new File(runtimesFolder, runtimes[0]).getAbsolutePath();
    }

    defaultMavenConfiguration = createDefaultMavenConfiguration();

    muleStandaloneController =
        new MuleStandaloneController(new File(muleDir), new MuleStandaloneConfiguration(DEFAULT_START_TIMEOUT,
                                                                                        DEFAULT_START_POLL_INTERVAL,
                                                                                        DEFAULT_START_POLL_DELAY,
                                                                                        DEFAULT_CONTROLLER_OPERATION_TIMEOUT));

    org.eclipse.aether.repository.RemoteRepository privateRepository =
        new org.eclipse.aether.repository.RemoteRepository.Builder("mulesoft-private", null, null).build();
    final Optional<AuthenticationSelector> authenticatorSelector =
        new AetherResolutionContext(defaultMavenConfiguration).getAuthenticatorSelector();

    if (!authenticatorSelector.isPresent()) {
      throw new IllegalStateException("Couldn't find Server entries on Maven user settings.xml");
    }
    Authentication authentication =
        authenticatorSelector.get().getAuthentication(privateRepository);
    if (authentication == null) {
      throw new IllegalStateException("Couldn't find Server entry on Maven user settings.xml for mulesoft-private");
    }
    defaultMuleStartArguments = newArrayList(REST_AGENT_TRANSPORT_PORT_SYS_PROP + agentPort,
                                             "-M-DmuleRuntimeConfig.maven.repositoryLocation="
                                                 + defaultMavenConfiguration.getLocalMavenRepositoryLocation().getAbsolutePath(),
                                             "-M-DmuleRuntimeConfig.maven.repositories.mulesoft-public.url="
                                                 + MULESOFT_PUBLIC_REPOSITORY,
                                             "-M-DmuleRuntimeConfig.maven.repositories.mulesoft-private.url="
                                                 + MULESOFT_PRIVATE_REPOSITORY);

    defaultMavenConfiguration.getUserSettingsLocation()
        .ifPresent(file -> defaultMuleStartArguments
            .add("-M-DmuleRuntimeConfig.maven.userSettingsLocation=" + file.getAbsolutePath()));

    defaultMavenConfiguration.getGlobalSettingsLocation()
        .ifPresent(file -> defaultMuleStartArguments
            .add("-M-DmuleRuntimeConfig.maven.globalSettingsLocation=" + file.getAbsolutePath()));

    if (valueOf(getProperty("muleRuntime.debug.enabled"))) {
      defaultMuleStartArguments.add("-debug");
    }

    String localhost = getLocalHost().getHostAddress();
    defaultAgentConfiguration = newAgentConfigurationBuilder()
        .withDefaultConnectionTimeout(5000)
        .withDefaultReadTimeout(50000)
        .withToolingApiURLSupplier(() -> "http://" + localhost + ":" + agentPort + "/mule/tooling")
        .build();

    if (isStartMuleBeforeEachTest()) {
      startMuleRuntime();
    }
  }

  public static MavenConfiguration createDefaultMavenConfiguration() throws IOException {
    MavenConfiguration mavenConfiguration = createDefaultMavenConfigurationBuilder().build();
    LOGGER.info("Using MavenConfiguration {}", mavenConfiguration);
    return mavenConfiguration;
  }

  public static MavenConfiguration.MavenConfigurationBuilder createDefaultMavenConfigurationBuilder() throws IOException {
    MavenConfiguration.MavenConfigurationBuilder mavenConfigurationBuilder =
        newMavenConfigurationBuilder().withForcePolicyUpdateNever(true);

    final File localMavenRepository = new DefaultLocalRepositorySupplierFactory().environmentMavenRepositorySupplier().get();
    mavenConfigurationBuilder.withLocalMavenRepositoryLocation(localMavenRepository);

    final DefaultSettingsSupplierFactory settingsSupplierFactory =
        new DefaultSettingsSupplierFactory(new MavenEnvironmentVariables());

    mavenConfigurationBuilder.withRemoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
        .withId("mulesoft-public")
        .withUrl(new URL(MULESOFT_PUBLIC_REPOSITORY))
        .build());
    mavenConfigurationBuilder.withRemoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
        .withId("mulesoft-private")
        .withUrl(new URL(MULESOFT_PRIVATE_REPOSITORY))
        .build());

    settingsSupplierFactory.environmentUserSettingsSupplier().ifPresent(mavenConfigurationBuilder::withUserSettingsLocation);
    settingsSupplierFactory.environmentGlobalSettingsSupplier().ifPresent(mavenConfigurationBuilder::withGlobalSettingsLocation);

    return mavenConfigurationBuilder;
  }

  public static String getMuleVersion() {
    if (muleVersion == null) {
      File toolingRuntimeDependenciesPomFile;
      try {
        toolingRuntimeDependenciesPomFile =
            new File(new File(new File(AbstractMuleRuntimeTestCase.class.getProtectionDomain().getCodeSource().getLocation()
                .toURI())
                    .getParentFile().getParentFile().getParentFile(), MULE_RUNTIME_TOOLING_CLIENT), POM_XML);
      } catch (URISyntaxException e) {
        throw new MuleRuntimeException(e);
      }
      Model mavenProject = createMavenProject(toolingRuntimeDependenciesPomFile);
      final String version = mavenProject.getVersion();
      if (version != null) {
        if (version.startsWith("${") && version.endsWith("}")) {
          Model parentProject = createMavenProject(new File(toolingRuntimeDependenciesPomFile.getParentFile(),
                                                            mavenProject.getParent().getRelativePath()));
          String resolvedVersion =
              (String) parentProject.getProperties().get(version.substring(version.indexOf("{") + 1, version.lastIndexOf("}")));
          if (resolvedVersion != null) {
            muleVersion = resolvedVersion;
          }
        } else {
          muleVersion = version;
        }
      } else {
        muleVersion = mavenProject.getParent().getVersion();
      }
    }
    return muleVersion;
  }

  public static String getToolingVersion() {
    File toolingRuntimeDependenciesPomFile;
    try {
      toolingRuntimeDependenciesPomFile =
          new File(new File(AbstractMuleRuntimeTestCase.class.getProtectionDomain().getCodeSource().getLocation().toURI())
              .getParentFile().getParentFile().getParentFile(), POM_XML);
    } catch (URISyntaxException e) {
      throw new MuleRuntimeException(e);
    }
    Model mavenProject = createMavenProject(toolingRuntimeDependenciesPomFile);

    Object result = mavenProject.getProperties().get(MULE_TOOLING_API_VERSION);
    return result != null ? result.toString() : mavenProject.getVersion();
  }

  private static Model createMavenProject(File pomFile) {
    MavenXpp3Reader mavenReader = new MavenXpp3Reader();

    if (pomFile != null && pomFile.exists()) {
      try (FileReader reader = new FileReader(pomFile)) {
        Model model = mavenReader.read(reader);
        model.setPomFile(pomFile);

        return model;
      } catch (Exception e) {
        throw new RuntimeException("Couldn't get Maven Artifact from pom: " + pomFile);
      }
    }
    throw new IllegalArgumentException("pom file doesn't exits for path: " + pomFile);
  }

  protected void startMuleRuntime() {
    List<String> args = newArrayList(defaultMuleStartArguments);
    args.addAll(asList(getStartMuleArguments()));
    muleStandaloneController.start(args.toArray(new String[0]), getProtocol());
  }

  protected String getProtocol() {
    return HTTP;
  }

  @After
  public final void after() {
    if (muleStandaloneController.isRunning()) {
      muleStandaloneController.stop();
    }
  }

  @AfterClass
  public static void checkDisposedApps() {
    File toolingDir = new File(muleDir + "/tmp/tooling");
    if (toolingDir.exists()) {
      assertThat(toolingDir.listFiles().length, is(0));
    }
  }

  protected String[] getStartMuleArguments() {
    return new String[0];
  }

  protected boolean isStartMuleBeforeEachTest() {
    return true;
  }

  public static URL toUrl(URI uri) {
    try {
      return uri.toURL();
    } catch (MalformedURLException e) {
      throw new RuntimeException("Error while getting URL", e);
    }
  }



}

