/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.test;

import static com.google.common.collect.Lists.newArrayList;
import static com.mashape.unirest.http.Unirest.get;
import static com.mashape.unirest.http.Unirest.setTimeouts;
import static java.lang.Boolean.valueOf;
import static java.lang.String.format;
import static java.lang.String.valueOf;
import static java.lang.System.getProperty;
import static java.net.InetAddress.getLocalHost;
import static java.util.Collections.emptyList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.io.FileUtils.deleteQuietly;
import static org.apache.commons.lang3.tuple.ImmutablePair.of;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.collection.IsArrayWithSize.arrayWithSize;
import static org.junit.Assert.assertThat;
import static org.mule.maven.client.test.MavenTestHelper.createDefaultEnterpriseMavenConfiguration;
import static org.mule.tooling.client.api.configuration.agent.AgentConfiguration.newAgentConfigurationBuilder;
import static org.mule.tooling.client.test.RuntimeType.EMBEDDED;
import static org.mule.tooling.client.test.RuntimeType.REMOTE;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.module.embedded.api.ContainerConfiguration;
import org.mule.runtime.module.embedded.api.EmbeddedContainer;
import org.mule.runtime.module.embedded.api.Product;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;
import org.mule.tooling.client.test.utils.MuleStandaloneConfiguration;
import org.mule.tooling.client.test.utils.MuleStandaloneController;
import org.mule.tooling.client.test.utils.SystemPropertiesManager;
import org.mule.tooling.client.test.utils.probe.PollingProber;
import org.mule.tooling.client.test.utils.probe.Probe;

import org.apache.commons.io.filefilter.WildcardFileFilter;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.maven.model.Model;
import org.apache.maven.model.io.xpp3.MavenXpp3Reader;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.File;
import java.io.FileReader;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Optional;

import com.google.common.collect.ImmutableList;

@RunWith(Parameterized.class)
public abstract class AbstractMuleRuntimeTestCase {

  protected static final int DEFAULT_START_TIMEOUT = 50000;
  protected static final int DEFAULT_START_POLL_INTERVAL = 500;
  protected static final int DEFAULT_START_POLL_DELAY = 300;
  protected static final int DEFAULT_CONTROLLER_OPERATION_TIMEOUT = 15000;
  public static final String HTTP = "http";

  private static final String MULESOFT_PUBLIC_REPOSITORY = "https://repository.mulesoft.org/nexus/content/repositories/public/";
  private static final String MULESOFT_PRIVATE_REPOSITORY = "https://repository.mulesoft.org/nexus/content/repositories/private/";

  public static final String RUNTIMES_FOLDER = "runtimes";
  public static final String REST_AGENT_TRANSPORT_PORT_SYS_PROP = "rest.agent.transport.port";
  public static final String MULE_RUNTIME_TOOLING_CLIENT = "mule-runtime-tooling-client";
  public static final String MULE_TOOLING_API_VERSION = "mule.tooling.api.version";

  public static final String POM_XML = "pom.xml";

  private static String muleDir = getProperty("MULE_HOME");
  protected final RuntimeType runtimeType;
  protected MuleStandaloneController muleStandaloneController;
  protected AgentConfiguration defaultAgentConfiguration;
  protected MavenConfiguration defaultMavenConfiguration;
  protected List<ImmutablePair<String, String>> defaultMuleStartArguments;
  protected Optional<EmbeddedContainer> embeddedContainerOptional = empty();
  private static String muleVersion;
  private SystemPropertiesManager systemPropertiesManager;
  private boolean runtimeStarted = false;

  @ClassRule
  public static TemporaryFolder temporaryFolder = new TemporaryFolder();
  @Rule
  public DynamicPort agentPort = new DynamicPort("agentPort");

  @Parameterized.Parameters(name = "{0}")
  public static Collection<Object[]> data() {
    return Arrays.asList(new Object[][] {
        {EMBEDDED}, {REMOTE}
    });
  }

  public AbstractMuleRuntimeTestCase(RuntimeType runtimeType) {
    this.runtimeType = runtimeType;
  }

  @Before
  public void before() throws Exception {
    defaultMavenConfiguration = createDefaultEnterpriseMavenConfiguration();

    defaultMuleStartArguments = newArrayList(of(REST_AGENT_TRANSPORT_PORT_SYS_PROP, valueOf(agentPort.getNumber())),
                                             of("muleRuntimeConfig.maven.repositoryLocation",
                                                defaultMavenConfiguration.getLocalMavenRepositoryLocation()
                                                    .getAbsolutePath()),
                                             of("muleRuntimeConfig.maven.repositories.mulesoft-public.url",
                                                MULESOFT_PUBLIC_REPOSITORY),
                                             of("muleRuntimeConfig.maven.repositories.mulesoft-private.url",
                                                MULESOFT_PRIVATE_REPOSITORY));

    defaultMavenConfiguration.getUserSettingsLocation()
        .ifPresent(file -> defaultMuleStartArguments
            .add(of("muleRuntimeConfig.maven.userSettingsLocation", file.getAbsolutePath())));

    defaultMavenConfiguration.getGlobalSettingsLocation()
        .ifPresent(file -> defaultMuleStartArguments
            .add(of("muleRuntimeConfig.maven.globalSettingsLocation", file.getAbsolutePath())));

    if (valueOf(getProperty("muleRuntime.debug.enabled"))) {
      defaultMuleStartArguments.add(of("debug", ""));
    }

    String toolingApiUrl = getAgentToolingUrl();
    defaultAgentConfiguration = newAgentConfigurationBuilder()
        .withDefaultConnectionTimeout(5000)
        .withDefaultReadTimeout(50000)
        .withToolingApiURLSupplier(() -> toolingApiUrl)
        .build();

    if (runtimeType.equals(REMOTE)) {
      if (muleDir == null) {
        File targetFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI()).getParentFile();
        File runtimesFolder = new File(targetFolder, RUNTIMES_FOLDER);
        final String[] runtimes = runtimesFolder.list(new WildcardFileFilter("mule*"));
        assertThat(runtimes, arrayWithSize(1));
        muleDir = new File(runtimesFolder, runtimes[0]).getAbsolutePath();
      }

      cleanUpRemoteServerTlsConfiguration();

      muleStandaloneController =
          new MuleStandaloneController(new File(muleDir), new MuleStandaloneConfiguration(DEFAULT_START_TIMEOUT,
                                                                                          DEFAULT_START_POLL_INTERVAL,
                                                                                          DEFAULT_START_POLL_DELAY,
                                                                                          DEFAULT_CONTROLLER_OPERATION_TIMEOUT));
    } else {
      // TODO MULE-12240 - remove grizzly setting once the memory leak gets fixed
      systemPropertiesManager = new SystemPropertiesManager(ImmutableList.<ImmutablePair<String, String>>builder()
          .add(of("mule.testingMode", "true"))
          .add(of("org.glassfish.grizzly.DEFAULT_MEMORY_MANAGER", "org.glassfish.grizzly.memory.HeapMemoryManager"))
          .addAll(getAllStartupArguments())
          .build());
    }

    if (isStartMuleBeforeEachTest()) {
      startMuleRuntime();
    }
  }

  protected String getAgentUrl() {
    try {
      String localhost = getLocalHost().getHostAddress();
      return "http://" + localhost + ":" + agentPort.getNumber();
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  protected String getAgentToolingUrl() {
    try {
      return getAgentUrl() + "/mule/tooling";
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  protected String getAgentApplicationsUrl() {
    try {
      return getAgentUrl() + "/mule/applications";
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  protected void cleanUpRemoteServerTlsConfiguration() {
    File confFolder = new File(muleDir, "conf");
    deleteQuietly(new File(confFolder, "mule-agent-default.yml"));
    deleteQuietly(new File(confFolder, "mule-agent.yml"));
  }

  public static String getMuleVersion() {
    if (muleVersion == null) {
      muleVersion = getProperty("tooling.test.mule.version");
      if (muleVersion != null) {
        return muleVersion;
      }
      File toolingRuntimeDependenciesPomFile;
      try {
        toolingRuntimeDependenciesPomFile =
            new File(new File(new File(AbstractMuleRuntimeTestCase.class.getProtectionDomain().getCodeSource().getLocation()
                .toURI())
                    .getParentFile().getParentFile().getParentFile(), MULE_RUNTIME_TOOLING_CLIENT), POM_XML);
      } catch (URISyntaxException e) {
        throw new MuleRuntimeException(e);
      }
      Model mavenProject = createMavenProject(toolingRuntimeDependenciesPomFile);
      final String version = mavenProject.getVersion();
      if (version != null) {
        if (version.startsWith("${") && version.endsWith("}")) {
          Model parentProject = createMavenProject(new File(toolingRuntimeDependenciesPomFile.getParentFile(),
                                                            mavenProject.getParent().getRelativePath()));
          String resolvedVersion =
              (String) parentProject.getProperties().get(version.substring(version.indexOf("{") + 1, version.lastIndexOf("}")));
          if (resolvedVersion != null) {
            muleVersion = resolvedVersion;
          }
        } else {
          muleVersion = version;
        }
      } else {
        muleVersion = mavenProject.getParent().getVersion();
      }
    }
    return muleVersion;
  }

  public static String getToolingVersion() {
    String toolingApiVersion = getProperty("tooling.test.api.version");
    if (toolingApiVersion != null) {
      return toolingApiVersion;
    }
    File toolingRuntimeDependenciesPomFile;
    try {
      toolingRuntimeDependenciesPomFile =
          new File(new File(AbstractMuleRuntimeTestCase.class.getProtectionDomain().getCodeSource().getLocation().toURI())
              .getParentFile().getParentFile().getParentFile(), POM_XML);
    } catch (URISyntaxException e) {
      throw new MuleRuntimeException(e);
    }
    Model mavenProject = createMavenProject(toolingRuntimeDependenciesPomFile);

    Object result = mavenProject.getProperties().get(MULE_TOOLING_API_VERSION);
    return result != null ? result.toString() : mavenProject.getVersion();
  }

  private static Model createMavenProject(File pomFile) {
    MavenXpp3Reader mavenReader = new MavenXpp3Reader();

    if (pomFile != null && pomFile.exists()) {
      try (FileReader reader = new FileReader(pomFile)) {
        Model model = mavenReader.read(reader);
        model.setPomFile(pomFile);

        return model;
      } catch (Exception e) {
        throw new RuntimeException("Couldn't get Maven Artifact from pom: " + pomFile);
      }
    }
    throw new IllegalArgumentException("pom file doesn't exits for path: " + pomFile);
  }

  protected void startMuleRuntime() {
    if (runtimeStarted) {
      throw new RuntimeException("Runtime already started");
    }
    runtimeStarted = true;
    if (runtimeType.equals(EMBEDDED)) {

      try {
        systemPropertiesManager.set();
        EmbeddedContainer embeddedContainer = EmbeddedContainer.builder()
            .withMuleVersion(getMuleVersion())
            .withLog4jConfigurationFile(getClass().getClassLoader().getResource("log4j2-default.xml").toURI())
            .withMavenConfiguration(defaultMavenConfiguration)
            .withContainerConfiguration(ContainerConfiguration.builder()
                .withContainerFolder(temporaryFolder.newFolder())
                .withServerPlugin(new BundleDescriptor.Builder().setGroupId("com.mulesoft.agent")
                    .setArtifactId("mule-agent-plugin")
                    .setVersion("1.7.0-mule-4.x-SNAPSHOT")
                    .setType("zip").build())
                .build())
            .withProduct(Product.MULE_EE).build();
        embeddedContainerOptional = of(embeddedContainer);
        embeddedContainer.start();
      } catch (Exception e) {
        throw new RuntimeException(e);
      }
    } else {
      List<String> args = getAllStartupArguments().stream()
          .map(pair -> {
            if (pair.getKey().equals("debug")) {
              return "-debug";
            } else {
              return format("-M-D%s=%s", pair.getKey(), pair.getValue());
            }
          }).collect(toList());
      muleStandaloneController.start(args.toArray(new String[0]), getProtocol());
    }
    validateAgentIsUpAndRunning();
  }

  protected void validateAgentIsUpAndRunning() {
    new PollingProber(10000, 100).check(new Probe() {

      @Override
      public boolean isSatisfied() {
        try {
          setTimeouts(100, 1000);
          int statusCode = get(getAgentApplicationsUrl()).asString().getStatus();
          return statusCode <= 400;
        } catch (Exception e) {
          e.printStackTrace();
          return false;
        }
      }

      @Override
      public String describeFailure() {
        return "could not get a valid agent response";
      }
    });
  }

  private List<ImmutablePair<String, String>> getAllStartupArguments() {
    List<ImmutablePair<String, String>> args = newArrayList(defaultMuleStartArguments);
    args.addAll(getStartupSystemProperties());
    return args;
  }

  protected String getProtocol() {
    return HTTP;
  }

  @After
  public final void after() {
    if (runtimeType.equals(REMOTE)) {
      if (muleStandaloneController.isRunning()) {
        muleStandaloneController.stop();
      }
    } else {
      embeddedContainerOptional.ifPresent(EmbeddedContainer::stop);
      if (systemPropertiesManager != null) {
        systemPropertiesManager.unset();
      }
    }
  }

  @AfterClass
  public static void checkDisposedApps() {
    File toolingDir = new File(muleDir + "/tmp/tooling");
    if (toolingDir.exists()) {
      assertThat(toolingDir.listFiles().length, is(0));
    }
  }

  protected List<ImmutablePair<String, String>> getStartupSystemProperties() {
    return emptyList();
  }

  protected boolean isStartMuleBeforeEachTest() {
    return true;
  }

  public static URL toUrl(URI uri) {
    try {
      return uri.toURL();
    } catch (MalformedURLException e) {
      throw new RuntimeException("Error while getting URL", e);
    }
  }



}

