/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.test.utils;

import static java.lang.System.clearProperty;
import static java.lang.System.setProperty;
import static org.apache.commons.lang3.tuple.ImmutablePair.of;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang3.tuple.ImmutablePair;

/**
 * Helper class to set and unset a collection of system properties
 */
public class SystemPropertiesManager {

  private List<ImmutablePair<String, String>> systemProperties;
  private List<ImmutablePair<String, String>> previousProperties = new ArrayList<>();

  public SystemPropertiesManager(List<ImmutablePair<String, String>> systemProperties) {
    this.systemProperties = systemProperties;
  }

  /**
   * Sets the system properties.
   */
  public void set() {
    for (ImmutablePair<String, String> systemProperty : systemProperties) {
      String previousValue = setSystemProperty(systemProperty);
      previousProperties.add(of(systemProperty.getKey(), previousValue));
    }
  }

  /**
   * Unsets the system properties reverting they state to the one at the moment of invoking {@link #set()}
   */
  public void unset() {
    for (ImmutablePair<String, String> previousProperty : previousProperties) {
      setSystemProperty(previousProperty);
    }
  }

  private String setSystemProperty(ImmutablePair<String, String> systemProperty) {
    String previousValue;
    if (systemProperty.getValue() != null) {
      previousValue = setProperty(systemProperty.getKey(), systemProperty.getValue());
    } else {
      previousValue = clearProperty(systemProperty.getKey());
    }
    return previousValue;
  }

}
