/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.test;

import static com.google.common.collect.Lists.newArrayList;
import static com.mashape.unirest.http.Unirest.get;
import static com.mashape.unirest.http.Unirest.setTimeouts;
import static java.lang.Boolean.valueOf;
import static java.lang.String.format;
import static java.lang.String.valueOf;
import static java.lang.System.getProperty;
import static java.net.InetAddress.getLocalHost;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.io.FileUtils.deleteQuietly;
import static org.apache.commons.lang3.tuple.ImmutablePair.of;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.collection.IsArrayWithSize.arrayWithSize;
import static org.junit.Assert.assertThat;
import static org.mule.maven.client.test.MavenTestHelper.createDefaultEnterpriseMavenConfigurationBuilder;
import static org.mule.maven.client.test.MavenTestUtils.getMavenProjectVersion;
import static org.mule.maven.client.test.MavenTestUtils.getMavenProperty;
import static org.mule.tooling.client.api.configuration.agent.AgentConfiguration.builder;
import static org.mule.tooling.client.test.RuntimeType.EMBEDDED;
import static org.mule.tooling.client.test.RuntimeType.NONE;
import static org.mule.tooling.client.test.RuntimeType.REMOTE;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.internal.DefaultSettingsSupplierFactory;
import org.mule.maven.client.internal.MavenEnvironmentVariables;
import org.mule.runtime.module.embedded.api.ContainerConfiguration;
import org.mule.runtime.module.embedded.api.EmbeddedContainer;
import org.mule.runtime.module.embedded.api.Product;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;
import org.mule.tooling.client.test.utils.MuleStandaloneConfiguration;
import org.mule.tooling.client.test.utils.MuleStandaloneController;
import org.mule.tooling.client.test.utils.SystemPropertiesManager;
import org.mule.tooling.client.test.utils.probe.PollingProber;
import org.mule.tooling.client.test.utils.probe.Probe;

import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableList;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableList;
import org.apache.commons.io.filefilter.WildcardFileFilter;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(Parameterized.class)
public abstract class AbstractMuleRuntimeTestCase {

  protected static final int DEFAULT_START_TIMEOUT = 50000;
  protected static final int DEFAULT_START_POLL_INTERVAL = 500;
  protected static final int DEFAULT_START_POLL_DELAY = 300;
  protected static final int DEFAULT_CONTROLLER_OPERATION_TIMEOUT = 15000;

  private static final String ORG_MULE_TOOLING = "org.mule.tooling";

  public static final String MULE_TOOLING_APPS = "/.mule/tooling/apps";

  public static final String HTTP = "http";

  private static final String MULESOFT_PUBLIC_REPOSITORY = "https://repository.mulesoft.org/nexus/content/repositories/public/";
  private static final String MULESOFT_PRIVATE_REPOSITORY = "https://repository.mulesoft.org/nexus/content/repositories/private/";

  public static final String RUNTIMES_FOLDER = "runtimes";
  public static final String REST_AGENT_TRANSPORT_PORT_SYS_PROP = "rest.agent.transport.port";
  public static final String MULE_RUNTIME_TOOLING_CLIENT = "mule-runtime-tooling-client";
  public static final String MULE_TOOLING_API_VERSION = "mule.tooling.api.version";
  public static final String MULE_AGENT_VERSION = "mule.agent.version";
  public static final String MULE_VERSION = "mule.version";
  public static final String MULE_JMS_CONNECTOR_VERSION = "muleJmsConnectorVersion";
  public static final String MUNIT_TOOLS_PLUGIN_VERSION = "munitToolsVersion";
  public static final String MUNIT_RUNNER_PLUGIN_VERSION = "munitRunnerVersion";
  public static final String MULE_EMAIL_CONNECTOR_VERSION = "muleEmailConnectorVersion";
  public static final String MULE_SPRING_MODULE_VERSION = "muleSpringModuleVersion";
  public static final String MULE_DB_CONNECTOR_VERSION = "muleDbConnectorVersion";
  public static final String MULE_WSC_CONNECTOR_VERSION = "muleWscConnectorVersion";
  public static final String MULE_OBJECTSTORE_CONNECTOR_VERSION = "muleObjectStoreConnectorVersion";
  public static final String MULE_VALIDATION_MODULE_VERSION = "muleValidationModuleVersion";
  public static final String MULE_HTTP_CONNECTOR_VERSION = "muleHttpConnectorVersion";
  public static final String MULE_SOCKETS_CONNECTOR_VERSION = "muleSocketsConnectorVersion";
  public static final String MULE_FILE_CONNECTOR_VERSION = "muleFileConnectorVersion";
  public static final String MULE_FTP_CONNECTOR_VERSION = "muleFileConnectorVersion";
  public static final String EXTENSION_WITH_METADATA_CONNECTOR_VERSION = "extensionWithMetadataVersion";

  static final class ClassContextProvider extends SecurityManager {

    public Class<?>[] getClassContext() {
      return super.getClassContext();
    }
  }

  static ClassContextProvider contextClassContextProvider;

  static {
    AccessController.doPrivileged(new PrivilegedAction<Void>() {

      public Void run() {
        contextClassContextProvider = new ClassContextProvider();
        return null;
      }
    });
  }

  public static final Supplier<File> POM_FOLDER_FINDER = () -> {
    try {
      Class<?>[] classContext = contextClassContextProvider.getClassContext();
      Class<?> firstToolingClass = null;
      for (int i = classContext.length - 1; i >= 0 && firstToolingClass == null; i--) {
        final Class<?> aClass = classContext[i];
        if (aClass.getPackage().getName().startsWith(ORG_MULE_TOOLING)) {
          firstToolingClass = aClass;
        }
      }
      if (firstToolingClass == null) {
        throw new IllegalStateException("Couldn't find pom for resolving Mule version and Tooling version");
      }

      File sourceCodeLocation = new File(firstToolingClass.getProtectionDomain().getCodeSource().getLocation().toURI());
      if (sourceCodeLocation.isDirectory()) {
        return sourceCodeLocation.getParentFile().getParentFile().getParentFile();
      }
      return sourceCodeLocation;
    } catch (URISyntaxException e) {
      throw new RuntimeException(e);
    }
  };

  protected static String muleDir = getProperty("MULE_HOME");
  protected final RuntimeType runtimeType;
  protected MuleStandaloneController muleStandaloneController;
  protected AgentConfiguration defaultAgentConfiguration;
  protected MavenConfiguration defaultMavenConfiguration;
  protected List<ImmutablePair<String, String>> defaultMuleStartArguments;
  protected Optional<EmbeddedContainer> embeddedContainerOptional = empty();
  private static String muleVersion;
  private static String toolingVersion;
  private SystemPropertiesManager systemPropertiesManager;
  private boolean runtimeStarted = false;

  @ClassRule
  public static TemporaryFolder temporaryFolder = new TemporaryFolder();
  @Rule
  public DynamicPort agentPort = new DynamicPort("agentPort");
  @Rule
  public DynamicPort proxyPort = new DynamicPort("proxyPort");

  @Parameterized.Parameters(name = "{0}")
  public static Collection<Object[]> data() {
    return Arrays.asList(new Object[][] {
        //{EMBEDDED}//,
        {REMOTE}
    });
  }

  public AbstractMuleRuntimeTestCase(RuntimeType runtimeType) {
    this.runtimeType = runtimeType;
  }

  @Before
  public void before() throws Exception {
    DefaultSettingsSupplierFactory settingsSupplierFactory =
        new DefaultSettingsSupplierFactory(new MavenEnvironmentVariables());

    MavenConfiguration.MavenConfigurationBuilder mavenConfigurationBuilder = createDefaultEnterpriseMavenConfigurationBuilder()
        .ignoreArtifactDescriptorRepositories(false);
    settingsSupplierFactory.environmentSettingsSecuritySupplier()
        .ifPresent(mavenConfigurationBuilder::settingsSecurityLocation);

    defaultMavenConfiguration = mavenConfigurationBuilder.build();

    defaultMuleStartArguments = newArrayList(of(REST_AGENT_TRANSPORT_PORT_SYS_PROP, valueOf(agentPort.getNumber())),
                                             of("muleRuntimeConfig.maven.repositoryLocation",
                                                defaultMavenConfiguration.getLocalMavenRepositoryLocation()
                                                    .getAbsolutePath()),
                                             of("muleRuntimeConfig.maven.repositories.mulesoft-public.url",
                                                MULESOFT_PUBLIC_REPOSITORY),
                                             of("muleRuntimeConfig.maven.repositories.mulesoft-private.url",
                                                MULESOFT_PRIVATE_REPOSITORY),
                                             of("mule.testingMode", "true"));

    for (Map.Entry<Object, Object> sysPropEntry : System.getProperties().entrySet()) {
      final String key = (String) sysPropEntry.getKey();
      final String value = (String) sysPropEntry.getValue();

      if (key.startsWith("-M")) {
        defaultMuleStartArguments.add(of(key.replace("-M-D", ""), value));
      }
    }

    defaultMavenConfiguration.getUserSettingsLocation()
        .ifPresent(file -> defaultMuleStartArguments
            .add(of("muleRuntimeConfig.maven.userSettingsLocation", file.getAbsolutePath())));

    defaultMavenConfiguration.getSettingsSecurityLocation()
        .ifPresent(file -> defaultMuleStartArguments
            .add(of("muleRuntimeConfig.maven.settingsSecurityLocation", file.getAbsolutePath())));

    defaultMavenConfiguration.getGlobalSettingsLocation()
        .ifPresent(file -> defaultMuleStartArguments
            .add(of("muleRuntimeConfig.maven.globalSettingsLocation", file.getAbsolutePath())));

    if (valueOf(getProperty("muleRuntime.debug.enabled"))) {
      defaultMuleStartArguments.add(of("debug", ""));
    }

    defaultAgentConfiguration = newDefaultAgentConfigurationBuilder().build();

    if (runtimeType.equals(REMOTE)) {
      if (muleDir == null) {
        File targetFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI()).getParentFile();
        File runtimesFolder = new File(targetFolder, RUNTIMES_FOLDER);
        String[] runtimes = runtimesFolder.list(new WildcardFileFilter("mule*4.2*SNAPSHOT"));
        //TODO MTS-21 migrate all the tests!
        if (runtimes.length == 0) {
          runtimes = runtimesFolder.list(new WildcardFileFilter("mule*"));
        }
        assertThat(runtimes, arrayWithSize(1));
        muleDir = new File(runtimesFolder, runtimes[0]).getAbsolutePath();
      }

      cleanUpRemoteServerTlsConfiguration();

      muleStandaloneController =
          new MuleStandaloneController(new File(muleDir), new MuleStandaloneConfiguration(DEFAULT_START_TIMEOUT,
                                                                                          DEFAULT_START_POLL_INTERVAL,
                                                                                          DEFAULT_START_POLL_DELAY,
                                                                                          DEFAULT_CONTROLLER_OPERATION_TIMEOUT));
    } else if (runtimeType.equals(EMBEDDED)) {
      // TODO MULE-12240 - remove grizzly setting once the memory leak gets fixed
      systemPropertiesManager = new SystemPropertiesManager(ImmutableList.<ImmutablePair<String, String>>builder()
          .add(of("mule.testingMode", "true"))
          .add(of("org.glassfish.grizzly.DEFAULT_MEMORY_MANAGER", "org.glassfish.grizzly.memory.HeapMemoryManager"))
          .addAll(getAllStartupArguments())
          .build());
    }

    if (isStartMuleBeforeEachTest()) {
      startMuleRuntime();
    }
  }

  protected AgentConfiguration.Builder newDefaultAgentConfigurationBuilder() throws URISyntaxException {
    return builder()
        .withDefaultConnectionTimeout(6000)
        .withDefaultReadTimeout(60000)
        .withMuleVersion(getMuleVersion())
        .withToolingApiUrl(toUrl(new URI(getAgentToolingUrl())));
  }

  protected String getAgentUrl() {
    try {
      String localhost = getLocalHost().getHostAddress();
      return "http://" + localhost + ":" + agentPort.getNumber();
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  protected String getAgentToolingUrl() {
    try {
      return getAgentUrl() + "/mule/tooling";
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  protected String getAgentApplicationsUrl() {
    try {
      return getAgentUrl() + "/mule/applications";
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  protected void cleanUpRemoteServerTlsConfiguration() {
    File confFolder = new File(muleDir, "conf");
    deleteQuietly(new File(confFolder, "mule-agent-default.yml"));
    deleteQuietly(new File(confFolder, "mule-agent.yml"));
  }

  public static URI getTestLog4JConfigurationFile() {
    try {
      return AbstractMuleRuntimeTestCase.class.getClassLoader().getResource("log4j2-tooling.xml").toURI();
    } catch (URISyntaxException e) {
      throw new IllegalStateException("Cannot find log4j default configuration file", e);
    }
  }

  public static String getMuleVersion() {
    if (muleVersion == null) {
      muleVersion = getMavenProperty("tooling.test.mule.version", POM_FOLDER_FINDER);
      if (muleVersion != null) {
        return muleVersion;
      }

      muleVersion = toolingVersion;
    }
    return muleVersion;
  }

  public static String getToolingVersion() {
    if (toolingVersion == null) {
      toolingVersion = getMavenProjectVersion(POM_FOLDER_FINDER);
    }
    return toolingVersion;
  }

  protected void startMuleRuntime() {
    if (runtimeType.equals(NONE)) {
      return;
    }
    if (runtimeStarted) {
      throw new RuntimeException("Runtime already started");
    }
    runtimeStarted = true;
    if (runtimeType.equals(EMBEDDED)) {

      try {
        systemPropertiesManager.set();
        EmbeddedContainer embeddedContainer = EmbeddedContainer.builder()
            .muleVersion(getMuleVersion())
            .log4jConfigurationFile(getTestLog4JConfigurationFile())
            .mavenConfiguration(defaultMavenConfiguration)
            .containerConfiguration(ContainerConfiguration.builder()
                .containerFolder(temporaryFolder.newFolder())
                .serverPlugins(new BundleDescriptor.Builder().setGroupId("com.mulesoft.agent")
                    .setArtifactId("mule-agent-plugin")
                    .setVersion(getMavenProperty(MULE_AGENT_VERSION, POM_FOLDER_FINDER))
                    .setClassifier("mule-server-plugin").build())
                .build())
            .product(Product.MULE_EE).build();
        embeddedContainerOptional = of(embeddedContainer);
        embeddedContainer.start();
      } catch (Exception e) {
        throw new RuntimeException(e);
      }
    } else {
      List<String> args = getAllStartupArguments().stream()
          .map(pair -> {
            if (pair.getKey().equals("debug")) {
              return "-debug";
            } else {
              return format("-M-D%s=%s", pair.getKey(), pair.getValue());
            }
          }).collect(toList());
      muleStandaloneController.start(args.toArray(new String[0]), getProtocol());
    }
    validateAgentIsUpAndRunning();
  }

  protected void validateAgentIsUpAndRunning() {
    new PollingProber(10000, 100).check(new Probe() {

      @Override
      public boolean isSatisfied() {
        try {
          setTimeouts(100, 1000);
          int statusCode = get(getAgentApplicationsUrl()).asString().getStatus();
          return statusCode <= 400;
        } catch (Exception e) {
          e.printStackTrace();
          return false;
        }
      }

      @Override
      public String describeFailure() {
        return "could not get a valid agent response";
      }
    });
  }

  private List<ImmutablePair<String, String>> getAllStartupArguments() {
    List<ImmutablePair<String, String>> args = newArrayList(defaultMuleStartArguments);
    args.addAll(getStartupSystemProperties());
    return args;
  }

  protected String getProtocol() {
    return HTTP;
  }

  @After
  public final void after() {
    stopMuleRuntime();
  }

  protected void stopMuleRuntime() {
    if (runtimeType.equals(REMOTE)) {
      if (muleStandaloneController.isRunning()) {
        muleStandaloneController.stop();
      }
    } else {
      embeddedContainerOptional.ifPresent(EmbeddedContainer::stop);
      if (systemPropertiesManager != null) {
        systemPropertiesManager.unset();
      }
    }
  }

  @AfterClass
  public static void checkDisposedApps() {
    File toolingDir = new File(muleDir + MULE_TOOLING_APPS);
    if (toolingDir.exists()) {
      assertThat(toolingDir.listFiles().length, is(0));
    }
  }

  protected List<ImmutablePair<String, String>> getStartupSystemProperties() {
    return emptyList();
  }

  protected boolean isStartMuleBeforeEachTest() {
    return true;
  }

  public static URL toUrl(URI uri) {
    try {
      return uri.toURL();
    } catch (MalformedURLException e) {
      throw new RuntimeException("Error while getting URL", e);
    }
  }

  protected void doWithToolingArtifact(ToolingRuntimeClient toolingRuntimeClient, URL applicationContentUrl,
                                       Map<String, String> artifactProperties,
                                       CheckedConsumer<ToolingArtifact> consumer) {
    ToolingArtifact toolingArtifact = null;
    try {
      toolingArtifact = toolingRuntimeClient.newToolingArtifact(applicationContentUrl, artifactProperties);
      consumer.accept(toolingArtifact);
    } finally {
      if (toolingArtifact != null) {
        toolingArtifact.dispose();
      }
    }
  }

  protected void doWithToolingArtifact(ToolingRuntimeClient toolingRuntimeClient, URL applicationContentUrl,
                                       CheckedConsumer<ToolingArtifact> consumer) {
    doWithToolingArtifact(toolingRuntimeClient, applicationContentUrl, emptyMap(), consumer);
  }

  protected void doWithToolingArtifact(ToolingRuntimeClient toolingRuntimeClient, URL applicationContentUrl,
                                       ToolingArtifact parent,
                                       CheckedConsumer<ToolingArtifact> consumer) {
    doWithToolingArtifact(toolingRuntimeClient, applicationContentUrl, parent, consumer);
  }

  protected void doWithToolingArtifact(ToolingRuntimeClient toolingRuntimeClient, URL applicationContentUrl,
                                       Map<String, String> artifactProperties,
                                       ToolingArtifact parent,
                                       CheckedConsumer<ToolingArtifact> consumer) {
    ToolingArtifact toolingArtifact = null;
    try {
      toolingArtifact = toolingRuntimeClient.newToolingArtifact(applicationContentUrl, artifactProperties, parent.getId());
      consumer.accept(toolingArtifact);
    } finally {
      if (toolingArtifact != null) {
        toolingArtifact.dispose();
      }
    }
  }

  protected void doWithFetchedToolingArtifact(ToolingRuntimeClient toolingRuntimeClient, String id,
                                              CheckedConsumer<ToolingArtifact> consumer) {
    ToolingArtifact toolingArtifact = null;
    try {
      toolingArtifact = toolingRuntimeClient.fetchToolingArtifact(id);
      consumer.accept(toolingArtifact);
    } finally {
      if (toolingArtifact != null) {
        toolingArtifact.dispose();
      }
    }
  }

  @FunctionalInterface
  public interface CheckedConsumer<T> extends Consumer<T> {

    @Override
    default void accept(T t) {
      try {
        acceptChecked(t);
      } catch (Exception e) {
        throw Throwables.propagate(e);
      }
    }

    void acceptChecked(T t) throws Exception;
  }

}

