/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.test.utils.matchers;

import static com.google.common.collect.Lists.newArrayList;
import static java.lang.String.format;
import static java.lang.System.lineSeparator;
import static java.util.Arrays.stream;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.CoreMatchers.containsString;

import java.util.List;

import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.StringDescription;
import org.hamcrest.TypeSafeMatcher;
import org.hamcrest.collection.IsIterableContainingInAnyOrder;

public class DataWeaveMatcher extends TypeSafeMatcher<String> {

  private static final String START_INTERNAL_MATCHER_ERROR = "[internal matcher error -> ";
  private static final String END_INTERNAL_MATCHER_ERROR = "]";
  private static final String AS_OBJECT = "as Object";
  private static final String AS_OBJECT_CLASS_MATCHER = "class: \"%s\"";
  private final String objectClassName;
  private final WeaveMatcherField[] weaveMatcherFields;
  private StringBuilder mismatchingDescription = new StringBuilder();

  public DataWeaveMatcher(String objectClassName, WeaveMatcherField... weaveMatcherFields) {
    this.objectClassName = objectClassName;
    this.weaveMatcherFields = weaveMatcherFields;
  }

  private Matcher withSomeEncoding() {
    return containsString(format(AS_OBJECT_CLASS_MATCHER, objectClassName));
  }

  @Override
  protected boolean matchesSafely(String object) {
    String[] weaveSplit = object.split(AS_OBJECT, 2);

    String asObject = weaveSplit[1].trim();
    boolean matches = internalEvaluate(asObject, withSomeEncoding());

    String[] fields = weaveSplit[0].split(",");
    List<String> fieldLists = newArrayList(fields);
    matches &= internalEvaluate(fieldLists, new IsIterableContainingInAnyOrder(stream(this.weaveMatcherFields)
        .map(weaveMatcherField -> weaveMatcherField.asMatcher())
        .collect(toList())));
    return matches;
  }

  private boolean internalEvaluate(Object object, Matcher matcher) {
    boolean matches = matcher.matches(object);
    if (!matches) {
      final StringDescription description = new StringDescription();
      matcher.describeMismatch(object, description);
      mismatchingDescription.append(lineSeparator());
      mismatchingDescription.append(START_INTERNAL_MATCHER_ERROR);
      mismatchingDescription.append(description);
      mismatchingDescription.append(END_INTERNAL_MATCHER_ERROR);
    }
    return matches;
  }

  @Override
  public void describeTo(Description description) {
    description.appendText(mismatchingDescription.toString());
  }

  public static DataWeaveMatcher weaveMatcher(String objectClassName, WeaveMatcherField... weaveMatcherFields) {
    return new DataWeaveMatcher(objectClassName, weaveMatcherFields);
  }

  public static class WeaveMatcherField {

    private final String fieldName;
    private final String fieldValue;
    private final String type;
    private final String className;

    private WeaveMatcherField(String fieldName, String fieldValue, String type, String className) {
      this.fieldName = fieldName;
      this.fieldValue = fieldValue;
      this.type = type;
      this.className = className;
    }

    Matcher asMatcher() {
      return containsString(format("%s: %s as %s {class: \"%s\"}", fieldName, fieldValue, type, className));
    }

    @Override
    public String toString() {
      return "WeaveMatcherField{" +
          "fieldName='" + fieldName + '\'' +
          ", fieldValue='" + fieldValue + '\'' +
          ", type='" + type + '\'' +
          ", className='" + className + '\'' +
          '}';
    }

    public static Builder newBuilder() {
      return new Builder();
    }

    public static class Builder {

      private String fieldName;
      private String fieldValue;
      private String type;
      private String className;

      public Builder withFieldName(String fieldName) {
        this.fieldName = fieldName;
        return this;
      }

      public Builder withFieldValue(String fieldValue) {
        this.fieldValue = fieldValue;
        return this;
      }

      public Builder asType(String type) {
        this.type = type;
        return this;
      }

      public Builder withClassName(String className) {
        this.className = className;
        return this;
      }

      public WeaveMatcherField build() {
        return new WeaveMatcherField(fieldName, fieldValue, type, className);
      }

    }

  }

}
