/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.test;

import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static org.mule.maven.client.test.MavenTestHelper.createDefaultEnterpriseMavenConfigurationBuilder;
import static org.mule.tooling.client.api.configuration.agent.AgentConfiguration.builder;
import static org.mule.tooling.client.test.utils.MavenUtils.getMuleVersion;
import static org.mule.tooling.client.test.utils.MavenUtils.getToolingVersion;

import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.internal.DefaultSettingsSupplierFactory;
import org.mule.maven.client.internal.MavenEnvironmentVariables;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrap;
import org.mule.tooling.client.test.junit4.ToolingJUnitTestRunner;
import org.mule.tooling.client.test.junit4.annotations.AgentPort;
import org.mule.tooling.client.test.junit4.annotations.AgentProtocol;
import org.mule.tooling.client.test.junit4.annotations.BootstrapVersions;
import org.mule.tooling.client.test.junit4.annotations.Log4JConfiguration;
import org.mule.tooling.client.test.junit4.annotations.MuleVersion;
import org.mule.tooling.client.test.junit4.annotations.RuntimeMavenConfiguration;
import org.mule.tooling.client.test.junit4.annotations.ToolingRuntimeBootstrap;
import org.mule.tooling.client.test.utils.CheckedConsumer;
import org.mule.tooling.client.test.utils.CheckedFunction;

import java.io.File;
import java.net.InetAddress;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.UnknownHostException;
import java.util.List;
import java.util.Map;

import org.junit.ClassRule;
import org.junit.runner.RunWith;

@RunWith(ToolingJUnitTestRunner.class)
public abstract class BaseToolingBootstrapTestCase {

  private static final String HTTP = "http";

  protected static final String AGENT_BASE_URL = "/mule/tooling";

  @ClassRule
  public static DynamicPort providedAgentPort = new DynamicPort("providedAgentPort");

  @MuleVersion
  protected static String muleVersion;

  @AgentPort
  protected static Integer agentPort;

  @AgentProtocol
  protected static String agentProtocol;

  @RuntimeMavenConfiguration
  protected static MavenConfiguration mavenConfiguration;

  @ToolingRuntimeBootstrap
  private static ToolingRuntimeClientBootstrap bootstrap;

  @AgentPort
  private static Integer provideAgentPort() {
    return providedAgentPort.getNumber();
  }

  @AgentProtocol
  private static String provideAgentProtocol() {
    return HTTP;
  }

  @RuntimeMavenConfiguration
  private static MavenConfiguration provideMavenConfiguration() throws Exception {
    DefaultSettingsSupplierFactory settingsSupplierFactory =
        new DefaultSettingsSupplierFactory(new MavenEnvironmentVariables());
    MavenConfiguration.MavenConfigurationBuilder mavenConfigurationBuilder =
        createDefaultEnterpriseMavenConfigurationBuilder().ignoreArtifactDescriptorRepositories(false);
    settingsSupplierFactory.environmentSettingsSecuritySupplier()
        .ifPresent(mavenConfigurationBuilder::settingsSecurityLocation);
    return mavenConfigurationBuilder.build();
  }

  @Log4JConfiguration
  private static URI provideLog4JConfigFile() throws URISyntaxException {
    return Thread.currentThread().getContextClassLoader().getResource("log4j2-tooling.xml").toURI();
  }

  @BootstrapVersions
  private static List<BootstrapVersionConfiguration> provideBootstrapVersions() {
    return singletonList(new BootstrapVersionConfiguration(getToolingVersion(), getMuleVersion()));
  }

  private final ToolingRuntimeClient defaultToolingRuntimeClient;

  public BaseToolingBootstrapTestCase() {
    this.defaultToolingRuntimeClient = getDefaultToolingClientBuilder().build();
  }

  protected ToolingRuntimeClient getDefaultToolingRuntimeClient() {
    return defaultToolingRuntimeClient;
  }

  protected ToolingRuntimeClient.Builder getDefaultToolingClientBuilder() {
    try {
      ToolingRuntimeClient.Builder toolingBuilder = bootstrap.getToolingRuntimeClientBuilderFactory().create();
      toolingBuilder.withMavenConfiguration(mavenConfiguration)
          .withRemoteAgentConfiguration(builder()
              .withDefaultConnectionTimeout(6000)
              .withDefaultReadTimeout(60000)
              .withMuleVersion(muleVersion)
              .withToolingApiUrl(new URL(agentProtocol, InetAddress.getLocalHost().getHostName(), agentPort, AGENT_BASE_URL))
              .build())
          .build();
      toolingBuilder.withTargetRuntimeVersion().ifEnabled(c -> c.perform(muleVersion));
      return toolingBuilder;
    } catch (MalformedURLException | UnknownHostException e) {
      throw new RuntimeException(e);
    }
  }

  protected ToolingRuntimeClient.Builder getBaseToolingClientBuilder() {
    return bootstrap.getToolingRuntimeClientBuilderFactory().create();
  }

  protected void doWithToolingArtifact(ToolingRuntimeClient toolingRuntimeClient,
                                       URL artifactUrl,
                                       ToolingArtifact parent,
                                       Map<String, String> artifactProperties,
                                       CheckedConsumer<ToolingArtifact> consumer) {
    ToolingArtifact toolingArtifact = null;
    try {
      if (parent == null) {
        toolingArtifact = toolingRuntimeClient.newToolingArtifact(artifactUrl, artifactProperties);
      } else {
        toolingArtifact = toolingRuntimeClient.newToolingArtifact(artifactUrl, artifactProperties, parent.getId());
      }
      consumer.accept(toolingArtifact);
    } finally {
      if (toolingArtifact != null) {
        toolingArtifact.dispose();
      }
    }
  }

  protected void doWithToolingArtifact(ToolingRuntimeClient toolingRuntimeClient,
                                       URL artifactUrl,
                                       Map<String, String> artifactProperties,
                                       CheckedConsumer<ToolingArtifact> consumer) {
    doWithToolingArtifact(toolingRuntimeClient, artifactUrl, null, artifactProperties, consumer);
  }

  protected void doWithToolingArtifact(ToolingRuntimeClient toolingRuntimeClient,
                                       String artifactLocation,
                                       ToolingArtifact parent,
                                       Map<String, String> artifactProperties,
                                       CheckedConsumer<ToolingArtifact> consumer) {

    doWithToolingArtifact(toolingRuntimeClient, getUrl(artifactLocation), parent, artifactProperties, consumer);
  }

  protected void doWithToolingArtifact(ToolingRuntimeClient toolingRuntimeClient,
                                       String artifactLocation,
                                       ToolingArtifact parent,
                                       CheckedConsumer<ToolingArtifact> consumer) {
    doWithToolingArtifact(toolingRuntimeClient, artifactLocation, parent, emptyMap(), consumer);
  }

  protected void doWithToolingArtifact(ToolingRuntimeClient toolingRuntimeClient,
                                       String artifactLocation,
                                       Map<String, String> artifactProperties,
                                       CheckedConsumer<ToolingArtifact> consumer) {
    doWithToolingArtifact(toolingRuntimeClient, artifactLocation, null, artifactProperties, consumer);
  }

  protected void doWithToolingArtifact(ToolingRuntimeClient toolingRuntimeClient,
                                       String artifactLocation,
                                       CheckedConsumer<ToolingArtifact> consumer) {
    doWithToolingArtifact(toolingRuntimeClient, artifactLocation, null, emptyMap(), consumer);
  }

  protected void doWithToolingArtifact(String artifactLocation,
                                       Map<String, String> artifactProperties,
                                       ToolingArtifact parent,
                                       CheckedConsumer<ToolingArtifact> consumer) {
    doWithToolingArtifact(defaultToolingRuntimeClient, artifactLocation, parent, artifactProperties, consumer);
  }

  protected void doWithToolingArtifact(String artifactLocation,
                                       ToolingArtifact parent,
                                       CheckedConsumer<ToolingArtifact> consumer) {
    doWithToolingArtifact(defaultToolingRuntimeClient, artifactLocation, parent, consumer);
  }

  protected void doWithToolingArtifact(String artifactLocation,
                                       Map<String, String> artifactProperties,
                                       CheckedConsumer<ToolingArtifact> consumer) {
    doWithToolingArtifact(defaultToolingRuntimeClient, artifactLocation, artifactProperties, consumer);
  }

  protected void doWithToolingArtifact(String artifactLocation,
                                       CheckedConsumer<ToolingArtifact> consumer) {
    doWithToolingArtifact(defaultToolingRuntimeClient, artifactLocation, consumer);
  }

  protected <T> T doWithToolingArtifact(ToolingRuntimeClient toolingRuntimeClient,
                                        String artifactLocation,
                                        ToolingArtifact parent,
                                        Map<String, String> artifactProperties,
                                        CheckedFunction<ToolingArtifact, T> function) {
    ToolingArtifact toolingArtifact = null;
    try {
      URL artifactUrl = getUrl(artifactLocation);
      if (parent == null) {
        toolingArtifact = toolingRuntimeClient.newToolingArtifact(artifactUrl, artifactProperties);
      } else {
        toolingArtifact = toolingRuntimeClient.newToolingArtifact(artifactUrl, artifactProperties, parent.getId());
      }
      return function.apply(toolingArtifact);
    } finally {
      if (toolingArtifact != null) {
        toolingArtifact.dispose();
      }
    }
  }

  protected <T> T doWithToolingArtifact(ToolingRuntimeClient toolingRuntimeClient,
                                        String artifactLocation,
                                        ToolingArtifact parent,
                                        CheckedFunction<ToolingArtifact, T> function) {
    return doWithToolingArtifact(toolingRuntimeClient, artifactLocation, parent, emptyMap(), function);
  }

  protected <T> T doWithToolingArtifact(ToolingRuntimeClient toolingRuntimeClient,
                                        String artifactLocation,
                                        Map<String, String> artifactProperties,
                                        CheckedFunction<ToolingArtifact, T> function) {
    return doWithToolingArtifact(toolingRuntimeClient, artifactLocation, null, artifactProperties, function);
  }

  protected <T> T doWithToolingArtifact(ToolingRuntimeClient toolingRuntimeClient,
                                        String artifactLocation,
                                        CheckedFunction<ToolingArtifact, T> function) {
    return doWithToolingArtifact(toolingRuntimeClient, artifactLocation, null, emptyMap(), function);
  }

  protected <T> T doWithToolingArtifact(String artifactLocation,
                                        Map<String, String> artifactProperties,
                                        ToolingArtifact parent,
                                        CheckedFunction<ToolingArtifact, T> function) {
    return doWithToolingArtifact(defaultToolingRuntimeClient, artifactLocation, parent, artifactProperties, function);
  }

  protected <T> T doWithToolingArtifact(String artifactLocation,
                                        ToolingArtifact parent,
                                        CheckedFunction<ToolingArtifact, T> function) {
    return doWithToolingArtifact(defaultToolingRuntimeClient, artifactLocation, parent, function);
  }

  protected <T> T doWithToolingArtifact(String artifactLocation,
                                        Map<String, String> artifactProperties,
                                        CheckedFunction<ToolingArtifact, T> function) {
    return doWithToolingArtifact(defaultToolingRuntimeClient, artifactLocation, artifactProperties, function);
  }

  protected <T> T doWithToolingArtifact(String artifactLocation,
                                        CheckedFunction<ToolingArtifact, T> function) {
    return doWithToolingArtifact(defaultToolingRuntimeClient, artifactLocation, function);
  }

  protected void doWithFetchedToolingArtifact(String id,
                                              CheckedConsumer<ToolingArtifact> consumer) {
    ToolingArtifact toolingArtifact = null;
    try {
      toolingArtifact = defaultToolingRuntimeClient.fetchToolingArtifact(id);
      consumer.accept(toolingArtifact);
    } finally {
      if (toolingArtifact != null) {
        toolingArtifact.dispose();
      }
    }
  }

  private URL toUrl(URI uri) {
    try {
      return uri.toURL();
    } catch (MalformedURLException e) {
      throw new RuntimeException("Error while getting URL", e);
    }
  }

  private URL getUrl(String artifactLocation) {
    try {
      File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
      File applicationLocation = new File(targetTestClassesFolder, artifactLocation);
      return toUrl(applicationLocation.toURI());
    } catch (URISyntaxException e) {
      throw new RuntimeException("Invalid artifact location");
    }
  }

}
