/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.test.runtime;

import static com.google.common.io.Files.createTempDir;
import static java.lang.Boolean.parseBoolean;
import static java.lang.Boolean.valueOf;
import static java.lang.String.format;
import static java.lang.String.join;
import static java.lang.System.getProperty;
import static java.lang.Thread.currentThread;
import static java.util.Objects.requireNonNull;
import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static org.apache.commons.io.FileUtils.copyDirectory;
import static org.apache.commons.io.FileUtils.copyFile;
import static org.apache.commons.io.FileUtils.copyInputStreamToFile;
import static org.apache.commons.io.FileUtils.deleteDirectory;
import static org.awaitility.Awaitility.with;
import static org.awaitility.Duration.FOREVER;
import static org.mule.maven.client.api.MavenClientProvider.discoverProvider;
import static org.mule.maven.client.test.MavenTestHelper.createDefaultEnterpriseMavenConfigurationBuilder;
import static org.mule.tooling.client.test.utils.MavenUtils.getToolingVersion;
import static org.mule.tooling.runtime.process.controller.MuleProcessControllerFactory.createController;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.tooling.client.api.extension.model.MuleVersion;
import org.mule.tooling.runtime.process.controller.MuleProcessController;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.net.ssl.SSLHandshakeException;
import org.awaitility.Duration;
import org.codehaus.plexus.archiver.tar.TarGZipUnArchiver;
import org.codehaus.plexus.archiver.zip.ZipUnArchiver;
import org.codehaus.plexus.logging.console.ConsoleLogger;
import org.slf4j.Logger;

/**
 * Manages an started MuleRuntime instance with an Agent installed.
 */
public class MuleRuntimeWithAgent {

  private static final String REST_AGENT_TRANSPORT_PORT_SYS_PROP = "-M-Drest.agent.transport.port";

  private static final String AGENT_BASE_URL = "/mule/tooling";
  private static final String HTTPS_PROTOCOL = "https";
  private static final String HTTP_PROTOCOL = "http";

  private static final Logger LOGGER = getLogger(MuleRuntimeWithAgent.class);

  public static MuleRuntimeWithAgent.Builder builder() {
    return new Builder();
  }

  private final File muleHome;
  private final int agentPort;
  private final String agentProtocol;
  private final MuleStandaloneConfiguration muleStandaloneConfiguration;
  private final List<String> startUpArgs;

  private boolean started = false;

  private final MuleProcessController muleProcessController;

  private MuleRuntimeWithAgent(File muleHome,
                               MuleStandaloneConfiguration muleStandaloneConfiguration,
                               int agentPort,
                               String agentProtocol,
                               List<String> startupArgs) {
    this.muleHome = muleHome;
    this.muleStandaloneConfiguration = muleStandaloneConfiguration;
    this.agentPort = agentPort;
    this.agentProtocol = agentProtocol;
    this.startUpArgs = startupArgs;
    this.muleProcessController = createController(muleHome, muleStandaloneConfiguration.getControllerOperationTimeout());
  }

  public File getMuleHome() {
    return this.muleHome;
  }

  public synchronized boolean isRunning() {
    return started;
  }

  public synchronized void start(List<String> extraStartupArgs) {
    if (started) {
      throw new RuntimeAlreadyStartedException("Runtime is already started. It needs to be stopped first to be restarted");
    }
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Starting muleRuntime from: " + muleHome);
    }
    List<String> allStartUpArgs = new ArrayList<>(startUpArgs);
    allStartUpArgs.add(REST_AGENT_TRANSPORT_PORT_SYS_PROP + "=" + agentPort);
    allStartUpArgs.addAll(extraStartupArgs);
    System
        .getProperties()
        .entrySet()
        .stream()
        .filter(e -> ((String) e.getKey()).startsWith("-M"))
        .forEach(p -> allStartUpArgs.add(p.getKey().toString() +
            (p.getValue() == null ? "" : "=" + p.getValue().toString())));

    if (parseBoolean(getProperty("muleRuntime.debug.enabled"))) {
      allStartUpArgs.add("-debug");
    }

    try {
      this.muleProcessController.start(allStartUpArgs.toArray(new String[0]));
      waitUntilRuntimeIsUpAndRunning();
      this.started = true;
    } catch (Exception e) {
      stop();
      throw new IllegalArgumentException("Could not start mule runtime", e);
    }
  }

  public synchronized void stop() {
    try {
      if (muleProcessController != null && muleProcessController.isRunning()) {
        muleProcessController.stop();
      }
      started = false;
    } catch (Exception e) {
      throw new RuntimeException("Could not stop Mule Runtime", e);
    }
  }

  private void waitUntilRuntimeIsUpAndRunning() throws MalformedURLException, UnknownHostException {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Waiting for Runtime to be operational...");
    }

    URL agentUrl = new URL(agentProtocol,
                           InetAddress.getLocalHost().getHostName(),
                           agentPort,
                           AGENT_BASE_URL);

    with()
        .timeout(muleStandaloneConfiguration.getStartTimeout() == 0 ? FOREVER
            : new Duration(muleStandaloneConfiguration.getStartTimeout(), MILLISECONDS))
        .pollInterval(muleStandaloneConfiguration.getStartPollInterval(), MILLISECONDS)
        .pollDelay(muleStandaloneConfiguration.getStartPollDelay(), MILLISECONDS)
        .await("Waiting for Remote Tooling Service to be operational")
        .until(() -> {
          final URLConnection urlConnection = agentUrl.openConnection();
          urlConnection.setConnectTimeout(200);
          urlConnection.setReadTimeout(200);
          try {
            urlConnection.connect();
            return true;
          } catch (Exception e) {
            return HTTPS_PROTOCOL.equals(agentProtocol) && (e instanceof SSLHandshakeException);
          }
        });
  }

  public static class Builder {

    private static final String AGENT_ARTIFACT_ID = "mule-agent-plugin";
    private static final String AGENT_CONFIG_FILE_NAME = "mule-agent.yml";
    private static final BundleDescriptor.Builder RUNTIME_DISTRO_BUNDLE_DESCRIPTOR_BUILDER = new BundleDescriptor.Builder()
        .setGroupId("com.mulesoft.mule.distributions")
        .setArtifactId("mule-ee-distribution-standalone")
        .setType("tar.gz");

    private static final BundleDescriptor.Builder AGENT_BUNDLE_DESCRIPTOR_BUILDER = new BundleDescriptor.Builder()
        .setGroupId("com.mulesoft.agent")
        .setArtifactId(AGENT_ARTIFACT_ID)
        .setClassifier("mule-server-plugin");

    private static final BundleDescriptor TOOLING_AGENT_PLUGIN_BUNDLE_DESCRIPTOR = new BundleDescriptor.Builder()
        .setGroupId("org.mule.tooling")
        .setArtifactId("tooling-agent-plugin")
        .setVersion(getToolingVersion())
        .build();

    private String muleVersion;

    private String agentVersion;
    private Integer agentPort;
    private String agentProtocol = HTTP_PROTOCOL;

    private MavenConfiguration mavenConfiguration;
    private MuleStandaloneConfiguration muleStandaloneConfiguration = new MuleStandaloneConfiguration(50000,
                                                                                                      500,
                                                                                                      300,
                                                                                                      15000);
    private File baseDir;
    private List<String> muleStartupArguments = new ArrayList<>();

    private Builder() {}

    public Builder withAgentVersion(String agentVersion) {
      this.agentVersion = agentVersion;
      return this;
    }

    public Builder withAgentPort(int agentPort) {
      this.agentPort = agentPort;
      return this;
    }

    public Builder withAgentProtocol(String protocol) {
      this.agentProtocol = protocol;
      return this;
    }

    public Builder withMavenConfiguration(MavenConfiguration mavenConfiguration) {
      this.mavenConfiguration = mavenConfiguration;
      return this;
    }

    public Builder withStandaloneConfiguration(MuleStandaloneConfiguration configuration) {
      this.muleStandaloneConfiguration = configuration;
      return this;
    }

    public Builder withMuleVersion(String muleVersion) {
      this.muleVersion = muleVersion;
      return this;
    }

    public Builder withBaseDirectory(File baseDir) {
      this.baseDir = baseDir;
      return this;
    }

    public Builder withRuntimeStartupArg(String arg) {
      this.muleStartupArguments.add(arg);
      return this;
    }

    public Builder withRuntimeStartupArg(String key, String value) {
      return this.withRuntimeStartupArg(format("%s=%s", key, value));
    }

    public MuleRuntimeWithAgent build() throws RuntimeAlreadyStartedException {
      requireNonNull(agentVersion, "agentVersion cannot be null");
      requireNonNull(muleVersion, "muleVersion cannot be null");
      requireNonNull(agentPort, "agentPort cannot be null");
      requireNonNull(mavenConfiguration, "mavenConfiguration cannot be null");

      if (baseDir == null) {
        baseDir = createTempDir();
      }

      File muleHome;
      try {
        // Create maven client for downloading Distro
        MavenClient mavenClient = discoverProvider(currentThread().getContextClassLoader())
            .createMavenClient(createDefaultEnterpriseMavenConfigurationBuilder().build());
        BundleDependency distroBundleDependency =
            mavenClient.resolveBundleDescriptor(RUNTIME_DISTRO_BUNDLE_DESCRIPTOR_BUILDER.setVersion(muleVersion).build());
        BundleDependency agentBundleDependency =
            mavenClient.resolveBundleDescriptor(AGENT_BUNDLE_DESCRIPTOR_BUILDER.setVersion(agentVersion).build());

        File distroTarGzFile = new File(distroBundleDependency.getBundleUri());
        // Extract into temporary file to be able to rename
        File extractedDistroParentFolder = createTempDir();

        TarGZipUnArchiver tarGzipUnarchiver = new TarGZipUnArchiver();
        tarGzipUnarchiver.enableLogging(new ConsoleLogger());
        tarGzipUnarchiver.setSourceFile(distroTarGzFile);
        tarGzipUnarchiver.setDestDirectory(extractedDistroParentFolder);
        tarGzipUnarchiver.extract();

        File runtimesFolder = new File(baseDir, "runtimes");
        File extractedDistroFolder = new File(runtimesFolder, muleVersion);
        // delete in case it was already created
        deleteDirectory(extractedDistroFolder);
        copyDirectory(extractedDistroParentFolder.listFiles()[0], extractedDistroFolder);
        deleteDirectory(extractedDistroParentFolder);

        File agentPluginFile = new File(agentBundleDependency.getBundleUri());
        File extractedDistroServerPlugins = new File(extractedDistroFolder, "server-plugins");
        File extractedAgentPluginFolder = new File(extractedDistroServerPlugins, join("-", AGENT_ARTIFACT_ID, agentVersion));
        extractedAgentPluginFolder.mkdir();

        // Extract into temporary file to be able to rename
        ZipUnArchiver zipUnArchiver = new ZipUnArchiver();
        zipUnArchiver.enableLogging(new ConsoleLogger());
        zipUnArchiver.setSourceFile(agentPluginFile);
        zipUnArchiver.setDestDirectory(extractedAgentPluginFolder);
        zipUnArchiver.extract();

        InputStream agentDefaultConfigInputStream =
            this.getClass().getClassLoader().getResourceAsStream("descriptors/" + AGENT_CONFIG_FILE_NAME);
        File agentConfigTargetFolder = new File(extractedDistroFolder, "conf");
        File agentConfigTargetFile = new File(agentConfigTargetFolder, AGENT_CONFIG_FILE_NAME);
        copyInputStreamToFile(agentDefaultConfigInputStream, agentConfigTargetFile);

        if (new MuleVersion(new MuleVersion(muleVersion).toCompleteNumericVersion()).atLeast("4.4.0")) {
          // Only copy tooling agent plugin if runtime is greater than 4.4.0
          BundleDependency toolingAgentPluginBundleDependency =
              mavenClient.resolveBundleDescriptor(TOOLING_AGENT_PLUGIN_BUNDLE_DESCRIPTOR);

          File agentPluginsFolder = new File(extractedAgentPluginFolder, "lib");
          File toolingAgentPluginFile = new File(toolingAgentPluginBundleDependency.getBundleUri());
          copyFile(toolingAgentPluginFile, new File(agentPluginsFolder, toolingAgentPluginFile.getName()));
        }

        File muleBinFolder = new File(extractedDistroFolder, "bin");
        Arrays.stream(muleBinFolder.listFiles()).forEach(f -> f.setExecutable(true));

        muleHome = extractedDistroFolder;
      } catch (Exception e) {
        throw new RuntimeException("Could not generate Runtime Distribution correctly", e);
      }

      withRuntimeStartupArg("-M-DmuleRuntimeConfig.maven.repositoryLocation",
                            mavenConfiguration.getLocalMavenRepositoryLocation().getAbsolutePath());
      mavenConfiguration.getUserSettingsLocation()
          .ifPresent(loc -> withRuntimeStartupArg("-M-DmuleRuntimeConfig.maven.userSettingsLocation", loc.getAbsolutePath()));
      mavenConfiguration.getSettingsSecurityLocation()
          .ifPresent(loc -> withRuntimeStartupArg("-M-DmuleRuntimeConfig.maven.settingsSecurityLocation", loc.getAbsolutePath()));
      mavenConfiguration.getGlobalSettingsLocation()
          .ifPresent(loc -> withRuntimeStartupArg("-M-DmuleRuntimeConfig.maven.globalSettingsLocation", loc.getAbsolutePath()));

      return new MuleRuntimeWithAgent(muleHome,
                                      muleStandaloneConfiguration,
                                      agentPort,
                                      agentProtocol,
                                      muleStartupArguments);
    }
  }
}
