/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.test.junit4;

import static java.lang.Thread.sleep;
import static java.nio.file.Files.createTempDirectory;
import static java.util.Collections.emptyList;
import static org.mule.maven.client.test.MavenTestUtils.getMavenProperty;
import static org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapFactory.newToolingRuntimeClientBootstrap;
import static org.mule.tooling.client.test.junit4.JunitStaticUtils.getConfiguredValue;
import static org.mule.tooling.client.test.junit4.JunitStaticUtils.getConfiguredValueOrFail;
import static org.mule.tooling.client.test.junit4.JunitStaticUtils.setFieldsIfPresent;
import static org.mule.tooling.client.test.utils.MavenUtils.pomFromParentOrJar;

import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrap;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapConfiguration;
import org.mule.tooling.client.test.BootstrapVersionConfiguration;
import org.mule.tooling.client.test.junit4.annotations.AgentPort;
import org.mule.tooling.client.test.junit4.annotations.AgentProtocol;
import org.mule.tooling.client.test.junit4.annotations.BootstrapVersions;
import org.mule.tooling.client.test.junit4.annotations.Log4JConfiguration;
import org.mule.tooling.client.test.junit4.annotations.MuleHome;
import org.mule.tooling.client.test.junit4.annotations.MuleVersion;
import org.mule.tooling.client.test.junit4.annotations.NeedsRuntime;
import org.mule.tooling.client.test.junit4.annotations.RuntimeMavenConfiguration;
import org.mule.tooling.client.test.junit4.annotations.RuntimeStartupArgs;
import org.mule.tooling.client.test.junit4.annotations.ToolingRuntimeBootstrap;
import org.mule.tooling.client.test.runtime.MuleRuntimeWithAgent;
import org.mule.tooling.client.test.runtime.MuleStandaloneConfiguration;

import java.io.File;
import java.io.IOException;
import java.lang.annotation.Annotation;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.runner.Runner;
import org.junit.runner.notification.RunNotifier;
import org.junit.runners.BlockJUnit4ClassRunner;
import org.junit.runners.Suite;
import org.junit.runners.model.FrameworkMethod;
import org.junit.runners.model.InitializationError;
import org.junit.runners.model.MultipleFailureException;
import org.junit.runners.model.Statement;

public class ToolingJUnitTestRunner extends Suite {

  private static final String MULESOFT_PUBLIC_REPOSITORY = "https://repository.mulesoft.org/nexus/content/repositories/public/";
  private static final String MULESOFT_PRIVATE_REPOSITORY = "https://repository.mulesoft.org/nexus/content/repositories/private/";
  private static final String MAVEN_CENTRAL = "https://repo.maven.apache.org/maven2/";
  private static final String MULE_TOOLING_APPS = "/.mule/tooling/apps";
  private static final String AGENT_VERSION =
      getMavenProperty("mule.agent.version", pomFromParentOrJar(ToolingJUnitTestRunner.class));

  private static final List<Runner> NO_RUNNERS = emptyList();

  private final List<Runner> runners;

  public ToolingJUnitTestRunner(Class<?> clazz) throws Throwable {
    super(clazz, NO_RUNNERS);
    this.runners = getRunners(clazz);
  }

  private List<Runner> getRunners(final Class<?> clazz) throws Throwable {
    final List<Throwable> errors = new ArrayList<>();
    final List<Runner> runners = new ArrayList<>();

    List<BootstrapVersionConfiguration> bootstrapVersions =
        getConfiguredValueOrFail(getTestClass(), BootstrapVersions.class, List.class);

    if (bootstrapVersions.isEmpty()) {
      throw new IllegalArgumentException(BootstrapVersions.class + " annotated method can't return an empty list");
    }

    for (BootstrapVersionConfiguration bootstrapVersion : bootstrapVersions) {
      try {
        addRunner(clazz, runners, bootstrapVersion.getMuleVersion(), bootstrapVersion.getToolingVersion(),
                  getTestClass().getAnnotation(NeedsRuntime.class) != null);
      } catch (Throwable t) {
        errors.add(t);
      }
    }

    if (!errors.isEmpty()) {
      throw new InitializationError(errors);
    }

    if (runners.isEmpty()) {
      throw new InitializationError(new Exception("No configured test mule runtime"));
    }

    return runners;
  }

  private void addRunner(Class<?> clazz,
                         List<Runner> runners,
                         String muleVersion,
                         String toolingVersion,
                         boolean needsRuntime)
      throws Exception {
    File targetFolder;
    URL targetLocation = clazz.getProtectionDomain().getCodeSource().getLocation();
    try {
      targetFolder = new File(targetLocation.toURI()).getParentFile();
    } catch (URISyntaxException e) {
      throw new IllegalArgumentException("Could not find target folder at: " + targetLocation);
    }

    String testName = createTestName(muleVersion, toolingVersion);

    runners.add(new SingleRunner(clazz, testName, muleVersion, toolingVersion, targetFolder, needsRuntime));
  }

  private static String createTestName(String muleVersion, String toolingVersion) {
    return "Mule-" + muleVersion + "/Tooling-" + toolingVersion;
  }

  @Override
  protected Statement classBlock(RunNotifier notifier) {
    return childrenInvoker(notifier);
  }

  @Override
  protected List<Runner> getChildren() {
    return runners;
  }

  /**
   * {@link BlockJUnit4ClassRunner} implementation that allows to run the same tests using different versions of Mule Runtime and
   * Tooling Runtime Client.
   */
  private static class SingleRunner extends BlockJUnit4ClassRunner {

    private final String name;
    private final String muleVersion;
    private final String toolingVersion;
    private final File baseDirectory;
    private final boolean needsRuntime;

    private MuleRuntimeWithAgent.Builder muleRuntimeWithAgentBuilder;
    private MuleRuntimeWithAgent muleRuntimeWithAgent;
    private ToolingRuntimeClientBootstrap bootstrap;

    SingleRunner(Class<?> klass,
                 String name,
                 String muleVersion,
                 String toolingVersion,
                 File baseDirectory,
                 boolean needsRuntime)
        throws InitializationError {
      super(klass);
      this.name = name;
      this.baseDirectory = baseDirectory;
      this.muleVersion = muleVersion;
      this.toolingVersion = toolingVersion;
      this.needsRuntime = needsRuntime;
    }

    private void configureStatic() {
      try {
        MavenConfiguration mavenConfiguration =
            getConfiguredValueOrFail(getTestClass(), RuntimeMavenConfiguration.class, MavenConfiguration.class);

        ToolingRuntimeClientBootstrapConfiguration.ToolingRuntimeClientBootstrapConfigurationBuilder bootstrapConfigurationBuilder =
            ToolingRuntimeClientBootstrapConfiguration.builder();
        bootstrapConfigurationBuilder.muleVersion(this.muleVersion);
        bootstrapConfigurationBuilder.toolingVersion(this.toolingVersion);
        getConfiguredValue(getTestClass(), Log4JConfiguration.class, URI.class)
            .ifPresent(bootstrapConfigurationBuilder::log4jConfiguration);
        bootstrapConfigurationBuilder.mavenConfiguration(mavenConfiguration);
        bootstrapConfigurationBuilder.workingFolder(createTempDirectory("workingDir").toFile());

        this.bootstrap = newToolingRuntimeClientBootstrap(bootstrapConfigurationBuilder.build());

        Integer agentPort;
        String agentProtocol;
        if (needsRuntime) {
          agentPort = getConfiguredValueOrFail(getTestClass(), AgentPort.class, Integer.class);
          agentProtocol = getConfiguredValueOrFail(getTestClass(), AgentProtocol.class, String.class);
          muleRuntimeWithAgentBuilder = MuleRuntimeWithAgent
              .builder()
              .withMuleVersion(muleVersion)
              .withBaseDirectory(baseDirectory)
              .withStandaloneConfiguration(new MuleStandaloneConfiguration(50000, 500, 300, 15000))
              .withMavenConfiguration(mavenConfiguration)
              .withAgentVersion(AGENT_VERSION)
              .withAgentPort(agentPort)
              .withAgentProtocol(agentProtocol)
              .withRuntimeStartupArg("-M-Dmule.testingMode", "true")
              .withRuntimeStartupArg("-M-DmuleRuntimeConfig.maven.repositories.mulesoft-public.url", MULESOFT_PUBLIC_REPOSITORY)
              .withRuntimeStartupArg("-M-DmuleRuntimeConfig.maven.repositories.mulesoft-private.url",
                                     MULESOFT_PRIVATE_REPOSITORY)
              .withRuntimeStartupArg("-M-DmuleRuntimeConfig.maven.repositories.mavenCentral.url", MAVEN_CENTRAL);
        } else {
          agentPort = getConfiguredValue(getTestClass(), AgentPort.class, Integer.class).orElse(null);
          agentProtocol = getConfiguredValue(getTestClass(), AgentProtocol.class, String.class).orElse(null);
        }

        setFieldsIfPresent(getTestClass(), RuntimeMavenConfiguration.class, MavenConfiguration.class, mavenConfiguration);
        setFieldsIfPresent(getTestClass(), AgentPort.class, Integer.class, agentPort);
        setFieldsIfPresent(getTestClass(), AgentProtocol.class, String.class, agentProtocol);
        setFieldsIfPresent(getTestClass(), ToolingRuntimeBootstrap.class, ToolingRuntimeClientBootstrap.class, bootstrap);
        setFieldsIfPresent(getTestClass(), MuleVersion.class, String.class, muleVersion);

      } catch (IOException | NoSuchMethodException e) {
        throw new RuntimeException(e);
      }
    }

    @Override
    protected Statement withBeforeClasses(Statement statement) {
      List<FrameworkMethod> befores = getTestClass().getAnnotatedMethods(BeforeClass.class);
      return new Statement() {

        @Override
        public void evaluate() throws Throwable {
          // Startup class. Done lazily to allow test to configure ClassRules
          configureStatic();

          // Configure everything related to the runtime if there is one
          if (muleRuntimeWithAgentBuilder != null) {
            muleRuntimeWithAgent = muleRuntimeWithAgentBuilder.build();
            setFieldsIfPresent(getTestClass(), MuleHome.class, File.class, muleRuntimeWithAgent.getMuleHome());
          }

          // Execute befores
          for (FrameworkMethod method : befores) {
            method.invokeExplosively(null);
          }

          // Start runtime with configured startupArgs. The method that provides startupArgs may depend on ClassRules or
          // @BeforeClass methods, we must wait until the last time to execute it
          if (muleRuntimeWithAgent != null) {
            List<String> extraStartupArgs =
                getConfiguredValue(getTestClass(), RuntimeStartupArgs.class, List.class).orElse(emptyList());
            muleRuntimeWithAgent.start(extraStartupArgs);
          }

          // Execute next statement
          statement.evaluate();
        }
      };
    }

    @Override
    protected Statement withAfterClasses(Statement statement) {
      List<FrameworkMethod> afters = getTestClass().getAnnotatedMethods(AfterClass.class);
      return new Statement() {

        @Override
        public void evaluate() throws Throwable {
          List<Throwable> errors = new ArrayList<>();
          try {
            // Invoke last statement
            statement.evaluate();

            // Give some time for async operations to complete. This is not OK but it should avoid some misleading error logs
            sleep(200);

            // Stop runtime and dispose bootstrap
            if (bootstrap != null) {
              bootstrap.dispose();
            }
            if (muleRuntimeWithAgent != null) {
              if (muleRuntimeWithAgent.isRunning()) {
                muleRuntimeWithAgent.stop();
              }
              File muleHome = muleRuntimeWithAgent.getMuleHome();
              File toolingDir = new File(muleHome + MULE_TOOLING_APPS);
              if (toolingDir.exists()) {
                if (toolingDir.listFiles().length > 0) {
                  throw new RuntimeException("Runtime in: " + muleHome + " was not cleaned up, applications remain");
                }
              }
            }

          } catch (Throwable e) {
            errors.add(e);
          } finally {
            for (FrameworkMethod each : afters) {
              try {
                each.invokeExplosively(null);
              } catch (Throwable e) {
                errors.add(e);
              }
            }
          }
          MultipleFailureException.assertEmpty(errors);
        }
      };
    }


    @Override
    protected String getName() {
      return "[" + name + "]";
    }

    @Override
    protected Annotation[] getRunnerAnnotations() {
      return new Annotation[0];
    }

    @Override
    protected Statement classBlock(RunNotifier notifier) {
      return super.classBlock(notifier);
    }
  }
}
