/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.tests.integration.tooling.client;

import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.startsWith;
import static org.junit.Assert.assertThat;
import static org.mule.tooling.client.test.utils.matchers.DataWeaveMatcher.WeaveMatcherField.newBuilder;
import static org.mule.tooling.client.test.utils.matchers.DataWeaveMatcher.weaveMatcher;
import org.mule.tooling.client.api.dataweave.DataWeavePreviewRequest;
import org.mule.tooling.client.api.dataweave.DataWeavePreviewResponse;
import org.mule.tooling.client.api.dataweave.DataWeaveService;
import org.mule.tooling.client.test.RuntimeType;
import org.mule.tooling.client.tests.integration.category.NeedMuleRuntimeTest;
import org.mule.tooling.event.model.DataTypeModel;
import org.mule.tooling.event.model.EventModel;
import org.mule.tooling.event.model.MessageModel;
import org.mule.tooling.event.model.TypedValueModel;

import com.google.common.collect.ImmutableMap;
import com.google.common.net.MediaType;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.function.Consumer;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Test;
import org.junit.experimental.categories.Category;

@Category(NeedMuleRuntimeTest.class)
@Feature("DataWeaveService")
@Story("Integration tests for dynamic DataSense resolution using ToolingBootstrap and ToolingRuntimeClient")
public class DataWeaveExecutionTestCase extends DataWeaveTestCase {

  private final String SCHEMAS_APP = "applications/dw-schemas";
  private static final int SMALL_TIMEOUT = 1;
  private static final int BIG_TIMEOUT = 10000;

  public DataWeaveExecutionTestCase(RuntimeType runtimeType) {
    super(runtimeType);
  }

  @Test
  @Description("Check simple script execution")
  public void dataWeaveSimpleScript() throws Exception {
    doWithToolingArtifact(localToolingRuntimeClient(), getApplicationUrlContent(POJO_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      final String expected = "{\n"
          + "  \"a\": 1\n"
          + "}";

      // ****************************************
      // Script
      // ****************************************

      final String script = "%dw 2.0\n"
          + "output application/json\n"
          + "---\n"
          + "{\n"
          + "a: 1\n"
          + "}";

      final DataWeavePreviewRequest build = DataWeavePreviewRequest.builder().withScript(script).build();

      final DataWeavePreviewResponse execute = dataWeaveService.execute(build);

      assertThat(execute.getResult(), is(expected));
    });
  }

  @Test
  @Description("Check that bindings are still present when no event/message/payload/variables is set, neither attributes")
  public void bindingsShouldNotFailWhenNoEventAndMessageIsPresentOnLocalRunner() throws Exception {
    doWithToolingArtifact(localToolingRuntimeClient(), getApplicationUrlContent(POJO_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      assertThat(dataWeaveService.execute(DataWeavePreviewRequest.builder().withScript("%dw 2.0\n"
          + "output application/json\n"
          + "---\n"
          + "{\n"
          + "aVariable: vars.someVar,\n"
          + "thePayload: payload,\n"
          + "theAttributes: attributes\n"
          + "}").build()).getResult(),
                 is("{\n"
                     + "  \"aVariable\": null,\n"
                     + "  \"thePayload\": null,\n"
                     + "  \"theAttributes\": null\n"
                     + "}"));


      // Setting event with message, variables and attributes
      TypedValueModel typedValue = TypedValueModel.builder()
          .withContent("value".getBytes())
          .withDataType(DataTypeModel.builder()
              .withType(String.class.getName())
              .withMediaType("*/*")
              .build())
          .build();

      final EventModel eventModel = EventModel.builder()
          .withMessage(MessageModel.builder()
              .withPayload(typedValue)
              .withAttributes(typedValue)
              .build())
          .withVariables(ImmutableMap.of("variable", typedValue))
          .build();

      assertThat(dataWeaveService.execute(DataWeavePreviewRequest.builder().withEvent(
                                                                                      eventModel)
          .withScript("%dw 2.0\n"
              + "output application/json\n"
              + "---\n"
              + "{\n"
              + "aVariable: vars.variable,\n"
              + "thePayload: payload,\n"
              + "theAttributes: attributes\n"
              + "}")
          .build()).getResult(),
                 is("{\n"
                     + "  \"aVariable\": \"value\",\n"
                     + "  \"thePayload\": \"value\",\n"
                     + "  \"theAttributes\": \"value\"\n"
                     + "}"));

    });
  }

  @Test
  @Description("Check that bindings are still present when no event/message/payload/variables is set, neither attributes")
  public void bindingsShouldNotFailWhenNoEventAndMessageIsPresentOnRemoteRunner() throws Exception {
    doWithToolingArtifact(remoteToolingRuntimeClient(), getApplicationUrlContent(POJO_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      assertThat(dataWeaveService.execute(DataWeavePreviewRequest.builder().withScript("%dw 2.0\n"
          + "output application/java\n"
          + "---\n"
          + "{\n"
          + "aVariable: vars.someVar,\n"
          + "thePayload: payload,\n"
          + "theAttributes: attributes\n"
          + "}").build()).getResult(),
                 startsWith("{\n"
                     + "  aVariable: null,\n"
                     + "  thePayload: null,\n"
                     + "  theAttributes: null\n"
                     + "}"));


      // Setting event with message, variables and attributes
      TypedValueModel typedValue = TypedValueModel.builder()
          .withContent("value".getBytes())
          .withDataType(DataTypeModel.builder()
              .withType(String.class.getName())
              .withMediaType("*/*")
              .build())
          .build();

      final EventModel eventModel = EventModel.builder()
          .withMessage(MessageModel.builder()
              .withPayload(typedValue)
              .withAttributes(typedValue)
              .build())
          .withVariables(ImmutableMap.of("variable", typedValue))
          .build();

      assertThat(dataWeaveService.execute(DataWeavePreviewRequest.builder().withEvent(
                                                                                      eventModel)
          .withScript("%dw 2.0\n"
              + "output application/java\n"
              + "---\n"
              + "{\n"
              + "aVariable: vars.variable,\n"
              + "thePayload: payload,\n"
              + "theAttributes: attributes\n"
              + "}")
          .build()).getResult(),
                 startsWith("{\n"
                     + "  aVariable: \"value\" as String {class: \"java.lang.String\"},\n"
                     + "  thePayload: \"value\" as String {class: \"java.lang.String\"},\n"
                     + "  theAttributes: \"value\" as String {class: \"java.lang.String\"}\n"
                     + "}"));

    });
  }


  @Test
  @Description("Check simple script execution that fails")
  public void dataWeaveSimpleScriptFailure() throws Exception {
    doWithToolingArtifact(remoteToolingRuntimeClient(), getApplicationUrlContent(POJO_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      // ****************************************
      // Script
      // ****************************************

      final String script = "%dw 2.0\n"
          + "output application/java\n"
          + "---\n"
          + "{\n"
          + "\"foo\" : (\"2014-01-01\" as Date) as Number\n"
          + "}";

      final DataWeavePreviewRequest build = DataWeavePreviewRequest.builder().withScript(script).build();

      DataWeavePreviewResponse response = dataWeaveService.execute(build);

      assertThat(response.isValid(), is(false));
      assertThat(response.getErrorMessage(), containsString("Cannot coerce Date (|2014-01-01|) to Number"));
    });
  }

  @Test
  @Description("Check java execution within context using weave modules")
  public void accessWeaveModules() throws Exception {
    doWithToolingArtifact(remoteToolingRuntimeClient(), getApplicationUrlContent(POJO_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      // ****************************************
      // Script
      // ****************************************

      final String script = "output application/java\n"
          + "import * from myLibraries::mylib\n"
          + "---\n"
          + "{\n" +
          "name: echo('Matias'),\n" +
          "age: 29\n" +
          "} as Object { class: \"pojo.Poyito\" }";

      final DataWeavePreviewRequest build = DataWeavePreviewRequest.builder().withScript(script).build();

      final DataWeavePreviewResponse execute = dataWeaveService.execute(build);

      assertThat(execute.getResult(), weaveMatcher("pojo.Poyito",
                                                   newBuilder()
                                                       .withFieldName("age")
                                                       .withFieldValue("29")
                                                       .asType("Number")
                                                       .withClassName("int")
                                                       .build(),
                                                   newBuilder()
                                                       .withFieldName("name")
                                                       .withFieldValue("\"Matias\"")
                                                       .asType("String")
                                                       .withClassName(String.class.getName())
                                                       .build()));
    });
  }

  @Test
  @Description("Mule p function should work locally")
  public void mulePropertiesFunctionRunsLocally() throws Exception {
    doWithToolingArtifact(localToolingRuntimeClient(), getApplicationUrlContent(POJO_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      final String expected = "someValue";

      // ****************************************
      // Script
      // ****************************************

      final String script = "%dw 2.0\n"
          + "output text/plain\n"
          + "---\n"
          + "p(\"appPropertyObject\")";

      final DataWeavePreviewRequest build = DataWeavePreviewRequest.builder().withScript(script).build();

      final DataWeavePreviewResponse execute = dataWeaveService.execute(build);

      assertThat(execute.getResult(), is(expected));
    });
  }

  @Test
  @Description("Mule p function should work locally")
  public void mulePropertiesFunctionRunsLocallyWithDefaults() throws Exception {
    doWithToolingArtifact(localToolingRuntimeClient(), getApplicationUrlContent(POJO_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      final String expected = "someValue";

      // ****************************************
      // Script
      // ****************************************

      final String script = "%dw 2.0\n"
          + "output text/plain\n"
          + "---\n"
          + "p('_xProperty') default 'someValue'";

      final DataWeavePreviewRequest build = DataWeavePreviewRequest.builder().withScript(script).build();

      final DataWeavePreviewResponse execute = dataWeaveService.execute(build);

      assertThat(execute.getResult(), is(expected));
    });
  }

  @Test
  @Description("Mule global bindings should run locally with mocked values")
  public void muleGlobalBindingsShouldWorkOnLocalExecution() throws Exception {
    doWithToolingArtifact(localToolingRuntimeClient(), getApplicationUrlContent(POJO_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      final String expected = "{\n"
          + "  \"authentication\": {\n"
          + "    \"credentials\": null,\n"
          + "    \"propertyKey\": null\n"
          + "  },\n"
          + "  \"correlationId\": null,\n"
          + "  \"flow\": {\n"
          + "    \"name\": null\n"
          + "  },\n"
          + "  \"itemSequenceInfo\": {\n"
          + "    \"position\": null,\n"
          + "    \"sequenceSize\": null\n"
          + "  }\n"
          + "}";

      // ****************************************
      // Script
      // ****************************************

      final String script = "%dw 2.0\n"
          + "output application/json\n"
          + "---\n"
          + "{"
          + " \"authentication\": {"
          + "     \"credentials\": authentication.credentials,"
          + "     \"propertyKey\": authentication.properties['key']"
          + "   },"
          + " \"correlationId\": correlationId,"
          + " \"flow\": { \"name\": flow.name },"
          + " \"itemSequenceInfo\" : {"
          + "     \"position\": itemSequenceInfo.position,"
          + "     \"sequenceSize\": itemSequenceInfo.sequenceSize"
          + "   }"
          + "}";

      final DataWeavePreviewRequest build = DataWeavePreviewRequest.builder().withScript(script).build();

      final DataWeavePreviewResponse execute = dataWeaveService.execute(build);

      assertThat(execute.getErrorMessage(), is(nullValue()));
      assertThat(execute.getResult(), is(expected));
    });
  }

  @Test
  @Description("Mule global bindings should run remote with mocked values")
  public void muleGlobalBindingsShouldWorkOnRemoteExecution() throws Exception {
    doWithToolingArtifact(remoteToolingRuntimeClient(), getApplicationUrlContent(POJO_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      final String expected = "{\n"
          + "  \"authentication.credentials\": null,\n"
          + "  \"authentication.propertyKey\": null,\n"
          + "  correlationId: null,\n"
          + "  \"flow.name\": null,\n"
          + "  \"itemSequenceInfo.position\": null,\n"
          + "  \"itemSequenceInfo.sequenceSize\": null\n"
          + "} as Object";

      // ****************************************
      // Script
      // ****************************************

      final String script = "%dw 2.0\n"
          + "output application/java\n"
          + "---\n"
          + "{"
          + " \"authentication.credentials\": authentication.credentials,"
          + " \"authentication.propertyKey\": authentication.properties['key'],"
          + " \"correlationId\": correlationId,"
          + " \"flow.name\": flow.name,"
          + " \"itemSequenceInfo.position\": itemSequenceInfo.position,"
          + " \"itemSequenceInfo.sequenceSize\": itemSequenceInfo.sequenceSize,"
          + "}";

      final DataWeavePreviewRequest build = DataWeavePreviewRequest.builder().withScript(script).build();

      final DataWeavePreviewResponse execute = dataWeaveService.execute(build);

      assertThat(execute.getErrorMessage(), is(nullValue()));
      // Encoding changes depending on OS
      assertThat(execute.getResult(), startsWith(expected));
    });
  }

  @Test
  @Description("Mule p function should work with secure properties")
  public void mulePropertiesFunctionWithSecureProperties() throws Exception {
    doWithToolingArtifact(localToolingRuntimeClient(), getApplicationUrlContent("applications/properties"),
                          ImmutableMap.of("mule.key", "blowkey"), toolingArtifact -> {
                            final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

                            final String expected = "Riquelme is Happy";

                            // ****************************************
                            // Script
                            // ****************************************

                            final String script = "%dw 2.0\n"
                                + "output text/plain\n"
                                + "---\n"
                                + "p(\"secure::encrypted.key\")";

                            final DataWeavePreviewRequest build = DataWeavePreviewRequest.builder().withScript(script).build();

                            final DataWeavePreviewResponse execute = dataWeaveService.execute(build);

                            assertThat(execute.getResult(), is(expected));
                          });
  }

  @Test
  @Description("Mule lookup function should fail locally as flow cannot be initialized during Tooling phases")
  public void muleLookUpFunctionFailsLocally() throws Exception {
    doWithToolingArtifact(localToolingRuntimeClient(), getApplicationUrlContent(POJO_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      // ****************************************
      // Script
      // ****************************************

      final String script = "%dw 2.0\n"
          + "output text/plain\n"
          + "---\n"
          + "lookup(\"myFlow\", \"somePayload\")";

      final DataWeavePreviewRequest build = DataWeavePreviewRequest.builder().withScript(script).build();

      final DataWeavePreviewResponse execute = dataWeaveService.execute(build);

      assertThat(execute.isValid(), is(false));
      assertThat(execute.getErrorMessage(), containsString("There is no component named 'myFlow'"));
    });
  }

  @Test
  @Description("Mule causeBy function should always return true locally")
  public void muleCauseByFunctionShouldAlwaysReturnTrueLocally() throws Exception {
    doWithToolingArtifact(localToolingRuntimeClient(), getApplicationUrlContent(POJO_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      // ****************************************
      // Script
      // ****************************************

      final String script = "%dw 2.0\n"
          + "output text/plain\n"
          + "---\n"
          + "causedBy(\"ANY\")";

      final DataWeavePreviewRequest build = DataWeavePreviewRequest.builder().withScript(script).build();

      final DataWeavePreviewResponse execute = dataWeaveService.execute(build);

      assertThat(execute.isValid(), is(true));
      assertThat(execute.getResult(), is("true"));
    });
  }

  @Test
  @Description("Check java execution within context")
  public void dataWeaveWithinContextShouldExecuteInAgent() throws Exception {
    doDataWeaveWithinContextShouldExecuteInAgent(POJO_APP_LOCATION);
  }

  private void doDataWeaveWithinContextShouldExecuteInAgent(String artifactContent) throws Exception {
    doWithToolingArtifact(remoteToolingRuntimeClient(), getApplicationUrlContent(artifactContent), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      // ****************************************
      // Script
      // ****************************************

      final String script = "output application/java\n"
          + "---\n"
          + "{\n" +
          "name: 'Matias',\n" +
          "age: 29\n" +
          "} as Object { class: \"pojo.Poyito\" }";

      final DataWeavePreviewRequest build = DataWeavePreviewRequest.builder().withScript(script).build();

      final DataWeavePreviewResponse execute = dataWeaveService.execute(build);

      assertThat(execute.getResult(), weaveMatcher("pojo.Poyito",
                                                   newBuilder()
                                                       .withFieldName("age")
                                                       .withFieldValue("29")
                                                       .asType("Number")
                                                       .withClassName("int")
                                                       .build(),
                                                   newBuilder()
                                                       .withFieldName("name")
                                                       .withFieldValue("\"Matias\"")
                                                       .asType("String")
                                                       .withClassName(String.class.getName())
                                                       .build()));
    });
  }

  @Test
  @Description("Check java execution within context works also on Domains")
  public void dataWeaveWithinContextShouldExecuteInAgentUsingDomains() throws Exception {
    doDataWeaveWithinContextShouldExecuteInAgent(POJO_DOMAIN_LOCATION);
  }

  @Test
  @Description("Check java execution within context with Java output")
  public void dataWeaveWithJavaOutput() throws Exception {
    doWithToolingArtifact(remoteToolingRuntimeClient(), getApplicationUrlContent(POJO_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      // ****************************************
      // Input
      // ****************************************

      final String inputPayload = "{\n"
          + "  \"name\": \"Miguel\",\n"
          + "  \"age\": 28\n"
          + "}";

      final TypedValueModel payload = TypedValueModel
          .builder()
          .withContent(
                       inputPayload.getBytes("UTF-8"))
          .withDataType(
                        DataTypeModel
                            .builder()
                            .withType("")
                            .withMediaType("application/json")
                            .build())
          .build();
      final MessageModel message = MessageModel
          .builder()
          .withPayload(payload)
          .build();
      final EventModel event = EventModel.builder().withMessage(message).build();

      // ****************************************
      // Script
      // ****************************************

      final String script = "output application/java\n"
          + "---\n"
          + "payload";

      final DataWeavePreviewRequest build = DataWeavePreviewRequest.builder().withEvent(event).withScript(script).build();

      final DataWeavePreviewResponse execute = dataWeaveService.execute(build);

      assertThat(execute.getResult(), weaveMatcher(LinkedHashMap.class.getName(),
                                                   newBuilder()
                                                       .withFieldName("age")
                                                       .withFieldValue("28")
                                                       .asType("Number")
                                                       .withClassName(Integer.class.getName())
                                                       .build(),
                                                   newBuilder()
                                                       .withFieldName("name")
                                                       .withFieldValue("\"Miguel\"")
                                                       .asType("String")
                                                       .withClassName(String.class.getName())
                                                       .build()));
    });
  }

  @Test
  public void messageShouldBePresentOnBindingContext() throws Exception {
    doWithToolingArtifact(remoteToolingRuntimeClient(), getApplicationUrlContent(POJO_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      // ****************************************
      // Input
      // ****************************************

      final String inputPayload = "{\n"
          + "  \"name\": \"Miguel\",\n"
          + "  \"age\": 28\n"
          + "}";

      final TypedValueModel payload = TypedValueModel
          .builder()
          .withContent(
                       inputPayload.getBytes("UTF-8"))
          .withDataType(
                        DataTypeModel
                            .builder()
                            .withType("")
                            .withMediaType("application/json")
                            .build())
          .build();
      final MessageModel message = MessageModel
          .builder()
          .withPayload(payload)
          .build();
      final EventModel event = EventModel.builder().withMessage(message).build();

      // ****************************************
      // Script
      // ****************************************

      final String script = "output application/java\n"
          + "---\n"
          + "message.payload";

      final DataWeavePreviewRequest build = DataWeavePreviewRequest.builder().withEvent(event).withScript(script).build();

      final DataWeavePreviewResponse execute = dataWeaveService.execute(build);

      assertThat(execute.getResult(), weaveMatcher(LinkedHashMap.class.getName(),
                                                   newBuilder()
                                                       .withFieldName("age")
                                                       .withFieldValue("28")
                                                       .asType("Number")
                                                       .withClassName(Integer.class.getName())
                                                       .build(),
                                                   newBuilder()
                                                       .withFieldName("name")
                                                       .withFieldValue("\"Miguel\"")
                                                       .asType("String")
                                                       .withClassName(String.class.getName())
                                                       .build()));
    });
  }


  @Test
  @Description("Check java execution within context with json (local) and check dataType binding")
  public void dataWeaveWithJsonOutputForDataTypeBinding() throws Exception {
    doWithToolingArtifact(localToolingRuntimeClient(), getApplicationUrlContent(POJO_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      // ****************************************
      // Input
      // ****************************************

      final String inputPayload = "{\n"
          + "  \"name\": \"Miguel\",\n"
          + "  \"age\": 28\n"
          + "}";

      final TypedValueModel payload = TypedValueModel
          .builder()
          .withContent(
                       inputPayload.getBytes("UTF-8"))
          .withDataType(
                        DataTypeModel
                            .builder()
                            .withType(String.class.getName())
                            .withMediaType(MediaType.JSON_UTF_8.toString())
                            .build())
          .build();
      final MessageModel message = MessageModel
          .builder()
          .withPayload(payload)
          .build();
      final EventModel event = EventModel.builder().withMessage(message).build();

      // ****************************************
      // Script
      // ****************************************

      final String script = "output application/json\n"
          + "---\n"
          + "dataType";

      final DataWeavePreviewRequest build = DataWeavePreviewRequest.builder().withEvent(event).withScript(script).build();

      final DataWeavePreviewResponse execute = dataWeaveService.execute(build);

      assertThat(execute.getResult(), equalTo("{\n"
          + "  \"type\": \"java.lang.String\",\n"
          + "  \"streamType\": false,\n"
          + "  \"mediaType\": {\n"
          + "    \"primaryType\": \"application\",\n"
          + "    \"charset\": {\n"
          + "      \"registered\": true\n"
          + "    },\n"
          + "    \"subType\": \"json\"\n"
          + "  }\n"
          + "}"));
    });

  }

  @Test
  @Description("Check java execution within context with Java output (remote) and check dataType binding")
  public void dataWeaveWithJavaOutputDataType() throws Exception {
    doWithToolingArtifact(remoteToolingRuntimeClient(), getApplicationUrlContent(POJO_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      // ****************************************
      // Input
      // ****************************************

      final String inputPayload = "{\n"
          + "  \"name\": \"Miguel\",\n"
          + "  \"age\": 28\n"
          + "}";

      final TypedValueModel payload = TypedValueModel
          .builder()
          .withContent(
                       inputPayload.getBytes("UTF-8"))
          .withDataType(
                        DataTypeModel
                            .builder()
                            .withType(String.class.getName())
                            .withMediaType(MediaType.JSON_UTF_8.toString())
                            .build())
          .build();
      final MessageModel message = MessageModel
          .builder()
          .withPayload(payload)
          .build();
      final EventModel event = EventModel.builder().withMessage(message).build();

      // ****************************************
      // Script
      // ****************************************

      final String script = "output application/java\n"
          + "---\n"
          + "dataType";

      final DataWeavePreviewRequest build = DataWeavePreviewRequest.builder().withEvent(event).withScript(script).build();

      final DataWeavePreviewResponse execute = dataWeaveService.execute(build);

      // Partial assertion as the encoding depends on the Runtime
      assertThat(execute.getResult(), startsWith("{\n"
          + "  \"type\": \"java.lang.String\" as String {class: \"java.lang.Class\"},\n"
          + "  streamType: false as Boolean {class: \"boolean\"},\n"
          + "  mediaType: {\n"
          + "    primaryType: \"application\" as String {class: \"java.lang.String\"},\n"
          + "    charset: {\n"
          + "      registered: true as Boolean {class: \"boolean\"}\n"
          + "    } as Object {class: \"java.util.Optional\"},\n"
          + "    subType: \"json\" as String {class: \"java.lang.String\"}\n"
          + "  } as Object {class: \"org.mule.runtime.api.metadata.MediaType\"}\n"
          + "} as Object {mediaType: \"*/*\", encoding: "));
    });
  }

  @Test
  @Description("DataType can be resolved with local runner")
  public void dataWeaveWithLocalDataType() throws Exception {
    doWithToolingArtifact(localToolingRuntimeClient(), getApplicationUrlContent(POJO_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      // ****************************************
      // Input
      // ****************************************

      final String inputPayload = "{\n"
          + "  \"name\": \"Miguel\",\n"
          + "  \"age\": 28\n"
          + "}";

      final TypedValueModel payload = TypedValueModel
          .builder()
          .withContent(
                       inputPayload.getBytes("UTF-8"))
          .withDataType(
                        DataTypeModel
                            .builder()
                            .withType(String.class.getName())
                            .withMediaType(MediaType.JSON_UTF_8.toString())
                            .build())
          .build();
      final MessageModel message = MessageModel
          .builder()
          .withPayload(payload)
          .build();
      final EventModel event = EventModel.builder().withMessage(message).build();

      // ****************************************
      // Script
      // ****************************************

      final String script = "output application/dw ignoreSchema=true\n"
          + "---\n"
          + "dataType";

      final DataWeavePreviewRequest build = DataWeavePreviewRequest.builder().withEvent(event).withScript(script).build();

      final DataWeavePreviewResponse execute = dataWeaveService.execute(build);

      assertThat(execute.getResult(), equalTo("{\n"
          + "  \"type\": \"java.lang.String\",\n"
          + "  streamType: false,\n"
          + "  mediaType: {\n"
          + "    primaryType: \"application\",\n"
          + "    charset: {\n"
          + "      registered: true\n"
          + "    },\n"
          + "    subType: \"json\"\n"
          + "  }\n"
          + "}"));
    });
  }

  @Test
  @Description("Check simple script execution with non-Java input payload")
  public void dataWeaveSimpleScriptWithNonJavaInputPayload() throws Exception {
    doWithToolingArtifact(localToolingRuntimeClient(), getApplicationUrlContent(POJO_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      final String expected = "{\n  \"name\": \"Miguel\"\n}";

      // ****************************************
      // Input
      // ****************************************

      final TypedValueModel payload = TypedValueModel
          .builder()
          .withContent(expected
              .getBytes("UTF-8"))
          .withDataType(
                        DataTypeModel
                            .builder()
                            .withType("")
                            .withMediaType("application/json")
                            .build())
          .build();
      final MessageModel message = MessageModel
          .builder()
          .withPayload(payload)
          .build();
      final EventModel event = EventModel.builder().withMessage(message).build();

      // ****************************************
      // Script
      // ****************************************

      final String script = "%dw 2.0\n"
          + "output application/json\n"
          + "---\n"
          + "payload";

      final DataWeavePreviewRequest build = DataWeavePreviewRequest.builder().withEvent(event).withScript(script).build();

      final DataWeavePreviewResponse execute = dataWeaveService.execute(build);

      assertThat(execute.getErrorMessage(), execute.isValid(), is(true));
      assertThat(execute.getResult(), is(expected));
    });
  }

  @Test
  @Description("Check script execution that uses flatfile schema directives as output works locally and can access local application resources")
  public void flatFileOutput() throws Exception {
    doWithToolingArtifact(localToolingRuntimeClient(), getApplicationUrlContent(SCHEMAS_APP), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      StringBuilder expected = new StringBuilder();
      expected.append("100   ");
      appendNullCharacter(expected, 9);
      appendNullCharacter(expected, 926);
      expected.append("\n");

      // ****************************************
      // Script
      // ****************************************

      final String script = "%dw 2.0\n"
          + "output application/flatfile schemaPath = \"schemas/claim.ffd\", segmentIdent = \"O5400SA-OUTPUT-TOTAL-RECORD\"\n"
          + "---\n"
          + "[{\n"
          + "  \"O5400SA-CAPTION\": \"100\"\n"
          + "}]";

      final DataWeavePreviewRequest build = DataWeavePreviewRequest.builder().withScript(script).build();

      final DataWeavePreviewResponse execute = dataWeaveService.execute(build);

      assertThat(execute.getResult(), is(expected.toString()));
    });
  }

  private void appendNullCharacter(StringBuilder builder, int i) {
    for (int j = 1; j <= i; j++) {
      builder.append("\u0000");
    }
  }

  @Test
  @Description("Check script execution that uses flatfile fixedWith directives as output works locally and can access local application resources")
  public void fixedWidthOutput() throws Exception {
    doWithToolingArtifact(localToolingRuntimeClient(), getApplicationUrlContent(SCHEMAS_APP), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      final String expected = "0         1         2         3         4         \n";

      // ****************************************
      // Script
      // ****************************************

      final String script = "%dw 2.0\n"
          + "output application/flatfile schemaPath = \"fixedwidth.ffd\"\n"
          + "---\n"
          + "[{\n"
          + "  field_0: \"0\",\n"
          + "  field_1: \"1\",\n"
          + "  field_2: \"2\",\n"
          + "  field_3: \"3\",\n"
          + "  field_4: \"4\"\n"
          + "}]";

      final DataWeavePreviewRequest build = DataWeavePreviewRequest.builder().withScript(script).build();

      final DataWeavePreviewResponse execute = dataWeaveService.execute(build);

      assertThat(execute.getResult(), is(expected));
    });
  }

  @Test(expected = IllegalArgumentException.class)
  @Description("Check script execution with Java input payload should throw an error")
  public void dataWeaveWithJavaInputPayloadShouldThrowError() throws Exception {
    doWithToolingArtifact(localToolingRuntimeClient(), getApplicationUrlContent(POJO_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      // ****************************************
      // Input
      // ****************************************

      final TypedValueModel payload = TypedValueModel
          .builder()
          .withContent(
                       new byte[0])
          .withDataType(
                        DataTypeModel
                            .builder()
                            .withType("")
                            .withMediaType("application/java")
                            .build())
          .build();
      final MessageModel message = MessageModel
          .builder()
          .withPayload(payload)
          .build();
      final EventModel event = EventModel.builder().withMessage(message).build();

      // ****************************************
      // Script
      // ****************************************

      final String script = "";

      final DataWeavePreviewRequest build = DataWeavePreviewRequest.builder().withEvent(event).withScript(script).build();

      dataWeaveService.execute(build);
    });
  }

  @Test
  @Description("Check simple script execution with DW script input payload with Java output")
  public void dataWeaveSimpleScriptWithDwScriptInputPayloadWithJavaOutput() throws Exception {
    doWithToolingArtifact(remoteToolingRuntimeClient(), getApplicationUrlContent(POJO_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      final String expected = "\"Miguel\"";

      // ****************************************
      // Input
      // ****************************************

      final String inputPayload = "%dw 2.0\n"
          + "output application/java\n"
          + "---\n"
          + "{\n"
          + "  name: \"Miguel\",\n"
          + "  age: 28\n"
          + "} as Object { class: \"pojo.Poyito\" }";

      final TypedValueModel payload = TypedValueModel
          .builder()
          .withContent(
                       inputPayload.getBytes("UTF-8"))
          .withDataType(
                        DataTypeModel
                            .builder()
                            .withType("")
                            .withMediaType("application/dw")
                            .build())
          .build();
      final MessageModel message = MessageModel
          .builder()
          .withPayload(payload)
          .build();
      final EventModel event = EventModel.builder().withMessage(message).build();

      // ****************************************
      // Script
      // ****************************************

      final String script = "%dw 2.0\n"
          + "output application/json\n"
          + "---\n"
          + "payload.name";

      final DataWeavePreviewRequest build = DataWeavePreviewRequest.builder().withEvent(event).withScript(script).build();

      final DataWeavePreviewResponse execute = dataWeaveService.execute(build);

      assertThat(execute.getErrorMessage(), execute.isValid(), is(true));
      assertThat(execute.getResult(), is(expected));
    });
  }

  @Test
  @Description("Check simple script execution with DW script input variable with Java output")
  public void dataWeaveScriptWithDwScriptInputVariableAndJavaOutput() throws Exception {
    doWithToolingArtifact(remoteToolingRuntimeClient(), getApplicationUrlContent(POJO_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      final String expected = "\"Miguel\"";

      // ****************************************
      // Input
      // ****************************************

      final String inputVariable = "%dw 2.0\n"
          + "output application/java\n"
          + "---\n"
          + "{\n"
          + "  name: \"Miguel\",\n"
          + "  age: 28\n"
          + "} as Object { class: \"pojo.Poyito\" }";

      final TypedValueModel variable = TypedValueModel
          .builder()
          .withContent(
                       inputVariable.getBytes("UTF-8"))
          .withDataType(
                        DataTypeModel
                            .builder()
                            .withType("")
                            .withMediaType("application/dw")
                            .build())
          .build();
      final Map<String, TypedValueModel> variables = new HashMap<>();
      variables.put("var1", variable);
      final MessageModel message = MessageModel.builder().build();
      final EventModel event = EventModel.builder().withMessage(message).withVariables(variables).build();

      // ****************************************
      // Script
      // ****************************************

      final String script = "%dw 2.0\n"
          + "output application/json\n"
          + "---\n"
          + "vars.var1.name";

      final DataWeavePreviewRequest build = DataWeavePreviewRequest.builder().withEvent(event).withScript(script).build();

      final DataWeavePreviewResponse execute = dataWeaveService.execute(build);

      assertThat(execute.getErrorMessage(), execute.isValid(), is(true));
      assertThat(execute.getResult(), is(expected));
    });
  }

  @Test
  @Description("Check simple script execution with DW script input payload without Java output")
  public void dataWeaveScriptWithDwScriptInputPayloadAndNonJavaOutput() throws Exception {
    doWithToolingArtifact(localToolingRuntimeClient(), getApplicationUrlContent(POJO_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      final String expected = "\"Miguel\"";

      // ****************************************
      // Input
      // ****************************************

      final String inputPayload = "%dw 2.0\n"
          + "output application/json\n"
          + "---\n"
          + "{\n"
          + "  name: \"Miguel\",\n"
          + "  age: 28\n"
          + "}";

      final TypedValueModel payload = TypedValueModel
          .builder()
          .withContent(
                       inputPayload.getBytes("UTF-8"))
          .withDataType(
                        DataTypeModel
                            .builder()
                            .withType("")
                            .withMediaType("application/dw")
                            .build())
          .build();
      final MessageModel message = MessageModel
          .builder()
          .withPayload(payload)
          .build();
      final EventModel event = EventModel.builder().withMessage(message).build();

      // ****************************************
      // Script
      // ****************************************

      final String script = "%dw 2.0\n"
          + "output application/json\n"
          + "---\n"
          + "payload.name";

      final DataWeavePreviewRequest build = DataWeavePreviewRequest.builder().withEvent(event).withScript(script).build();

      final DataWeavePreviewResponse execute = dataWeaveService.execute(build);

      assertThat(execute.getErrorMessage(), execute.isValid(), is(true));
      assertThat(execute.getResult(), is(expected));
    });
  }

  public void withTimeout(int timeout, Consumer<DataWeavePreviewResponse> assertion) throws Exception {
    doWithToolingArtifact(localToolingRuntimeClient(), getApplicationUrlContent(POJO_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      // ****************************************
      // Script
      // ****************************************

      final String script = "%dw 2.0\n"
          + "output application/json\n"
          + "---\n"
          + "{\n"
          + "a: 1\n"
          + "}";

      final DataWeavePreviewRequest build =
          DataWeavePreviewRequest.builder().withScript(script).withRequestTimeout(timeout).build();

      assertion.accept(dataWeaveService.execute(build));
    });
  }

  @Test
  @Description("Checks simple script with big timeout executes correctly")
  public void dataWeaveBigTimeout() throws Exception {
    withTimeout(BIG_TIMEOUT, response -> assertThat(response.isValid(), is(true)));
  }

  @Test
  @Description("Checks simple script with small timeout terminates with execution error")
  public void dataWeaveSmallTimeout() throws Exception {
    withTimeout(SMALL_TIMEOUT, response -> {
      assertThat(response.isValid(), is(false));
      assertThat(response.getErrorMessage(), containsString("Execution took more than "));
    });
  }
}
