/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.tests.integration.tooling.client;

import static java.util.Collections.emptyMap;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.collection.IsArrayWithSize.arrayWithSize;
import static org.junit.Assert.assertThat;
import static org.mule.tooling.client.api.configuration.agent.AgentConfiguration.builder;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;
import org.mule.tooling.client.api.configuration.ssl.KeyStoreConfig;
import org.mule.tooling.client.api.configuration.ssl.SslConfiguration;
import org.mule.tooling.client.api.configuration.ssl.TrustStoreConfig;
import org.mule.tooling.client.api.connectivity.ConnectionValidationResult;
import org.mule.tooling.client.api.connectivity.ConnectivityTestingRequest;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrap;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapConfiguration;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapFactory;
import org.mule.tooling.client.test.AbstractMuleRuntimeTestCase;
import org.mule.tooling.client.test.RuntimeType;
import org.mule.tooling.client.tests.integration.category.NeedMuleRuntimeTest;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Arrays;
import java.util.Collection;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.filefilter.WildcardFileFilter;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.runners.Parameterized;

@Category(NeedMuleRuntimeTest.class)
@Feature("TLS Support")
@Story("Tests for TLS Support in the Agent Configuration")
public class TLSClientConfigurationTestCase extends AbstractMuleRuntimeTestCase {

  private static final String CONF_FOLDER = "/conf";
  private static final String AGENT_FILE = "/mule-agent.yml";
  private static final String AGENT_FILE_MOVE_TO = "/mule-agent-default.yml";
  private static final String HTTPS = "https";
  private static final String RESOURCES = "src/test/resources/tls";
  private static final String KEYSTORE_FILE = "/rmakeystore.jks";
  private static final String TRUSTSTORE_FILE = "/trustStore.jks";
  private static final String FILE_APP_LOCATION = "applications/file";
  private static final String PASSWORD = "mulepassword";

  private static File muleAgentFile;
  private static File tempDefaultMuleAgentFile;
  private static File keyStore;
  private static File trustStore;

  private ToolingRuntimeClientBootstrap bootstrap;
  private ToolingArtifact toolingArtifact;

  public TLSClientConfigurationTestCase(RuntimeType runtimeType) {
    super(runtimeType);
  }

  @Override
  protected String getProtocol() {
    return HTTPS;
  }

  @Parameterized.Parameters(name = "{0}")
  public static Collection<Object[]> data() {
    return Arrays.asList(new Object[][] {
        {RuntimeType.REMOTE}
    });
  }

  @Override
  protected void cleanUpRemoteServerTlsConfiguration() {
    // Do nothing, we need TLS
  }

  @BeforeClass
  public static void setUpMuleAgentConfig() throws IOException, URISyntaxException {
    // This is a workaround due to AGENT-404, for which we cannot configure TLS for the Agent as a System Property
    File targetFolder = new File(TLSClientConfigurationTestCase.class.getProtectionDomain().getCodeSource().getLocation().toURI())
        .getParentFile();
    File runtimesFolder = new File(targetFolder, RUNTIMES_FOLDER);
    String[] runtimes = runtimesFolder.list(new WildcardFileFilter("mule*4.1*SNAPSHOT"));
    //TODO MTS-21 migrate all the tests!
    if (runtimes.length == 0) {
      muleDir = new File(runtimesFolder, runtimes[0]).getAbsolutePath();
    }
    assertThat(runtimes, arrayWithSize(1));

    muleAgentFile = new File(runtimesFolder, runtimes[0] + CONF_FOLDER + AGENT_FILE);
    File confFile = muleAgentFile.getParentFile();

    tempDefaultMuleAgentFile = new File(muleAgentFile.getParentFile().getAbsolutePath() + AGENT_FILE_MOVE_TO);
    muleAgentFile.renameTo(tempDefaultMuleAgentFile);

    File muleAgentNewConfig = new File(RESOURCES + AGENT_FILE);
    File testkeyStore = new File(RESOURCES + KEYSTORE_FILE);
    File testtrustStore = new File(RESOURCES + TRUSTSTORE_FILE);

    keyStore = new File(confFile.getAbsolutePath() + KEYSTORE_FILE);

    // truststore name must be hardcoded as is:
    trustStore = new File(confFile.getAbsolutePath() + "/anypoint-truststore.jks");

    FileUtils.copyFile(muleAgentNewConfig, muleAgentFile);
    FileUtils.copyFile(testkeyStore, keyStore);
    FileUtils.copyFile(testtrustStore, trustStore);
  }

  @AfterClass
  public static void tearDownMuleAgentConfig() {
    keyStore.delete();
    trustStore.delete();
    tempDefaultMuleAgentFile.renameTo(muleAgentFile);
  }

  private SslConfiguration getSslConfiguration() throws Exception {
    return SslConfiguration.builder()
        .withKeyStoreConfig(KeyStoreConfig.builder()
            .withKeyStoreFile(FileUtils.toFile(this.getClass().getClassLoader().getResource("tls" + KEYSTORE_FILE)))
            .withKeyStorePassword(PASSWORD)
            .build())
        .withTrustStoreConfig(TrustStoreConfig.builder()
            .withTrustStoreFile(FileUtils.toFile(this.getClass().getClassLoader().getResource("tls" + TRUSTSTORE_FILE)))
            .withTrustStorePassword(PASSWORD)
            .build())
        .build();
  }

  protected ToolingRuntimeClient createToolingRuntimeClient() throws Exception {
    bootstrap = ToolingRuntimeClientBootstrapFactory.newToolingRuntimeClientBootstrap(
                                                                                      ToolingRuntimeClientBootstrapConfiguration
                                                                                          .builder()
                                                                                          .muleVersion(getMuleVersion())
                                                                                          .toolingVersion(getToolingVersion())
                                                                                          .mavenConfiguration(defaultMavenConfiguration)
                                                                                          .log4jConfiguration(getTestLog4JConfigurationFile())
                                                                                          .workingFolder(temporaryFolder
                                                                                              .newFolder())
                                                                                          .build());

    AgentConfiguration tlsAgentConfiguration = builder()
        .withDefaultConnectionTimeout(5000)
        .withDefaultReadTimeout(50000)
        .withToolingApiUrl(new URL(getAgentToolingUrl()))
        .withSslConfiguration(getSslConfiguration())
        .build();

    return bootstrap.getToolingRuntimeClientBuilderFactory().create()
        .withMavenConfiguration(defaultMavenConfiguration)
        .withRemoteAgentConfiguration(tlsAgentConfiguration)
        .build();
  }

  protected String getAgentToolingUrl() {
    return "https://localhost:" + agentPort.getNumber() + "/mule/tooling";
  }

  private void newToolingArtifact(ToolingRuntimeClient toolingRuntimeClient, String appFolder) throws Exception {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File appLocation = new File(targetTestClassesFolder, appFolder);
    toolingArtifact = toolingRuntimeClient.newToolingArtifact(toUrl(appLocation.toURI()), emptyMap());

    assertThat(toolingArtifact, not(nullValue()));
  }

  @After
  public void disposeApplication() throws Exception {
    if (toolingArtifact != null) {
      toolingArtifact.dispose();
    }
    if (bootstrap != null) {
      bootstrap.dispose();
    }
  }

  @Test
  @Description("Checks connection to Rest Agent with TLS configured")
  public void tlsAgentConfigurationTestCase() throws Exception {
    newToolingArtifact(createToolingRuntimeClient(), FILE_APP_LOCATION);

    ConnectivityTestingRequest request = new ConnectivityTestingRequest();
    request.setComponentId("fileConfig");
    ConnectionValidationResult connectionValidationResult =
        toolingArtifact.connectivityTestingService().testConnection(request);

    assertThat(connectionValidationResult.isValid(), is(true));
  }

  @Override
  protected void validateAgentIsUpAndRunning() {
    // Do nothing. This will be validated by the test.
  }
}
