/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.tests.integration.tooling.client;

import org.mule.tooling.client.api.datasense.MetadataCache;
import org.mule.tooling.client.api.metadata.MetadataFailure;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheBuilderSpec;
import com.google.common.cache.CacheStats;

import java.io.Serializable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Callable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ToolingMetadataCacheProvider implements MetadataCache {

  static final transient Logger logger = LoggerFactory.getLogger(ToolingMetadataCacheProvider.class);
  static final String OPERATION_CACHE = "operationMetadataCache";
  static final String SOURCE_CACHE = "sourceMetadataCache";
  static final String METADATA_KEYS_CACHE = "metadataKeysCache";
  private final Cache<Serializable, Serializable> operationMetadataCache;
  private final Cache<Serializable, Serializable> sourceMetadataCache;
  private final Cache<Serializable, Serializable> metadataKeysCache;
  private final Map<String, List<MetadataFailure>> failingComponents = new HashMap<>();

  public ToolingMetadataCacheProvider(CacheBuilderSpec cacheBuilderSpec) {
    operationMetadataCache = CacheBuilder.from(cacheBuilderSpec).recordStats().build();
    sourceMetadataCache = CacheBuilder.from(cacheBuilderSpec).recordStats().build();
    metadataKeysCache = CacheBuilder.from(cacheBuilderSpec).recordStats().build();
  }

  private Optional<String> buildKey(String componentId, Long timestamp) {
    return Optional.ofNullable(!(componentId == null || timestamp == null)
        ? String.valueOf(componentId) + "#" + String.valueOf(timestamp) : null);
  }

  private Serializable getMetadata(String cachedOperationName, Cache<Serializable, Serializable> cache,
                                   MetadataCacheKeyInfo metadataCacheKeyInfo,
                                   Callable<MetadataResult> resolver) {
    try {
      final Optional<String> key = buildKey(metadataCacheKeyInfo.getComponentId(), metadataCacheKeyInfo.getTimestamp());
      if (key.isPresent()) {
        return cache.get(key.get(), (Callable) () -> resolveMetadata(metadataCacheKeyInfo, resolver));
      } else {
        return resolveMetadata(metadataCacheKeyInfo, resolver);
      }
    } catch (Exception e) {
      logger.debug(String.format("Failed resolving %s for location path %s.", cachedOperationName,
                                 metadataCacheKeyInfo.getLocation()),
                   e);
      return null;
    }
  }

  private Serializable resolveMetadata(MetadataCacheKeyInfo metadataCacheKeyInfo,
                                       Callable<MetadataResult> resolver)
      throws Exception {
    final MetadataResult result = resolver.call();
    if (!result.isSuccess()) {
      failingComponents.put(metadataCacheKeyInfo.getLocation(), result.getFailures());
    }
    return result.getResult();
  }

  @Override
  public Serializable getOperationMetadata(MetadataCacheKeyInfo metadataCacheKeyInfo,
                                           Callable<MetadataResult> resolver) {
    return getMetadata("operation metadata", operationMetadataCache, metadataCacheKeyInfo,
                       resolver);
  }


  @Override
  public Serializable getSourceMetadata(MetadataCacheKeyInfo metadataCacheKeyInfo,
                                        Callable<MetadataResult> resolver) {
    return getMetadata("source metadata", sourceMetadataCache, metadataCacheKeyInfo,
                       resolver);
  }

  @Override
  public Serializable getMetadataKeys(MetadataCacheKeyInfo metadataCacheKeyInfo,
                                      Callable<MetadataResult> resolver) {
    return getMetadata("metadata keys", metadataKeysCache, metadataCacheKeyInfo,
                       resolver);
  }

  public Map<String, CacheStats> stats() {
    Map<String, CacheStats> result = new HashMap<>();
    result.put(OPERATION_CACHE, operationMetadataCache.stats());
    result.put(SOURCE_CACHE, sourceMetadataCache.stats());
    result.put(METADATA_KEYS_CACHE, metadataKeysCache.stats());
    return result;
  }

  public Map<String, List<MetadataFailure>> getFailingComponents() {
    return failingComponents;
  }

}
