/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.tests.integration.tooling.client;

import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import static org.mule.maven.client.test.MavenTestHelper.createDefaultEnterpriseMavenConfiguration;
import static org.mule.maven.client.test.MavenTestUtils.getMavenProperty;
import static org.mule.runtime.app.declaration.api.fluent.ElementDeclarer.newArtifact;
import static org.mule.runtime.app.declaration.api.fluent.ElementDeclarer.newListValue;
import static org.mule.runtime.app.declaration.api.fluent.ElementDeclarer.newObjectValue;
import static org.mule.runtime.app.declaration.api.fluent.ElementDeclarer.newParameterGroup;
import static org.mule.runtime.app.declaration.api.fluent.ParameterSimpleValue.cdata;
import static org.mule.tooling.client.api.extension.model.parameter.ParameterGroupModel.CONNECTION;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.POM_FOLDER_FINDER;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getMuleVersion;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getTestLog4JConfigurationFile;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getToolingVersion;
import org.mule.runtime.app.declaration.api.ArtifactDeclaration;
import org.mule.runtime.app.declaration.api.GlobalElementDeclaration;
import org.mule.runtime.app.declaration.api.fluent.ElementDeclarer;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.artifact.declaration.ArtifactSerializationService;
import org.mule.tooling.client.api.artifact.declaration.request.JsonArtifactDeserializationRequest;
import org.mule.tooling.client.api.artifact.declaration.request.JsonArtifactSerializationRequest;
import org.mule.tooling.client.api.artifact.declaration.request.XmlArtifactDeserializationRequest;
import org.mule.tooling.client.api.artifact.declaration.request.XmlArtifactSerializationRequest;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrap;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapConfiguration;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapFactory;
import org.mule.tooling.client.tests.integration.category.DoesNotNeedMuleRuntimeTest;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.apache.commons.io.IOUtils;
import org.custommonkey.xmlunit.DetailedDiff;
import org.custommonkey.xmlunit.Diff;
import org.custommonkey.xmlunit.Difference;
import org.custommonkey.xmlunit.XMLUnit;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.rules.TemporaryFolder;
import org.skyscreamer.jsonassert.JSONAssert;

@Category(DoesNotNeedMuleRuntimeTest.class)
@Feature("XmlGenerationModelService")
@Story("Integration tests for XmlGenerationModelService using ToolingBootstrap and ToolingRuntimeClient")
public class ArtifactSerializationServiceTestCase {

  private static final String DSL_APP_XML = "dsl/app-serialization.xml";
  private static final String DSL_APP_JSON = "dsl/app-artifact.json";
  private ArtifactSerializationService artifactSerializationService;
  private ToolingRuntimeClientBootstrap bootstrap;
  private ToolingRuntimeClient toolingRuntimeClient;
  private ArtifactDeclaration applicationDeclaration;
  private List<ArtifactDescriptor> pluginArtifactDescriptors;
  private String expectedAppXml;
  private String expectedAppJson;
  private static final String MULE_HTTP_CONNECTOR_VERSION = "muleHttpConnectorVersion";
  private static final String MULE_DB_CONNECTOR_VERSION = "muleDbConnectorVersion";
  private static final String MULE_WSC_CONNECTOR_VERSION = "muleWscConnectorVersion";

  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  @Before
  public void setUpToolingRuntimeClient() throws Exception {
    bootstrap = ToolingRuntimeClientBootstrapFactory.newToolingRuntimeClientBootstrap(
                                                                                      ToolingRuntimeClientBootstrapConfiguration
                                                                                          .builder()
                                                                                          .muleVersion(getMuleVersion())
                                                                                          .toolingVersion(getToolingVersion())
                                                                                          .mavenConfiguration(createDefaultEnterpriseMavenConfiguration())
                                                                                          .log4jConfiguration(getTestLog4JConfigurationFile())
                                                                                          .workingFolder(temporaryFolder
                                                                                              .newFolder())
                                                                                          .build());
    toolingRuntimeClient = bootstrap.getToolingRuntimeClientBuilderFactory().create().build();
    artifactSerializationService = toolingRuntimeClient.artifactSerializationService();

    pluginArtifactDescriptors = new ArrayList<>();
    pluginArtifactDescriptors.add(buildPluginArtifact("mule-http-connector", MULE_HTTP_CONNECTOR_VERSION));
    pluginArtifactDescriptors.add(buildPluginArtifact("mule-db-connector", MULE_DB_CONNECTOR_VERSION));
    pluginArtifactDescriptors.add(buildPluginArtifact("mule-wsc-connector", MULE_WSC_CONNECTOR_VERSION));

    applicationDeclaration = createAppDeclaration();
  }

  private ArtifactDescriptor buildPluginArtifact(String artifactId, String pomPropertyWithVersion) {
    return ArtifactDescriptor.newBuilder()
        .withGroupId("org.mule.connectors")
        .withArtifactId(artifactId)
        .withVersion(getMavenProperty(pomPropertyWithVersion, POM_FOLDER_FINDER))
        .withExtension("jar")
        .withClassifier("mule-plugin")
        .build();
  }

  @After
  public void disposeBootstrap() {
    if (bootstrap != null) {
      bootstrap.dispose();
    }
  }

  @Before
  public void loadExpectedResult() throws IOException {
    expectedAppXml = IOUtils.toString(getResource(DSL_APP_XML));
    expectedAppJson = IOUtils.toString(getResource(DSL_APP_JSON));
  }

  @Test
  @Description("Checks the serialization of an ArtifactDeclaration to XML")
  public void serializeArtifactToXml() throws Exception {
    XmlArtifactSerializationRequest request = new XmlArtifactSerializationRequest(applicationDeclaration,
                                                                                  pluginArtifactDescriptors);
    String xml = artifactSerializationService.toXml(request);
    compareXML(expectedAppXml, xml);
  }

  @Test
  @Description("Checks the serialization of an ArtifactDeclaration to JSON")
  public void serializeArtifactToJson() {
    JsonArtifactSerializationRequest request = new JsonArtifactSerializationRequest(applicationDeclaration, true);
    String json = artifactSerializationService.toJson(request);

    JSONAssert.assertEquals(expectedAppJson, json, true);
  }

  @Test
  @Description("Validates that the ArtifactDeclaration can be loaded from an XML serialization")
  public void loadArtifactFromXml() {
    XmlArtifactDeserializationRequest request = new XmlArtifactDeserializationRequest(getResource(DSL_APP_XML),
                                                                                      pluginArtifactDescriptors);
    ArtifactDeclaration app = artifactSerializationService.fromXml(request);

    List<GlobalElementDeclaration> expectedElements = applicationDeclaration.getGlobalElements();
    assertThat(expectedElements.size(), is(app.getGlobalElements().size()));
    expectedElements.forEach(e -> assertTrue(app.getGlobalElements().stream()
        .anyMatch(g -> g.getName().equals(e.getName()) && g.getRefName().equals(e.getRefName()))));

    assertThat(app.getCustomConfigurationParameters().size(), is(8));
  }

  @Test
  @Description("Validates that the ArtifactDeclaration can be loaded from an JSON serialization")
  public void loadArtifactFromJson() {
    JsonArtifactDeserializationRequest request = new JsonArtifactDeserializationRequest(expectedAppJson);
    ArtifactDeclaration app = artifactSerializationService.fromJson(request);
    assertThat(applicationDeclaration, is(app));
  }

  @Test
  @Description("Validates that the ArtifactDeclaration can be loaded and serialized from XML")
  public void loadAndSerializeFromXml() throws Exception {
    XmlArtifactDeserializationRequest request = new XmlArtifactDeserializationRequest(getResource(DSL_APP_XML),
                                                                                      pluginArtifactDescriptors);
    ArtifactDeclaration app = artifactSerializationService.fromXml(request);

    XmlArtifactSerializationRequest serializeRequest = new XmlArtifactSerializationRequest(app, pluginArtifactDescriptors);
    String xml = artifactSerializationService.toXml(serializeRequest);

    compareXML(expectedAppXml, xml);
  }

  @Test
  @Description("Validates that the ArtifactDeclaration can be loaded and serialized from JSON")
  public void loadAndSerializeFromJson() {
    JsonArtifactDeserializationRequest request = new JsonArtifactDeserializationRequest(expectedAppJson);
    ArtifactDeclaration app = artifactSerializationService.fromJson(request);

    JsonArtifactSerializationRequest serializeRequest = new JsonArtifactSerializationRequest(app, true);
    String json = artifactSerializationService.toJson(serializeRequest);

    JSONAssert.assertEquals(json, expectedAppJson, json, true);
  }

  private static ArtifactDeclaration createAppDeclaration() {

    ElementDeclarer db = ElementDeclarer.forExtension("Database");
    ElementDeclarer http = ElementDeclarer.forExtension("HTTP");
    ElementDeclarer sockets = ElementDeclarer.forExtension("Sockets");
    ElementDeclarer core = ElementDeclarer.forExtension("mule");
    ElementDeclarer wsc = ElementDeclarer.forExtension("Web Service Consumer");

    return newArtifact()
        .withGlobalElement(db.newConfiguration("config")
            .withRefName("dbConfig")
            .withConnection(db
                .newConnection("derby")
                .withParameterGroup(newParameterGroup()
                    .withParameter("poolingProfile", newObjectValue()
                        .withParameter("maxPoolSize", "10")
                        .build())
                    .getDeclaration())
                .withParameterGroup(newParameterGroup(CONNECTION)
                    .withParameter("connectionProperties", newObjectValue()
                        .withParameter("first", "propertyOne")
                        .withParameter("second", "propertyTwo")
                        .build())
                    .withParameter("database", "target/muleEmbeddedDB")
                    .withParameter("create", "true")
                    .getDeclaration())
                .getDeclaration())
            .getDeclaration())
        .withGlobalElement(http.newConfiguration("listenerConfig")
            .withRefName("httpListener")
            .withParameterGroup(newParameterGroup()
                .withParameter("basePath", "/")
                .getDeclaration())
            .withConnection(http.newConnection("listener")
                .withParameterGroup(newParameterGroup()
                    .withParameter("tlsContext", newObjectValue()
                        .withParameter("key-store", newObjectValue()
                            .withParameter("path", "ssltest-keystore.jks")
                            .withParameter("password", "changeit")
                            .withParameter("keyPassword", "changeit")
                            .build())
                        .build())
                    .getDeclaration())
                .withParameterGroup(newParameterGroup(CONNECTION)
                    .withParameter("host", "localhost")
                    .withParameter("port", "49019")
                    .withParameter("protocol", "HTTPS")
                    .getDeclaration())
                .getDeclaration())
            .getDeclaration())
        .withGlobalElement(http.newConfiguration("requestConfig")
            .withRefName("httpRequester")
            .withConnection(http.newConnection("request")
                .withParameterGroup(newParameterGroup()
                    .withParameter("authentication",
                                   newObjectValue()
                                       .ofType(
                                               "org.mule.extension.http.api.request.authentication.BasicAuthentication")
                                       .withParameter("username", "user")
                                       .withParameter("password", "pass")
                                       .build())
                    .getDeclaration())
                .withParameterGroup(newParameterGroup(CONNECTION)
                    .withParameter("host", "localhost")
                    .withParameter("port", "49020")
                    .withParameter("clientSocketProperties",
                                   newObjectValue()
                                       .withParameter("connectionTimeout", "1000")
                                       .withParameter("keepAlive", "true")
                                       .withParameter("receiveBufferSize", "1024")
                                       .withParameter("sendBufferSize", "1024")
                                       .withParameter("clientTimeout", "1000")
                                       .withParameter("linger", "1000")
                                       .build())
                    .getDeclaration())
                .getDeclaration())
            .getDeclaration())
        .withGlobalElement(core.newConstruct("flow").withRefName("testFlow")
            .withParameterGroup(newParameterGroup()
                .withParameter("initialState", "stopped")
                .getDeclaration())
            .withComponent(http.newSource("listener")
                .withConfig("httpListener")
                .withParameterGroup(newParameterGroup()
                    .withParameter("path", "testBuilder")
                    .withParameter("redeliveryPolicy",
                                   newObjectValue()
                                       .withParameter("maxRedeliveryCount", "2")
                                       .withParameter("useSecureHash", "true")
                                       .build())
                    .getDeclaration())
                .withParameterGroup(newParameterGroup(CONNECTION)
                    .withParameter("reconnectionStrategy",
                                   newObjectValue()
                                       .ofType("reconnect")
                                       .withParameter("blocking", "true")
                                       .withParameter("count", "1")
                                       .withParameter("frequency", "0")
                                       .build())
                    .getDeclaration())
                .withParameterGroup(newParameterGroup("Response")
                    .withParameter("headers", cdata("#[{{'content-type' : 'text/plain'}}]"))
                    .withParameter("body",
                                   "<![CDATA[#[\n"
                                       + "                    %dw 2.0\n"
                                       + "                    output application/json\n"
                                       + "                    input payload application/xml\n"
                                       + "                    %var baseUrl=\"http://sample.cloudhub.io/api/v1.0/\"\n"
                                       + "                    ---\n"
                                       + "                    using (pageSize = payload.getItemsResponse.PageInfo.pageSize) {\n"
                                       + "                         links: [\n"
                                       + "                            {\n"
                                       + "                                href: fullUrl,\n"
                                       + "                                rel : \"self\"\n"
                                       + "                            }\n"
                                       + "                         ],\n"
                                       + "                         collection: {\n"
                                       + "                            size: pageSize,\n"
                                       + "                            items: payload.getItemsResponse.*Item map {\n"
                                       + "                                id: $.id,\n"
                                       + "                                type: $.type,\n"
                                       + "                                name: $.name\n"
                                       + "                            }\n"
                                       + "                         }\n"
                                       + "                    }\n"
                                       + "                ]]>")
                    .getDeclaration())
                .getDeclaration())
            .withComponent(core.newConstruct("choice")
                .withRoute(core.newRoute("when")
                    .withParameterGroup(newParameterGroup()
                        .withParameter("expression", "#[true]")
                        .getDeclaration())
                    .withComponent(db.newOperation("bulkInsert")
                        .withParameterGroup(newParameterGroup("Query")
                            .withParameter("sql",
                                           "INSERT INTO PLANET(POSITION, NAME) VALUES (:position, :name)")
                            .withParameter("parameterTypes",
                                           newListValue()
                                               .withValue(newObjectValue()
                                                   .withParameter("key", "name")
                                                   .withParameter("type", "VARCHAR")
                                                   .build())
                                               .withValue(newObjectValue()
                                                   .withParameter("key", "position")
                                                   .withParameter("type", "INTEGER")
                                                   .build())
                                               .build())
                            .getDeclaration())
                        .getDeclaration())
                    .getDeclaration())
                .withRoute(core.newRoute("otherwise")
                    .withComponent(core.newConstruct("foreach")
                        .withParameterGroup(newParameterGroup()
                            .withParameter("collection", "#[myCollection]")
                            .getDeclaration())
                        .withComponent(core.newOperation("logger")
                            .withParameterGroup(newParameterGroup()
                                .withParameter("message", "#[payload]")
                                .getDeclaration())
                            .getDeclaration())
                        .getDeclaration())
                    .getDeclaration())
                .getDeclaration())
            .withComponent(db.newOperation("bulkInsert")
                .withParameterGroup(newParameterGroup("Query")
                    .withParameter("sql", "INSERT INTO PLANET(POSITION, NAME) VALUES (:position, :name)")
                    .withParameter("parameterTypes",
                                   newListValue()
                                       .withValue(newObjectValue()
                                           .withParameter("key", "name")
                                           .withParameter("type", "VARCHAR").build())
                                       .withValue(newObjectValue()
                                           .withParameter("key", "position")
                                           .withParameter("type", "INTEGER").build())
                                       .build())
                    .getDeclaration())
                .getDeclaration())
            .withComponent(http.newOperation("request")
                .withConfig("httpRequester")
                .withParameterGroup(newParameterGroup("URI Settings")
                    .withParameter("path", "/nested")
                    .getDeclaration())
                .withParameterGroup(newParameterGroup()
                    .withParameter("method", "POST")
                    .getDeclaration())
                .getDeclaration())
            .withComponent(db.newOperation("insert")
                .withConfig("dbConfig")
                .withParameterGroup(newParameterGroup("Query")
                    .withParameter("sql",
                                   "INSERT INTO PLANET(POSITION, NAME, DESCRIPTION) VALUES (777, 'Pluto', :description)")
                    .withParameter("parameterTypes",
                                   newListValue()
                                       .withValue(newObjectValue()
                                           .withParameter("key", "description")
                                           .withParameter("type", "CLOB").build())
                                       .build())
                    .withParameter("inputParameters", "#[{{'description' : payload}}]")
                    .getDeclaration())
                .getDeclaration())
            .withComponent(
                           sockets.newOperation("sendAndReceive")
                               .withParameterGroup(newParameterGroup()
                                   .withParameter("streamingStrategy",
                                                  newObjectValue()
                                                      .ofType("repeatable-in-memory-stream")
                                                      .withParameter("bufferSizeIncrement", "8")
                                                      .withParameter("bufferUnit", "KB")
                                                      .withParameter("initialBufferSize", "51")
                                                      .withParameter("maxBufferSize", "1000")
                                                      .build())
                                   .getDeclaration())
                               .withParameterGroup(newParameterGroup("Output")
                                   .withParameter("target", "myVar")
                                   .getDeclaration())
                               .getDeclaration())
            .withComponent(
                           wsc.newOperation("consume")
                               .withParameterGroup(newParameterGroup()
                                   .withParameter("operation", "GetCitiesByCountry")
                                   .getDeclaration())
                               .withParameterGroup(newParameterGroup("Message")
                                   .withParameter("attachments", "#[{}]")
                                   .withParameter("headers",
                                                  "#[{\"headers\": {con#headerIn: \"Header In Value\",con#headerInOut: \"Header In Out Value\"}]")
                                   .withParameter("body", "#[payload]")
                                   .getDeclaration())
                               .getDeclaration())
            .getDeclaration())
        .withGlobalElement(core.newConstruct("flow").withRefName("cronSchedulerFlow")
            .withComponent(core.newSource("scheduler")
                .withParameterGroup(newParameterGroup()
                    .withParameter("schedulingStrategy", newObjectValue()
                        .ofType("org.mule.runtime.core.api.source.scheduler.CronScheduler")
                        .withParameter("expression", "0/1 * * * * ?")
                        .build())
                    .getDeclaration())
                .getDeclaration())
            .withComponent(core.newOperation("logger")
                .withParameterGroup(newParameterGroup()
                    .withParameter("message", "#[payload]").getDeclaration())
                .getDeclaration())
            .getDeclaration())
        .getDeclaration();
  }

  private InputStream getResource(String name) {
    InputStream is = Thread.currentThread().getContextClassLoader().getResourceAsStream(name);
    assertTrue("Missing required resource: " + name, is != null);
    return is;
  }

  private static void compareXML(String expected, String actual) throws Exception {
    XMLUnit.setNormalizeWhitespace(true);
    XMLUnit.setIgnoreWhitespace(true);
    XMLUnit.setIgnoreComments(true);
    XMLUnit.setIgnoreAttributeOrder(false);

    Diff diff = XMLUnit.compareXML(expected, actual);
    if (!(diff.similar() && diff.identical())) {
      System.out.println(actual);
      DetailedDiff detDiff = new DetailedDiff(diff);
      @SuppressWarnings("rawtypes")
      List differences = detDiff.getAllDifferences();
      StringBuilder diffLines = new StringBuilder();
      for (Object object : differences) {
        Difference difference = (Difference) object;
        diffLines.append(difference.toString() + '\n');
      }

      throw new IllegalArgumentException("Actual XML differs from expected: \n" + diffLines.toString());
    }
  }

}
