/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.tests.integration.tooling.client;

import static java.util.Collections.emptyMap;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;
import static org.junit.Assert.assertThat;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrap;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapConfiguration;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapFactory;
import org.mule.tooling.client.test.AbstractMuleRuntimeTestCase;
import org.mule.tooling.client.test.RuntimeType;

import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.Map;

import org.junit.After;
import org.junit.Before;

public abstract class BaseToolingClientTestCase extends AbstractMuleRuntimeTestCase {

  private ToolingRuntimeClientBootstrap bootstrap;
  private ToolingArtifact toolingArtifact;

  public BaseToolingClientTestCase(RuntimeType runtimeType) {
    super(runtimeType);
  }

  @Before
  public void setUp() {
    bootstrap = createToolingRuntimeClientBootstrap();
  }

  @After
  public void disposeApplication() throws Exception {
    if (toolingArtifact != null) {
      toolingArtifact.dispose();
    }
  }

  @After
  public void disposeBootstrap() throws Exception {
    if (bootstrap != null) {
      bootstrap.dispose();
      bootstrap = null;
    }
  }

  protected ToolingRuntimeClientBootstrap createToolingRuntimeClientBootstrap() {
    try {
      return ToolingRuntimeClientBootstrapFactory.newToolingRuntimeClientBootstrap(
                                                                                   ToolingRuntimeClientBootstrapConfiguration
                                                                                       .builder()
                                                                                       .muleVersion(getMuleVersion())
                                                                                       .toolingVersion(getToolingVersion())
                                                                                       .mavenConfiguration(defaultMavenConfiguration)
                                                                                       .log4jConfiguration(getTestLog4JConfigurationFile())
                                                                                       .workingFolder(temporaryFolder.newFolder())
                                                                                       .build());
    } catch (IOException e) {
      throw new UncheckedIOException(e);
    }
  }

  protected ToolingRuntimeClient createToolingRuntimeClient() {
    return createToolingRuntimeBuilder().build();
  }

  protected ToolingRuntimeClient.Builder createToolingRuntimeBuilder() {
    return bootstrap.getToolingRuntimeClientBuilderFactory().create()
        .withMavenConfiguration(defaultMavenConfiguration)
        .withRemoteAgentConfiguration(defaultAgentConfiguration);
  }

  public ToolingRuntimeClientBootstrap getBootstrap() {
    return bootstrap;
  }


  protected ToolingArtifact newToolingArtifact(ToolingRuntimeClient toolingRuntimeClient, String app) throws Exception {
    return newToolingArtifact(toolingRuntimeClient, app, emptyMap(), null);
  }

  protected ToolingArtifact newToolingArtifact(ToolingRuntimeClient toolingRuntimeClient, String app, ToolingArtifact parent)
      throws Exception {
    return newToolingArtifact(toolingRuntimeClient, app, emptyMap(), parent.getId());
  }

  protected ToolingArtifact newToolingArtifact(ToolingRuntimeClient toolingRuntimeClient, String app,
                                               Map<String, String> properties)
      throws Exception {
    return newToolingArtifact(toolingRuntimeClient, app, properties, null);
  }

  protected ToolingArtifact newToolingArtifact(ToolingRuntimeClient toolingRuntimeClient, String app,
                                               Map<String, String> properties, String parentId)
      throws Exception {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File appLocation = new File(targetTestClassesFolder, app);
    if (parentId == null) {
      toolingArtifact = newToolingArtifact(toolingRuntimeClient, appLocation, properties);
    } else {
      toolingArtifact = newToolingArtifact(toolingRuntimeClient, appLocation, properties, parentId);
    }

    assertThat(toolingArtifact, not(nullValue()));
    return toolingArtifact;
  }

  protected ToolingArtifact newToolingArtifact(ToolingRuntimeClient toolingRuntimeClient, File appLocation,
                                               Map<String, String> properties, String parentId)
      throws Exception {
    return toolingRuntimeClient.newToolingArtifact(toUrl(appLocation.toURI()), properties, parentId);
  }

  protected ToolingArtifact newToolingArtifact(ToolingRuntimeClient toolingRuntimeClient, File appLocation,
                                               Map<String, String> properties)
      throws Exception {
    return toolingRuntimeClient.newToolingArtifact(toUrl(appLocation.toURI()), properties);
  }

  protected ToolingArtifact newToolingArtifact(ToolingRuntimeClient toolingRuntimeClient, File appLocation) throws Exception {
    return newToolingArtifact(toolingRuntimeClient, appLocation, emptyMap());
  }
}
