/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.tests.integration.tooling.client;

import com.github.tomakehurst.wiremock.junit.WireMockClassRule;
import com.google.common.collect.ImmutableMap;
import com.icegreen.greenmail.util.GreenMail;
import com.icegreen.greenmail.util.ServerSetup;
import org.hamcrest.Matchers;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.connectivity.ConnectivityTestingRequest;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrap;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapConfiguration;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapFactory;
import org.mule.tooling.client.test.AbstractMuleRuntimeTestCase;
import org.mule.tooling.client.test.RuntimeType;

import java.io.File;
import java.net.URI;
import java.net.URL;

import static com.github.tomakehurst.wiremock.client.WireMock.*;
import static com.github.tomakehurst.wiremock.core.WireMockConfiguration.wireMockConfig;
import static org.apache.commons.io.FileUtils.readFileToByteArray;
import static org.apache.commons.io.FileUtils.toFile;
import static org.hamcrest.Matchers.containsString;
import static org.mule.tooling.client.test.utils.ZipUtils.compress;

public class DeploymentPropertiesTestCase extends AbstractMuleRuntimeTestCase {

  private static final String EMAIL_SERVER_PORT = "emailServerPort";
  private static final String APPLICATIONS_LOCATION = "applications/";
  private static final String EMAIL_APP_LOCATION = APPLICATIONS_LOCATION + "email";
  private static final String EMAIL_CONFIG = "emailConfig";
  private static final String REQUEST_PATH = "/app";
  @Rule
  public WireMockClassRule wireMockRule = new WireMockClassRule(wireMockConfig().dynamicPort());
  @Rule
  public DynamicPort emailServerPort = new DynamicPort(EMAIL_SERVER_PORT);
  @Rule
  public ExpectedException expectedException = ExpectedException.none();

  private GreenMail greenMail;
  private ToolingRuntimeClientBootstrap bootstrap;

  public DeploymentPropertiesTestCase(RuntimeType runtimeType) {
    super(runtimeType);
  }

  private void setUpEmailServer() {
    ServerSetup serverSetup = new ServerSetup(emailServerPort.getNumber(), null, "pop3");
    greenMail = new GreenMail(serverSetup);
    greenMail.setUser("foo", "pwd");
    greenMail.start();
  }

  @Before
  public void setUpToolingRuntimeClient() throws Exception {
    bootstrap = ToolingRuntimeClientBootstrapFactory.newToolingRuntimeClientBootstrap(
                                                                                      ToolingRuntimeClientBootstrapConfiguration
                                                                                          .builder()
                                                                                          .muleVersion(getMuleVersion())
                                                                                          .toolingVersion(getToolingVersion())
                                                                                          .mavenConfiguration(defaultMavenConfiguration)
                                                                                          .log4jConfiguration(getTestLog4JConfigurationFile())
                                                                                          .workingFolder(temporaryFolder
                                                                                              .newFolder())
                                                                                          .build());
  }

  @After
  public void stopEmailServerIfStarted() {
    if (greenMail != null) {
      greenMail.stop();
    }
  }

  @After
  public void disposeBootstrap() {
    if (bootstrap != null) {
      bootstrap.dispose();
    }
  }

  @Test
  public void connectionTestingShouldFailWhenPassingDeploymentPropertiesForAgentVersionLessThan415() throws Exception {
    File jarFile = temporaryFolder.newFile();
    compress(jarFile, toFile(this.getClass().getClassLoader().getResource(EMAIL_APP_LOCATION)));

    setUpEmailServer();
    wireMockRule.stubFor(
                         get(urlEqualTo("/app"))
                             .willReturn(aResponse()
                                 .withStatus(200)
                                 .withHeader("Content-Type", "application/octet-stream")
                                 .withBody(readFileToByteArray(jarFile))));
    int port = wireMockRule.port();
    URL applicationContentUrl = new URI("http://localhost:" + port + REQUEST_PATH).toURL();

    ToolingRuntimeClient toolingRuntimeClient = bootstrap.getToolingRuntimeClientBuilderFactory().create()
        .withMavenConfiguration(defaultMavenConfiguration)
        .withRemoteAgentConfiguration(newDefaultAgentConfigurationBuilder().withMuleVersion("4.1.4").build()).build();

    doWithToolingArtifact(toolingRuntimeClient, applicationContentUrl,
                          ImmutableMap.of(EMAIL_SERVER_PORT, emailServerPort.getValue()), toolingArtifact -> {
                            ConnectivityTestingRequest request = new ConnectivityTestingRequest();
                            request.setComponentId(EMAIL_CONFIG);

                            expectedException.expect(ToolingException.class);
                            expectedException.expect(Matchers
                                .hasProperty("rootCauseMessage",
                                             containsString("Couldn't find configuration property value for key ${emailServerPort}")));
                            expectedException.expectMessage(containsString("PropertyNotFoundException"));
                            toolingArtifact.connectivityTestingService().testConnection(request);
                          });
  }


}
