/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.tests.integration.tooling.client;

import static java.lang.System.clearProperty;
import static java.lang.System.setProperty;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.synchronizedList;
import static java.util.stream.IntStream.range;
import static org.apache.commons.io.FileUtils.cleanDirectory;
import static org.apache.commons.io.FileUtils.deleteQuietly;
import static org.apache.commons.lang.exception.ExceptionUtils.getFullStackTrace;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.anyOf;
import static org.hamcrest.Matchers.emptyCollectionOf;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.fail;
import static org.mule.maven.client.test.MavenTestHelper.createDefaultEnterpriseMavenConfiguration;
import static org.mule.maven.client.test.MavenTestHelper.createDefaultEnterpriseMavenConfigurationBuilder;
import static org.mule.maven.client.test.MavenTestUtils.getMavenProperty;
import static org.mule.tooling.client.api.descriptors.ArtifactDescriptor.newBuilder;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.MULE_FILE_CONNECTOR_VERSION;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.MULE_FTP_CONNECTOR_VERSION;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.MULE_HTTP_CONNECTOR_VERSION;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.MULE_SOCKETS_CONNECTOR_VERSION;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.POM_FOLDER_FINDER;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getMuleVersion;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getTestLog4JConfigurationFile;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getToolingVersion;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.component.location.Location;
import org.mule.tooling.client.api.datasense.DataSenseInfo;
import org.mule.tooling.client.api.datasense.DataSenseRequest;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrap;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapConfiguration;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapFactory;
import org.mule.tooling.client.tests.integration.category.DoesNotNeedMuleRuntimeTest;

import com.google.common.io.Files;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CountDownLatch;
import java.util.stream.Collectors;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.apache.commons.io.FileUtils;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Category(DoesNotNeedMuleRuntimeTest.class)
@RunWith(Parameterized.class)
@Feature("Maven")
@Story("Validates how Maven behaves when multiple clients use Tooling and ExtensionModels are loaded in parallel")
public class MavenTestCase {

  private final Logger logger = LoggerFactory.getLogger(this.getClass());
  private static final String APP_LOCATION = "applications/email";

  @Parameterized.Parameter
  public Boolean createBootstrapPerRequest;

  @Parameterized.Parameter(1)
  public Integer numberOfIterations;

  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  @Parameterized.Parameters(name = "{index}: Running {1} tests (create bootstrap per request: {0})")
  public static Collection<Object[]> data() {
    return asList(new Object[][] {
        // create bootstrap per request
        {true, 50},
        // same bootstrap for all request
        {false, 100}
    });
  }

  private static MavenConfiguration toolingClientMavenConfiguration;
  private ToolingRuntimeClientBootstrap singleBootstrap;

  @BeforeClass
  public static void setUpToolingRuntimeClient() throws Exception {
    MavenConfiguration.MavenConfigurationBuilder mavenConfigurationBuilder = createDefaultEnterpriseMavenConfigurationBuilder();
    mavenConfigurationBuilder.forcePolicyUpdateNever(false);
    mavenConfigurationBuilder.localMavenRepositoryLocation(Files.createTempDir());
    toolingClientMavenConfiguration = mavenConfigurationBuilder.build();
  }

  @Before
  public void before() throws IOException {
    if (!createBootstrapPerRequest) {
      singleBootstrap = ToolingRuntimeClientBootstrapFactory.newToolingRuntimeClientBootstrap(
                                                                                              ToolingRuntimeClientBootstrapConfiguration
                                                                                                  .builder()
                                                                                                  .muleVersion(getMuleVersion())
                                                                                                  .toolingVersion(getToolingVersion())
                                                                                                  .mavenConfiguration(createDefaultEnterpriseMavenConfiguration())
                                                                                                  .log4jConfiguration(getTestLog4JConfigurationFile())
                                                                                                  .workingFolder(temporaryFolder
                                                                                                      .newFolder())
                                                                                                  .build());
    }
    cleanDirectory(toolingClientMavenConfiguration.getLocalMavenRepositoryLocation());
  }

  @AfterClass
  public static void deleteTemporaryLocalRepository() {
    deleteQuietly(toolingClientMavenConfiguration.getLocalMavenRepositoryLocation());
  }

  @After
  public void disposeBootstrap() {
    if (singleBootstrap != null) {
      singleBootstrap.dispose();
    }
  }

  @Test
  @Description("Checks resolving datasense in parallel")
  public void resolveStaticDatasenseInParallel() throws Exception {
    final File localMavenRepositoryLocation = toolingClientMavenConfiguration.getLocalMavenRepositoryLocation();

    range(1, this.numberOfIterations).parallel().forEach(iteration -> {
      ToolingRuntimeClientBootstrap bootstrap = null;
      try {
        logger.debug("Iteration ---- Artifacts in LocalRepository: {}, lastModified: {}", listFiles(
                                                                                                    localMavenRepositoryLocation),
                     lookUpJarFile(localMavenRepositoryLocation));

        if (createBootstrapPerRequest) {
          bootstrap = ToolingRuntimeClientBootstrapFactory.newToolingRuntimeClientBootstrap(
                                                                                            ToolingRuntimeClientBootstrapConfiguration
                                                                                                .builder()
                                                                                                .muleVersion(getMuleVersion())
                                                                                                .toolingVersion(getToolingVersion())
                                                                                                .mavenConfiguration(createDefaultEnterpriseMavenConfiguration())
                                                                                                .log4jConfiguration(getTestLog4JConfigurationFile())
                                                                                                .workingFolder(temporaryFolder
                                                                                                    .newFolder())
                                                                                                .build());
        } else {
          bootstrap = this.singleBootstrap;
        }

        ToolingRuntimeClient toolingRuntimeClient =
            bootstrap.getToolingRuntimeClientBuilderFactory().create()
                .withMavenConfiguration(toolingClientMavenConfiguration)
                .build();

        ToolingArtifact toolingArtifact =
            toolingRuntimeClient.newToolingArtifact(this.getClass().getClassLoader().getResource(APP_LOCATION), emptyMap());
        final DataSenseRequest request = new DataSenseRequest();
        request.setLocation(Location.builder().globalName("emailFlow").addProcessorsPart().addIndexPart(0).build());
        Optional<DataSenseInfo> dataSenseInfo = toolingArtifact.dataSenseService().resolveDataSense(request);

        assertThat(dataSenseInfo.isPresent(), is(true));
      } catch (Exception e) {
        logger.error(e.getMessage(), e);
        fail(e.getMessage() + " - " + getFullStackTrace(e));
      } finally {
        if (createBootstrapPerRequest) {
          if (bootstrap != null) {
            bootstrap.dispose();
          }
        }
      }
    });
  }

  @Test
  @Description("Checks loading extension model in parallel for a plugin that depends on another one")
  public void getExtensionModelFromPluginThatDependsOnOtherPlugin() throws Exception {
    final File localMavenRepositoryLocation = toolingClientMavenConfiguration.getLocalMavenRepositoryLocation();
    logger.debug("Using local repository: " + localMavenRepositoryLocation.getAbsolutePath());

    final ArtifactDescriptor httpArtifactDescriptor = newBuilder()
        .withGroupId("org.mule.connectors")
        .withArtifactId("mule-http-connector")
        .withClassifier("mule-plugin")
        .withVersion(getMavenProperty(MULE_HTTP_CONNECTOR_VERSION, POM_FOLDER_FINDER))
        .build();

    final ArtifactDescriptor socketsArtifactDescriptor = newBuilder()
        .withGroupId("org.mule.connectors")
        .withArtifactId("mule-sockets-connector")
        .withClassifier("mule-plugin")
        .withVersion(getMavenProperty(MULE_SOCKETS_CONNECTOR_VERSION, POM_FOLDER_FINDER))
        .build();

    CountDownLatch startLatch = new CountDownLatch(1);
    CountDownLatch endLatch = new CountDownLatch(numberOfIterations);

    List<Exception> errors = synchronizedList(new ArrayList<>());
    runWithoutCacheSize(1, () -> {
      range(1, this.numberOfIterations + 1).parallel().forEach(iteration -> {
        boolean isOdd = iteration % 2 == 0;
        ArtifactDescriptor requestedArtifact = isOdd ? httpArtifactDescriptor : socketsArtifactDescriptor;

        logger.debug("Iteration ---- ({}) Artifacts in LocalRepository: {}, lastModified: {}", requestedArtifact.getArtifactId(),
                     listFiles(
                               localMavenRepositoryLocation),
                     lookUpJarFile(localMavenRepositoryLocation));

        new ExtensionModelLoaderThread(requestedArtifact, isOdd ? "HTTP" : "Sockets", startLatch, endLatch, errors).start();
      });

      startLatch.countDown();
      endLatch.await();

      assertThat(errors, emptyCollectionOf(Exception.class));
    });
  }

  @Test
  @Description("Checks loading extension model at the same for a plugin")
  public void triggerLoadExtensionModelExactlySameMoment() throws Exception {
    final File localMavenRepositoryLocation = toolingClientMavenConfiguration.getLocalMavenRepositoryLocation();
    logger.debug("Using local repository: " + localMavenRepositoryLocation.getAbsolutePath());

    final ArtifactDescriptor fileArtifactDescriptor = newBuilder()
        .withGroupId("org.mule.connectors")
        .withArtifactId("mule-file-connector")
        .withClassifier("mule-plugin")
        .withVersion(getMavenProperty(MULE_FILE_CONNECTOR_VERSION, POM_FOLDER_FINDER))
        .build();

    final ArtifactDescriptor ftpArtifactDescriptor = newBuilder()
        .withGroupId("org.mule.connectors")
        .withArtifactId("mule-ftp-connector")
        .withClassifier("mule-plugin")
        .withVersion(getMavenProperty(MULE_FTP_CONNECTOR_VERSION, POM_FOLDER_FINDER))
        .build();


    List<Exception> errors = synchronizedList(new ArrayList<>());
    runWithoutCacheSize(1, () -> {
      CountDownLatch startLatch = new CountDownLatch(1);
      CountDownLatch endLatch = new CountDownLatch(2);

      new ExtensionModelLoaderThread(fileArtifactDescriptor, "File", startLatch, endLatch, errors).start();
      new ExtensionModelLoaderThread(ftpArtifactDescriptor, "FTP", startLatch, endLatch, errors).start();

      startLatch.countDown();
      endLatch.await();

      assertThat(errors, emptyCollectionOf(Exception.class));
    });
  }

  private void runWithoutCacheSize(int cacheSize, TestCallback callback) throws Exception {
    try {
      setProperty("tooling.client.ExtensionModelServiceCache.cache", "1");
      callback.run();
    } finally {
      clearProperty("tooling.client.ExtensionModelServiceCache.cache");
    }
  }

  private interface TestCallback {

    void run() throws Exception;

  }

  private class ExtensionModelLoaderThread extends Thread {

    private ArtifactDescriptor artifactDescriptor;
    private String expectedExtensionModelName;
    private CountDownLatch startLatch;
    private CountDownLatch endLatch;
    private List<Exception> errors;

    public ExtensionModelLoaderThread(ArtifactDescriptor artifactDescriptor, String expectedExtensionModelName,
                                      CountDownLatch startLatch, CountDownLatch endLatch, List<Exception> errors) {
      this.artifactDescriptor = artifactDescriptor;
      this.expectedExtensionModelName = expectedExtensionModelName;
      this.startLatch = startLatch;
      this.endLatch = endLatch;
      this.errors = errors;
    }

    @Override
    public void run() {
      ToolingRuntimeClientBootstrap bootstrap = null;
      try {
        if (createBootstrapPerRequest) {
          bootstrap = ToolingRuntimeClientBootstrapFactory.newToolingRuntimeClientBootstrap(
                                                                                            ToolingRuntimeClientBootstrapConfiguration
                                                                                                .builder()
                                                                                                .muleVersion(getMuleVersion())
                                                                                                .toolingVersion(getToolingVersion())
                                                                                                .mavenConfiguration(createDefaultEnterpriseMavenConfiguration())
                                                                                                .log4jConfiguration(getTestLog4JConfigurationFile())
                                                                                                .workingFolder(temporaryFolder
                                                                                                    .newFolder())
                                                                                                .build());
        } else {
          bootstrap = singleBootstrap;
        }

        ToolingRuntimeClient toolingRuntimeClient =
            bootstrap.getToolingRuntimeClientBuilderFactory().create()
                .withMavenConfiguration(toolingClientMavenConfiguration)
                .build();

        startLatch.await();
        logger.debug("Resolving ExtensionModel for {}", artifactDescriptor);
        Optional<ExtensionModel> extensionModelOptional =
            toolingRuntimeClient.extensionModelService().loadExtensionModel(artifactDescriptor);
        assertThat(extensionModelOptional.isPresent(), is(true));
        final ExtensionModel extensionModel = extensionModelOptional.get();
        assertThat(extensionModel.getName(), equalTo(expectedExtensionModelName));
      } catch (Exception e) {
        logger.error(e.getMessage(), e);
        errors.add(e);
      } finally {
        endLatch.countDown();
        if (createBootstrapPerRequest) {
          if (bootstrap != null) {
            bootstrap.dispose();
          }
        }
      }
    }
  }

  private long lookUpJarFile(File directory) {
    Collection<File> listFiles = FileUtils.listFiles(directory, new String[] {"jar"}, true);
    listFiles = listFiles.stream().filter(file -> file.getName().endsWith("-mule-plugin")).collect(Collectors.toList());
    assertThat(listFiles, anyOf(hasSize(1), hasSize(0)));
    if (listFiles.isEmpty()) {
      return -1;
    }
    return listFiles.iterator().next().lastModified();
  }

  private List<String> listFiles(File directory) {
    List<String> output = new ArrayList<>();
    File[] listFiles = directory.listFiles();
    for (File file : listFiles) {
      if (file.isFile()) {
        output.add(file.getAbsolutePath());
      } else if (file.isDirectory()) {
        output.addAll(listFiles(file));
      }
    }
    return output;
  }
}
