/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.tests.integration.tooling.client;

import static java.net.InetAddress.getLocalHost;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsNull.notNullValue;
import static org.hamcrest.core.IsNull.nullValue;
import static org.junit.Assert.assertThat;
import static org.junit.rules.ExpectedException.none;
import static org.mule.maven.client.test.MavenTestHelper.createDefaultEnterpriseMavenConfiguration;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getMuleVersion;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getTestLog4JConfigurationFile;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getToolingVersion;
import org.mule.tooling.client.tests.integration.category.DoesNotNeedMuleRuntimeTest;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.message.history.MessageHistory;
import org.mule.tooling.client.api.types.Transaction;
import org.mule.tooling.client.api.types.TransactionStackEntry;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrap;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapConfiguration;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapFactory;
import org.mule.tooling.event.model.ErrorModel;
import org.mule.tooling.event.model.TypedValueModel;

import com.github.kristofa.test.http.MockHttpServer;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.file.Paths;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.apache.commons.io.IOUtils;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;

@Category(DoesNotNeedMuleRuntimeTest.class)
@Feature("MessageHistoryService")
@Story("Integration tests for MessageHistoryService using ToolingBootstrap and ToolingRuntimeClient and mocking Mule Runtime Agent")
public class MessageHistoryOfflineTestCase {

  private static final String EXPECTED_CONTENT = "{ \"a\" : 1, \"b\": \"hola\"}";

  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();
  @Rule
  public ExpectedException expectedException = none();
  @Rule
  public DynamicPort agentPort = new DynamicPort("agentPort");

  private ToolingRuntimeClient toolingRuntimeClient;
  private ToolingRuntimeClientBootstrap bootstrap;
  private MockHttpServer server;
  private TestHttpResponseProvider responseProvider;

  @Before
  public void setUpToolingRuntimeClient() throws Exception {
    bootstrap = ToolingRuntimeClientBootstrapFactory.newToolingRuntimeClientBootstrap(
                                                                                      ToolingRuntimeClientBootstrapConfiguration
                                                                                          .builder()
                                                                                          .muleVersion(getMuleVersion())
                                                                                          .toolingVersion(getToolingVersion())
                                                                                          .mavenConfiguration(createDefaultEnterpriseMavenConfiguration())
                                                                                          .log4jConfiguration(getTestLog4JConfigurationFile())
                                                                                          .workingFolder(temporaryFolder
                                                                                              .newFolder())
                                                                                          .build());
    String localhost = getLocalHost().getHostAddress();
    toolingRuntimeClient =
        bootstrap.getToolingRuntimeClientBuilderFactory().create()
            .withRemoteAgentConfiguration(AgentConfiguration.builder()
                .withDefaultConnectionTimeout(5000)
                .withDefaultReadTimeout(50000)
                .withToolingApiUrl(new URL("http://" + localhost + ":" + agentPort.getNumber() + "/mule/tooling"))
                .build())
            .build();

    responseProvider = new TestHttpResponseProvider();
    server = new MockHttpServer(agentPort.getNumber(), responseProvider);
    server.start();
  }

  @After
  public void disposeBootstrap() throws Exception {
    bootstrap.dispose();
    server.stop();
  }

  @Test
  @Description("Checks aggregating a transaction with a single transaction")
  public void singleTransactionEntry() throws IOException {
    MessageHistory messageHistory = retrieveMessageHistory("message-history-single-transaction.json");

    assertThat(messageHistory.getTransactions(), hasSize(1));
    Transaction transaction = messageHistory.getTransactions().get(0);
    assertThat(transaction.getTransactionStack(), hasSize(1));
    assertThat(transaction.getId(), is("txId"));
    assertThat(new String(transaction.getMessage().getPayload().getContent()), is(EXPECTED_CONTENT));
    TypedValueModel attributes = transaction.getMessage().getAttributes();
    assertThat(attributes.getDataType().getType(), is("com.mule.test.Attributes"));
    assertThat(attributes.getDataType().getMediaType(), is("application/java"));

    TransactionStackEntry transactionStackEntry = transaction.getTransactionStack().get(0);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), is("/sampleFlow/processors/0"));
    //TODO (MULE-13462)
    //assertThat(transactionStackEntry.getBranchId(), is("branchId"));
    //assertThat(transactionStackEntry.getParentBranchId(), is("parentBranchId"));
    assertThat(transactionStackEntry.getTimestamp(), is(1490727222222l));
    assertThat(transactionStackEntry.getData().getMessage().getPayload().getDataType().getMediaType(),
               is("application/json"));
    assertThat(new String(transactionStackEntry.getData().getMessage().getPayload().getContent()), is(EXPECTED_CONTENT));

    assertThat(transactionStackEntry.getData().getVariables().size(), is(2));
    assertVariable(transactionStackEntry.getData().getVariables().get("var1"), "var1");
    assertVariable(transactionStackEntry.getData().getVariables().get("var2"), "var2");

    ErrorModel errorModel = transactionStackEntry.getData().getError();
    assertThat(errorModel, notNullValue());
    assertThat(errorModel.getDescription(), is("description"));
    assertThat(errorModel.getDetailedDescription(), is("detailedDescription"));
    assertThat(errorModel.getType(), is("TRANSFORMATION"));
    assertThat(errorModel.getExceptionType(), is("java.lang.RuntimeException"));
    assertThat(errorModel.getMessage(), notNullValue());
  }

  @Test
  @Description("Checks a message history without transactions")
  public void noTransactions() throws Exception {
    MessageHistory messageHistory = retrieveMessageHistory("message-history-no-transaction.json");
    assertThat(messageHistory.getTransactions(), hasSize(0));
  }

  @Test(expected = ToolingException.class)
  @Description("Checks processing an invalid response")
  public void invalidResponse() throws Exception {
    retrieveMessageHistory("message-history-invalid-content");
  }

  @Test
  @Description("Checks processing the before event to generate the source message only")
  public void onlyPreInvokeNotification() throws Exception {
    MessageHistory messageHistory = retrieveMessageHistory("message-history-incomplete-transaction.json");
    assertThat(messageHistory.getTransactions(), hasSize(1));
    assertThat(messageHistory.getTransactions().get(0).getTransactionStack(), hasSize(0));
  }

  @Test
  @Description("Checks processing a pre/post notification of a partial consume therefore they are related to other message processors rather to the first one")
  public void noNotificationsForFirstMessageProcessor() throws Exception {
    MessageHistory messageHistory = retrieveMessageHistory("message-history-no-notifications-first-processor-transaction.json");
    assertThat(messageHistory.getTransactions(), hasSize(1));
    Transaction transaction = messageHistory.getTransactions().get(0);
    assertThat(transaction.getTransactionStack(), hasSize(1));
    assertThat(transaction.getMessage(), is(nullValue()));

    TransactionStackEntry transactionStackEntry = transaction.getTransactionStack().get(0);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), is("/sampleFlow/processors/1"));
  }

  @Test
  @Description("Checks processing multiple transactions")
  public void multipleTransactions() throws Exception {
    MessageHistory messageHistory = retrieveMessageHistory("message-history-multiple-transactions.json");

    assertThat(messageHistory.getTransactions(), hasSize(2));

    Transaction firstTransaction =
        messageHistory.getTransactions().stream().filter(transaction -> transaction.getId().equals("txId")).findAny().get();
    assertThat(firstTransaction.getTransactionStack(), hasSize(3));
    TransactionStackEntry transactionStackEntry = firstTransaction.getTransactionStack().get(2);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), is("/sampleFlow/processors/2"));
    //TODO (MULE-13462)
    //assertThat(transactionStackEntry.getBranchId(), is("branchId3"));
    //assertThat(transactionStackEntry.getParentBranchId(), is("branchId2"));

    Transaction secondTransaction = messageHistory.getTransactions().stream()
        .filter(transaction -> transaction.getId().equals("anotherTxId")).findAny().get();
    assertThat(secondTransaction.getTransactionStack(), hasSize(1));
    transactionStackEntry = secondTransaction.getTransactionStack().get(0);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), is("/anotherFlow/processors/3"));
    //TODO (MULE-13462)
    //assertThat(transactionStackEntry.getBranchId(), is("anotherBranchId"));
    //assertThat(transactionStackEntry.getParentBranchId(), nullValue());
  }

  private void assertVariable(TypedValueModel var, String variableName) {
    assertThat(new String(var.getContent()), is(variableName + "Value"));
    assertThat(var.getDataType().getMediaType(), is("application/" + variableName));
    assertThat(var.getDataType().getType(), is("org.mule.test." + variableName));
  }

  private MessageHistory retrieveMessageHistory(String messageHistoryFile) throws IOException {
    responseProvider.nextRequestRespondOk();
    toolingRuntimeClient.messageHistoryService().enable("fakeApp");

    InputStream fakeResponse = this.getClass().getClassLoader()
        .getResourceAsStream(Paths.get("fake-agent-responses", messageHistoryFile).toString());
    String agentTrackingNotificationsResponse = IOUtils.toString(fakeResponse);
    responseProvider.nextRequestRespondWithJson(agentTrackingNotificationsResponse);
    return toolingRuntimeClient.messageHistoryService().consume("fakeApp", 100);
  }

}
