/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.tests.integration.tooling.client;

import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.internal.utils.MetadataTypeWriter;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.component.location.Location;
import org.mule.tooling.client.api.datasense.DataSenseComponentInfo;
import org.mule.tooling.client.api.datasense.DataSenseRequest;
import org.mule.tooling.client.test.RuntimeType;

import com.google.common.cache.CacheBuilderSpec;
import com.google.common.cache.CacheStats;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Before;
import org.junit.Test;

@Feature("MetadataCache for DataSenseService ")
@Story("Integration tests for DataSense resolution using cache on domain artifacts")
public class MetadataCacheDataSenseTestCase extends BaseDynamicDataSenseTestCase {

  private static final String APP_LOCATION = "applications/datasense-cached-wsc-app";
  private static final String DOMAIN_LOCATION = "domains/datasense-cached-wsc-domain";
  private static final String PROJECT_APP_ID = "projectAppId";
  private static final String PROJECT_APP_ID_VALUE = "1";
  private static final String PROJECT_DOMAIN_ID = "projectDomainId";
  private static final String PROJECT_PROJECT_DOMAIN_ID_VALUE = "1";

  private ToolingArtifact application;
  private Map<String, String> toolingApplicationProperties;
  private ToolingArtifact domain;
  private Map<String, String> toolingDomainProperties;

  private ToolingMetadataCacheProvider applicationDataSenseMetadataCacheProvider;
  private ToolingMetadataCacheProvider domainDataSenseMetadataCacheProvider;

  public MetadataCacheDataSenseTestCase(RuntimeType runtimeType) {
    super(runtimeType);
  }

  @Before
  public void init() {
    applicationDataSenseMetadataCacheProvider = new ToolingMetadataCacheProvider(
                                                                                 CacheBuilderSpec
                                                                                     .parse("expireAfterWrite=50s, maximumSize=100"));
    domainDataSenseMetadataCacheProvider = new ToolingMetadataCacheProvider(
                                                                            CacheBuilderSpec
                                                                                .parse("expireAfterWrite=50s, maximumSize=100"));

    toolingApplicationProperties = new HashMap<>();
    toolingApplicationProperties.put(PROJECT_APP_ID, PROJECT_APP_ID_VALUE);

    toolingDomainProperties = new HashMap<>();
    toolingDomainProperties.put(PROJECT_DOMAIN_ID, PROJECT_PROJECT_DOMAIN_ID_VALUE);
  }

  @Test
  @Description("Checks that resolving dynamic DataSense using cache will generate miss, load and hit")
  public void resolveDataSense() throws Exception {
    ToolingRuntimeClient toolingRuntimeClient = createToolingRuntimeBuilder().withMetadataCacheFactory(
                                                                                                       (toolingArtifactId,
                                                                                                        artifactUrlContent,
                                                                                                        toolingArtifactProperties) -> {
                                                                                                         if (toolingArtifactId
                                                                                                             .equals(domain
                                                                                                                 .getId())) {
                                                                                                           assertThat(toolingArtifactProperties
                                                                                                               .get(PROJECT_DOMAIN_ID),
                                                                                                                      is(PROJECT_PROJECT_DOMAIN_ID_VALUE));
                                                                                                           return domainDataSenseMetadataCacheProvider;

                                                                                                         } else if (toolingArtifactId
                                                                                                             .equals(application
                                                                                                                 .getId())) {
                                                                                                           assertThat(toolingArtifactProperties
                                                                                                               .get(PROJECT_APP_ID),
                                                                                                                      is(PROJECT_APP_ID_VALUE));
                                                                                                           return applicationDataSenseMetadataCacheProvider;
                                                                                                         }
                                                                                                         throw new IllegalStateException("Couldn't create cache");
                                                                                                       })
        .build();

    domain = newToolingArtifact(toolingRuntimeClient, DOMAIN_LOCATION, toolingDomainProperties);
    application = newToolingArtifact(toolingRuntimeClient, APP_LOCATION, toolingApplicationProperties, domain.getId());

    DataSenseRequest dataSenseRequest =
        createDataSenseRequest(Location.builder().globalName("ws_consumer").addProcessorsPart()
            .addIndexPart(1).build());

    Optional<DataSenseComponentInfo> result;

    result = application.dataSenseService().resolveComponentDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));

    result = application.dataSenseService().resolveComponentDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));

    result = application.dataSenseService().resolveComponentDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));

    DataSenseComponentInfo componentInfo = result.get();

    assertThat(componentInfo.getOperationModel().isPresent(), is(true));

    MetadataType output = componentInfo.getOperationModel().get().getOutput().getType();

    MetadataTypeWriter metadataTypeWriter = new MetadataTypeWriter();
    assertEquals("%type _:Java = {\n"
        + "  \"body\"? : @typeId(\"value\" : \"{http://www.webserviceX.NET}GetWeatherResponse\") {\n"
        + "    \"{http://www.webserviceX.NET}GetWeatherResponse\" : @typeId(\"value\" : \"{http://www.webserviceX.NET}#AnonType_GetWeatherResponse\") {{\n"
        + "      \"{http://www.webserviceX.NET}GetWeatherResult\"? : String\n"
        + "    }}\n"
        + "  }\n"
        + "}",
                 metadataTypeWriter.toString(output));

    final Map<String, CacheStats> appStats = applicationDataSenseMetadataCacheProvider.stats();
    final CacheStats appCacheStats = appStats.get(ToolingMetadataCacheProvider.OPERATION_CACHE);
    assertEquals(1, appCacheStats.missCount());
    assertEquals(1, appCacheStats.loadCount());
    assertEquals(2, appCacheStats.hitCount());

    final Map<String, CacheStats> domainStats = domainDataSenseMetadataCacheProvider.stats();
    final CacheStats domainCacheStats = domainStats.get(ToolingMetadataCacheProvider.OPERATION_CACHE);
    assertEquals(0, domainCacheStats.missCount());
    assertEquals(0, domainCacheStats.loadCount());
    assertEquals(0, domainCacheStats.hitCount());
  }
}
