/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.tests.integration.tooling.client;

import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static org.apache.commons.lang3.tuple.ImmutablePair.of;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;
import static org.junit.Assert.assertThat;
import static org.junit.rules.ExpectedException.none;
import static org.mule.tooling.client.api.metadata.FailureCode.componentNotFoundFailureCode;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.component.location.Location;
import org.mule.tooling.client.api.metadata.MetadataKeysContainer;
import org.mule.tooling.client.api.metadata.MetadataKeysRequest;
import org.mule.tooling.client.api.metadata.MetadataResult;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrap;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapConfiguration;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapFactory;
import org.mule.tooling.client.test.AbstractMuleRuntimeTestCase;
import org.mule.tooling.client.test.RuntimeType;
import org.mule.tooling.client.tests.integration.category.NeedMuleRuntimeTest;

import com.icegreen.greenmail.util.GreenMail;
import com.icegreen.greenmail.util.ServerSetup;

import java.io.File;
import java.util.List;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.rules.ExpectedException;

@Category(NeedMuleRuntimeTest.class)
@Feature("MetadataService")
@Story("Integration tests for MetadataService using ToolingBootstrap and ToolingRuntimeClient")
public class MetadataTestCase extends AbstractMuleRuntimeTestCase {

  private static final String EMAIL_CONFIG = "emailConfig";
  private static final String EMAIL_FLOW = "emailFlow";
  @Rule
  public ExpectedException expectedException = none();
  @Rule
  public DynamicPort emailServerPort = new DynamicPort("emailServerPort");
  private ToolingRuntimeClientBootstrap bootstrap;
  private ToolingRuntimeClient toolingRuntimeClient;

  private ToolingArtifact artifact;
  private GreenMail greenMail;
  private MetadataKeysRequest request;

  public MetadataTestCase(RuntimeType runtimeType) {
    super(runtimeType);
  }

  @Override
  protected List<ImmutablePair<String, String>> getStartupSystemProperties() {
    return singletonList(of("emailServerPort", String.valueOf(emailServerPort.getNumber())));
  }

  @Before
  public void setUpToolingRuntimeClient() throws Exception {
    bootstrap = ToolingRuntimeClientBootstrapFactory.newToolingRuntimeClientBootstrap(
                                                                                      ToolingRuntimeClientBootstrapConfiguration
                                                                                          .builder()
                                                                                          .muleVersion(getMuleVersion())
                                                                                          .toolingVersion(getToolingVersion())
                                                                                          .mavenConfiguration(defaultMavenConfiguration)
                                                                                          .log4jConfiguration(getTestLog4JConfigurationFile())
                                                                                          .workingFolder(temporaryFolder
                                                                                              .newFolder())
                                                                                          .build());
    toolingRuntimeClient = bootstrap.getToolingRuntimeClientBuilderFactory().create()
        .withMavenConfiguration(defaultMavenConfiguration)
        .withRemoteAgentConfiguration(defaultAgentConfiguration)
        .build();
  }

  @Before
  public void setUpRequest() {
    request = new MetadataKeysRequest();
  }

  @After
  public void disposeApplication() throws Exception {
    if (artifact != null) {
      artifact.dispose();
    }
    stopEmailServer();
  }

  @After
  public void disposeBootstrap() throws Exception {
    if (bootstrap != null) {
      bootstrap.dispose();
    }
  }

  @Test
  @Description("Checks resolution of metadata keys for global elements")
  public void getMetadataKeyGlobalComponent() throws Exception {
    setUpEmailServer();
    newToolingArtifact("email");

    request.setLocation(Location.builderFromStringRepresentation(EMAIL_CONFIG).build());
    final MetadataResult<MetadataKeysContainer> result = artifact.metadataService().getMetadataKeys(request);

    assertThat(result.isSuccess(), is(true));
  }

  @Test
  @Description("Checks resolution of metadata keys for global element defined in a domain artifact")
  public void getMetadataKeyGlobalComponentOnDomain() throws Exception {
    setUpEmailServer();
    newToolingArtifactForDomain("email-domain");

    request.setLocation(Location.builderFromStringRepresentation(EMAIL_CONFIG).build());
    final MetadataResult<MetadataKeysContainer> result = artifact.metadataService().getMetadataKeys(request);

    assertThat(result.isSuccess(), is(true));
  }

  @Test
  @Description("Checks resolution of metadata keys for a component")
  public void getMetadataKeyByFlowAndComponent() throws Exception {
    setUpEmailServer();
    newToolingArtifact("email");

    request.setLocation(Location.builder().globalName(EMAIL_FLOW).addProcessorsPart().addIndexPart(0).build());
    final MetadataResult<MetadataKeysContainer> result = artifact.metadataService().getMetadataKeys(request);

    assertThat(result.isSuccess(), is(false));
  }

  @Test
  @Description("Checks resolution of metadata keys for a non existing component")
  public void getMetadataKeyDynamicComponentIdNotFound() throws Exception {
    setUpEmailServer();
    newToolingArtifact("email");

    request.setLocation(Location.builder().globalName(EMAIL_FLOW).addProcessorsPart().addIndexPart(1).build());
    final MetadataResult<MetadataKeysContainer> result = artifact.metadataService().getMetadataKeys(request);

    assertThat(result.isSuccess(), is(false));
    assertThat(result.getFailures().get(0).getFailureCode(), equalTo(componentNotFoundFailureCode("COMPONENT_NOT_FOUND")));
    assertThat(result.getFailures().get(0).getMessage(), is("No object found at location emailFlow/processors/1"));
  }

  @Test
  public void getMetadataKeysWithBadConfigRefAttribute() throws Exception {
    newToolingArtifact("bad-config-ref");
    request.setLocation(Location.builder().globalName("HTTP_Listener_config").build());
    final MetadataResult<MetadataKeysContainer> result = artifact.metadataService().getMetadataKeys(request);
    assertThat(result.isSuccess(), is(true));
  }

  @Test
  public void getMetadataKeysWithObjectSource() throws Exception {
    newToolingArtifact("mysql");

    request.setLocation(Location.builderFromStringRepresentation("mysqlFlow/processors/0").build());
    final MetadataResult<MetadataKeysContainer> result = artifact.metadataService().getMetadataKeys(request);

    assertThat(result.isSuccess(), is(true));
  }

  private void newToolingArtifact(String applicationName) throws Exception {
    newToolingArtifact("applications", applicationName);
  }

  private void newToolingArtifactForDomain(String domainName) throws Exception {
    newToolingArtifact("domains", domainName);
  }

  private void newToolingArtifact(String type, String domainName) throws Exception {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File applicationLocation = new File(targetTestClassesFolder, type + "/" + domainName);
    artifact = newToolingArtifact(applicationLocation);

    assertThat(artifact, not(nullValue()));
  }

  protected ToolingArtifact newToolingArtifact(File appLocation) throws Exception {
    return toolingRuntimeClient.newToolingArtifact(toUrl(appLocation.toURI()), emptyMap());
  }

  private void setUpEmailServer() {
    ServerSetup serverSetup = new ServerSetup(emailServerPort.getNumber(), null, "pop3");
    greenMail = new GreenMail(serverSetup);
    greenMail.setUser("foo", "pwd");
    greenMail.start();
  }

  private void stopEmailServer() {
    if (greenMail != null) {
      greenMail.stop();
    }
  }

}
