/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.tests.integration.tooling.client;

import static org.apache.commons.io.FileUtils.readFileToString;
import static org.apache.commons.io.FileUtils.toFile;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.junit.Assert.assertThat;
import static org.junit.Assume.assumeThat;
import static org.junit.rules.ExpectedException.none;
import static org.mule.maven.client.test.MavenTestHelper.createDefaultEnterpriseMavenConfiguration;
import static org.mule.tooling.client.api.descriptors.ArtifactDescriptor.newBuilder;
import static org.mule.tooling.client.api.metadata.type.MetadataTypeHelper.getObjectTypeHierarchies;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getMuleVersion;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getTestLog4JConfigurationFile;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getToolingVersion;
import static org.skyscreamer.jsonassert.JSONAssert.assertEquals;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import org.mule.tooling.client.api.metadata.type.ObjectTypeHierarchy;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrap;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapConfiguration;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapFactory;
import org.mule.tooling.client.internal.persistence.MetadataTypeAdapterFactory;
import org.mule.tooling.client.tests.integration.category.DoesNotNeedMuleRuntimeTest;

import com.google.common.collect.ImmutableSet;
import com.google.gson.GsonBuilder;

import java.nio.file.Paths;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.json.JSONArray;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Category(DoesNotNeedMuleRuntimeTest.class)
@Feature("MetadataTypeHelper")
@Story("Integration tests for MetadataTypeHelper")
public class MetadataTypeHelperTestCase {

  private final Logger logger = LoggerFactory.getLogger(this.getClass());

  private ToolingRuntimeClientBootstrap bootstrap;
  private ToolingRuntimeClient toolingRuntimeClient;

  @Rule
  public ExpectedException expectedException = none();
  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  @Before
  public void setUpToolingRuntimeClient() throws Exception {
    bootstrap = ToolingRuntimeClientBootstrapFactory.newToolingRuntimeClientBootstrap(
                                                                                      ToolingRuntimeClientBootstrapConfiguration
                                                                                          .builder()
                                                                                          .muleVersion(getMuleVersion())
                                                                                          .toolingVersion(getToolingVersion())
                                                                                          .mavenConfiguration(createDefaultEnterpriseMavenConfiguration())
                                                                                          .log4jConfiguration(getTestLog4JConfigurationFile())
                                                                                          .workingFolder(temporaryFolder
                                                                                              .newFolder())
                                                                                          .build());
    toolingRuntimeClient = bootstrap.getToolingRuntimeClientBuilderFactory().create().build();
  }

  @After
  public void disposeBootstrap() throws Exception {
    if (bootstrap != null) {
      bootstrap.dispose();
    }
  }

  @Test
  @Description("Resolve subTypes for HTTP and OAuth context")
  public void resolveSubTypesForHttpAndOauth() throws Exception {
    Set<ExtensionModel> extensionModels = ImmutableSet.<ExtensionModel>builder()
        .addAll(toolingRuntimeClient.extensionModelService().loadMuleExtensionModels())
        .add(getExtensionModel(newBuilder()
            .withGroupId("org.mule.connectors")
            .withArtifactId("mule-http-connector")
            .withClassifier("mule-plugin")
            .withVersion("1.1.0")
            .build()))
        .add(getExtensionModel(newBuilder()
            .withGroupId("org.mule.modules")
            .withArtifactId("mule-oauth-module")
            .withClassifier("mule-plugin")
            .withVersion("1.1.0")
            .build()))
        .build();

    List<ObjectTypeHierarchy> objectTypeHierarchies = getObjectTypeHierarchies(extensionModels);
    assertThat(objectTypeHierarchies, hasSize(5));

    JSONArray responseAsJson = new JSONArray(new GsonBuilder()
        .setPrettyPrinting()
        .disableHtmlEscaping()
        .registerTypeAdapterFactory(new MetadataTypeAdapterFactory())
        .create().toJson(objectTypeHierarchies));
    String expectedJsonResponse = readFileToString(toFile(getClass().getClassLoader().getResource(
                                                                                                  Paths
                                                                                                      .get("metadatatypes",
                                                                                                           "http-oauth-mule-muleee-type-hierarchies.json")
                                                                                                      .toString())));
    assertEquals(expectedJsonResponse, responseAsJson, false);
  }

  public ExtensionModel getExtensionModel(ArtifactDescriptor build) {
    final Optional<ExtensionModel> extensionModel = toolingRuntimeClient.extensionModelService().loadExtensionModel(build);
    assumeThat(extensionModel.isPresent(), is(true));
    return extensionModel.get();
  }

}
