/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.tests.integration.tooling.client;

import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.collection.IsArrayWithSize.arrayWithSize;
import static org.hamcrest.core.Is.is;
import static org.junit.rules.ExpectedException.none;
import static org.mule.maven.client.test.MavenTestHelper.createDefaultEnterpriseMavenConfiguration;
import static org.mule.maven.client.test.MavenTestUtils.getMavenProperty;
import static org.mule.tooling.client.api.descriptors.ArtifactDescriptor.newBuilder;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.MULE_WSC_CONNECTOR_VERSION;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.POM_FOLDER_FINDER;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getMuleVersion;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getTestLog4JConfigurationFile;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getToolingVersion;
import org.mule.tooling.client.tests.integration.category.DoesNotNeedMuleRuntimeTest;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrap;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapConfiguration;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapFactory;

import java.io.File;
import java.io.FileFilter;
import java.net.URISyntaxException;
import java.util.Optional;

import io.qameta.allure.Feature;
import org.apache.commons.io.filefilter.AndFileFilter;
import org.apache.commons.io.filefilter.PrefixFileFilter;
import org.apache.commons.io.filefilter.SuffixFileFilter;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;

@Category(DoesNotNeedMuleRuntimeTest.class)
@Feature("SchemaLoaderTestCase")
public class SchemaLoaderTestCase {

  private static final String EXTENSIONS_FOLDER = "extensions";
  private static final String JAR = ".jar";
  private ToolingRuntimeClientBootstrap bootstrap;
  private ToolingRuntimeClient toolingRuntimeClient;
  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();
  @Rule
  public ExpectedException expectedException = none();

  @Before
  public void setUpToolingRuntimeClient() throws Exception {
    bootstrap = ToolingRuntimeClientBootstrapFactory.newToolingRuntimeClientBootstrap(
                                                                                      ToolingRuntimeClientBootstrapConfiguration
                                                                                          .builder()
                                                                                          .muleVersion(getMuleVersion())
                                                                                          .toolingVersion(getToolingVersion())
                                                                                          .mavenConfiguration(createDefaultEnterpriseMavenConfiguration())
                                                                                          .log4jConfiguration(getTestLog4JConfigurationFile())
                                                                                          .workingFolder(temporaryFolder
                                                                                              .newFolder())
                                                                                          .build());
    toolingRuntimeClient = bootstrap.getToolingRuntimeClientBuilderFactory().create().build();
  }

  @After
  public void disposeBootstrap() throws Exception {
    if (bootstrap != null) {
      bootstrap.dispose();
    }
  }

  @Test
  public void getSchemaFromPluginJarFile() throws Exception {
    File pluginJarFile = findExtensionModelJarFile(this.getClass(), "mule-email-connector");
    Optional<String> schema = toolingRuntimeClient.extensionModelService().loadExtensionSchema(pluginJarFile);
    assertThat(schema.isPresent(), is(true));
    assertThat(isBlank(schema.get()), is(false));
  }

  @Test
  public void getSchemaForExtensionWithDependencies() throws Exception {
    File pluginJarFile = findExtensionModelJarFile(this.getClass(), "mule-http-connector");
    Optional<String> schema = toolingRuntimeClient.extensionModelService().loadExtensionSchema(pluginJarFile);
    assertThat(schema.isPresent(), is(true));
    assertThat(isBlank(schema.get()), is(false));
  }

  @Test
  public void getSchemaFromArtifactDescriptor() throws Exception {
    Optional<String> schema = toolingRuntimeClient.extensionModelService()
        .loadExtensionSchema(newBuilder()
            .withGroupId("org.mule.connectors")
            .withArtifactId("mule-wsc-connector")
            .withClassifier("mule-plugin")
            .withVersion(getMavenProperty(MULE_WSC_CONNECTOR_VERSION, POM_FOLDER_FINDER))
            .build());
    assertThat(schema.isPresent(), is(true));
    assertThat(isBlank(schema.get()), is(false));
  }

  private static File findExtensionModelJarFile(Class<?> testClass, String prefixFileFilter) {
    final AndFileFilter filter = new AndFileFilter(new PrefixFileFilter(prefixFileFilter), new SuffixFileFilter(JAR));
    return findExtensionFile(testClass, filter);
  }

  private static File findExtensionFile(Class<?> testClass, FileFilter filter) {
    File[] extensionsFound = findExtensionFiles(testClass, filter);
    assertThat(extensionsFound, arrayWithSize(1));
    return extensionsFound[0];

  }

  private static File[] findExtensionFiles(Class<?> testClass, FileFilter filter) {
    File extensionsFolder = getExtensionsFolder(testClass);

    return extensionsFolder.listFiles(filter);
  }

  private static File getExtensionsFolder(Class<?> testClass) {
    try {
      return new File(new File(testClass.getProtectionDomain().getCodeSource().getLocation().toURI()).getParentFile(),
                      EXTENSIONS_FOLDER);
    } catch (URISyntaxException e) {
      throw new ToolingException("Error while getting testClass location", e);
    }
  }

}
