/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.tests.integration.tooling.client;

import static org.hamcrest.CoreMatchers.allOf;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.core.IsNot.not;
import static org.junit.Assert.assertThat;
import static org.junit.rules.ExpectedException.none;
import static org.mule.tooling.client.api.datasense.DataSenseNotificationType.errorDataSenseNotificationType;
import static org.mule.tooling.client.api.datasense.DataSenseNotificationType.infoDataSenseNotificationType;
import static org.mule.tooling.client.tests.integration.tooling.client.DataSenseNotificationMatcher.notificationMessageContains;
import static org.mule.tooling.client.tests.integration.tooling.client.DataSenseNotificationMatcher.notificationTypeIs;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.internal.utils.MetadataTypeWriter;
import org.mule.metadata.message.api.MuleEventMetadataType;
import org.mule.tooling.client.api.component.location.Location;
import org.mule.tooling.client.api.datasense.DataSenseElementInfo;
import org.mule.tooling.client.api.datasense.DataSenseInfo;
import org.mule.tooling.client.api.datasense.DataSenseRequest;
import org.mule.tooling.client.api.exception.ToolingException;

import com.google.common.collect.ImmutableMap;

import java.util.Map;
import java.util.Optional;

import io.qameta.allure.Description;
import io.qameta.allure.Story;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

@Story("Integration tests for static DataSense resolution using ToolingBootstrap and ToolingRuntimeClient")
public class StaticDataSenseTestCase extends BaseStaticDataSenseTestCase {

  private static final String APP_LOCATION = "applications/datasense-static";

  @Rule
  public ExpectedException expectedException = none();

  @Override
  protected String getAppLocation() {
    return APP_LOCATION;
  }

  @Test
  @Description("Checks resolution of DataSense on Domains is not supported")
  public void resolveDataSenseOnDomainsNotSupported() throws Exception {
    newToolingArtifact("domains/email-domain");

    final DataSenseRequest request = getRequest();
    request.setLocation(Location.builder().globalName("httpListenerConfig").build());
    expectedException.expect(IllegalStateException.class);
    expectedException.expectMessage(containsString("DataSense cannot be calculated"));
    getToolingArtifact().dataSenseService().resolveDataSense(request);
  }

  @Test
  @Description("Checks resolution of DataSense when application has min Mule version not satisfied should fail")
  public void resolveDataSenseApplicationMinMuleVersionNotSatisfied() throws Exception {
    expectedException.expect(ToolingException.class);
    expectedException.expectMessage(containsString("requires a newest runtime version"));
    newToolingArtifact("applications/appMinMuleVersionTo4200");
  }

  @Test
  @Description("Checks resolution of DataSense static")
  public void resolveDataSense() throws Exception {
    newToolingArtifact();

    final DataSenseRequest request = getRequest();
    request.setLocation(Location.builder().globalName("staticDataSenseFlow").addProcessorsPart().addIndexPart(0).build());
    Optional<DataSenseInfo> result = getToolingArtifact().dataSenseService().resolveDataSense(request);

    assertThat(result.isPresent(), is(true));
  }

  @Test
  @Description("Checks resolution of DataSense static for an application that holds large schemas and metadata types")
  public void resolveDataSenseBigSchemas() throws Exception {
    newToolingArtifact("applications/dw-performance");

    final DataSenseRequest request = getRequest();
    request.setLocation(Location.builder().globalName("testFlow").addProcessorsPart().addIndexPart(0).build());
    Optional<DataSenseInfo> result = getToolingArtifact().dataSenseService().resolveDataSense(request);

    // It is so big the metadata type that I write to json to do assertions, the purpose of this test is to check that serialization works
    assertThat(result.isPresent(), is(true));
  }

  @Test
  @Description("Checks resolution of metadata keys for a non existing component")
  public void resolveDataSenseNonExistingComponent() throws Exception {
    newToolingArtifact();

    final DataSenseRequest request = getRequest();
    request.setLocation(Location.builder().globalName("staticDataSenseFlow").addProcessorsPart().addIndexPart(5).build());
    Optional<DataSenseInfo> result = getToolingArtifact().dataSenseService().resolveDataSense(request);

    assertThat(result.isPresent(), is(false));
  }

  @Test
  @Description("Checks resolution of DataSense static using an Smart Connector")
  public void resolveDataSenseSmartConnector() throws Exception {
    newToolingArtifact("applications/datasense-smart-connector");

    final DataSenseRequest request = getRequest();
    request.setLocation(Location.builder().globalName("flow").addProcessorsPart().addIndexPart(0).build());
    Optional<DataSenseInfo> result = getToolingArtifact().dataSenseService().resolveDataSense(request);

    assertThat(result.isPresent(), is(true));
    result.ifPresent(dataSenseInfo -> {
      final MetadataTypeWriter metadataTypeWriter = new MetadataTypeWriter();
      assertThat(dataSenseInfo.getOutput().isPresent(), is(true));
      final MetadataType outputType = dataSenseInfo.getOutput()
          .orElseThrow(() -> new AssertionError("Expected payload output type not present"));
      assertThat(metadataTypeWriter.toString(outputType),
                 is("%type _:Java = {\n  \"message\" : @typeId(\"value\" : \"org.mule.runtime.api.message.Message\") {\n    \"payload\" : {\n      \"name\"? : String, \n      \"lastname\"? : String\n    }, \n    \"attributes\" : Void\n  }, \n  \"variables\" : {\n\n  }\n}"));
    });
  }

  @Test
  @Description("Checks resolution of DataSense static using aggregators")
  public void resolveDataSenseJavaTypeLoaderNotVisibleClass() throws Exception {
    newToolingArtifact("applications/java-type-loader-isolation");

    final DataSenseRequest request = getRequest();
    request.setLocation(Location.builder().globalName("java-type-loader-isolationFlow").build());
    Optional<DataSenseInfo> result = getToolingArtifact().dataSenseService().resolveDataSense(request);

    assertThat(result.isPresent(), is(true));
    DataSenseInfo dataSenseInfo = result.get();
    assertThat(dataSenseInfo.getDataSenseNotifications(), hasSize(2));
    assertThat(dataSenseInfo.getDataSenseNotifications(), containsInAnyOrder(
                                                                             allOf(notificationTypeIs(errorDataSenseNotificationType("ERROR")),
                                                                                   notificationMessageContains("Failed to resolve message payload type expression 'not-visible-class' on element 'logger' custom metadata")),
                                                                             allOf(notificationTypeIs(infoDataSenseNotificationType("INFO")),
                                                                                   notificationMessageContains("Resolving datasense info"))));
  }

  @Test
  @Description("Checks resolution of DataSense static using aggregators")
  public void resolveDataSenseAggregators() throws Exception {
    newToolingArtifact("applications/aggregators-content-child-element");

    final DataSenseRequest request = getRequest();
    request.setLocation(Location.builder().globalName("incrementalAggregationRoute").build());
    Optional<DataSenseInfo> result = getToolingArtifact().dataSenseService().resolveDataSense(request);

    assertThat(result.isPresent(), is(true));
    result.ifPresent(dataSenseInfo -> {
      MetadataTypeWriter metadataTypeWriter = new MetadataTypeWriter();
      DataSenseElementInfo dataSenseElementInfo = dataSenseInfo.getComponentInfoByComponentPath().get()
          .get(Location.builderFromStringRepresentation("incrementalAggregationRoute/processors/2/route/0/processors/0").build());
      assertThat(dataSenseElementInfo, not(nullValue()));
      assertThat(dataSenseElementInfo.getOutput().isPresent(), is(true));
      final MetadataType incomingType = dataSenseElementInfo.getIncoming()
          .orElseThrow(() -> new AssertionError("Expected payload incoming type not present"));
      assertThat(metadataTypeWriter.toString(incomingType),
                 is("%type _:Java = {\n"
                     + "  \"message\" : @typeId(\"value\" : \"org.mule.runtime.api.message.Message\") {\n"
                     + "    \"payload\" : [Number], \n"
                     + "    \"attributes\" : {\n"
                     + "      \"groupId\"? : String, \n"
                     + "      \"firstValueArrivalTime\"? : Number, \n"
                     + "      \"lastValueArrivalTime\"? : Number, \n"
                     + "      \"groupComplete\"? : Boolean\n"
                     + "    }\n"
                     + "  }, \n"
                     + "  \"variables\" : {\n"
                     + "    \"v1\" : Number\n"
                     + "  }\n"
                     + "}"));
    });

    request.setLocation(Location.builder().globalName("aggregatorWithSmallPeriodListener").build());
    result = getToolingArtifact().dataSenseService().resolveDataSense(request);
    result.ifPresent(dataSenseInfo -> {
      MetadataTypeWriter metadataTypeWriter = new MetadataTypeWriter();
      DataSenseElementInfo dataSenseElementInfo = dataSenseInfo.getComponentInfoByComponentPath().get()
          .get(Location.builderFromStringRepresentation("aggregatorWithSmallPeriodListener/source").build());
      assertThat(dataSenseElementInfo, not(nullValue()));
      assertThat(dataSenseElementInfo.getOutput().isPresent(), is(true));
      final MetadataType outputType = dataSenseElementInfo.getOutput()
          .orElseThrow(() -> new AssertionError("Expected payload output type not present"));
      assertThat(metadataTypeWriter.toString(outputType),
                 is("%type _:Java = {\n"
                     + "  \"message\" : @typeId(\"value\" : \"org.mule.runtime.api.message.Message\") {\n"
                     + "    \"payload\" : [Number], \n"
                     + "    \"attributes\" : {\n"
                     + "      \"groupId\"? : String, \n"
                     + "      \"firstValueArrivalTime\"? : Number, \n"
                     + "      \"lastValueArrivalTime\"? : Number, \n"
                     + "      \"groupComplete\"? : Boolean\n"
                     + "    }\n"
                     + "  }, \n"
                     + "  \"variables\" : {\n"
                     + "\n"
                     + "  }\n"
                     + "}"));

    });
  }

  @Test
  @Description("Checks resolution of DataSense when using Secure Properties")
  public void resolveDataSenseUsingSecureProperties() throws Exception {
    newToolingArtifact("applications/properties", ImmutableMap.of("mule.key", "blowkey"));

    DataSenseRequest request = getRequest();
    request.setLocation(Location.builder().globalName("main").build());
    Optional<DataSenseInfo> result = getToolingArtifact().dataSenseService().resolveDataSense(request);

    assertThat(result.isPresent(), is(true));
    Map<Location, DataSenseElementInfo> locationDataSenseElementInfoMap = result.get().getComponentInfoByComponentPath().get();

    DataSenseElementInfo dataSenseElementInfo =
        locationDataSenseElementInfoMap.get(Location.builderFromStringRepresentation("main/processors/0").build());

    MetadataTypeWriter metadataTypeWriter = new MetadataTypeWriter();
    assertThat(dataSenseElementInfo.getOutput().isPresent(), is(true));
    MetadataType outputType = dataSenseElementInfo.getOutput()
        .orElseThrow(() -> new AssertionError("Expected payload output type not present"));
    assertThat(metadataTypeWriter.toString(outputType),
               is("%type _:Java = {\n"
                   + "  \"message\" : @typeId(\"value\" : \"org.mule.runtime.api.message.Message\") {\n"
                   + "    \"payload\" : String\n"
                   + "  }, \n"
                   + "  \"variables\" : {\n"
                   + "\n"
                   + "  }\n"
                   + "}"));

    dataSenseElementInfo =
        locationDataSenseElementInfoMap.get(Location.builderFromStringRepresentation("main/processors/1").build());

    metadataTypeWriter = new MetadataTypeWriter();
    assertThat(dataSenseElementInfo.getOutput().isPresent(), is(true));
    outputType = dataSenseElementInfo.getOutput()
        .orElseThrow(() -> new AssertionError("Expected payload output type not present"));
    assertThat(metadataTypeWriter.toString(outputType),
               is("%type _:Java = {\n"
                   + "  \"message\" : @typeId(\"value\" : \"org.mule.runtime.api.message.Message\") {\n"
                   + "    \"payload\" : @classInformation(\"classname\" : \"java.io.InputStream\", \"hasDefaultConstructor\" : true, \"isInterface\" : false, \"isInstantiable\" : false, \"isAbstract\" : true, \"isFinal\" : false, \"implementedInterfaces\" : [java.io.Closeable], \"parent\" : \"\", \"genericTypes\" : [], \"isMap\" : false) Any, \n"
                   + "    \"attributes\" : @typeId(\"value\" : \"org.mule.extension.http.api.HttpResponseAttributes\") @classInformation(\"classname\" : \"org.mule.extension.http.api.HttpResponseAttributes\", \"hasDefaultConstructor\" : false, \"isInterface\" : false, \"isInstantiable\" : false, \"isAbstract\" : false, \"isFinal\" : false, \"implementedInterfaces\" : [], \"parent\" : \"org.mule.extension.http.api.HttpAttributes\", \"genericTypes\" : [], \"isMap\" : false) @typeAlias(\"value\" : \"HttpResponseAttributes\") {\n"
                   + "      @expressionSupport(\"value\" : SUPPORTED) \"statusCode\" : @classInformation(\"classname\" : \"int\", \"hasDefaultConstructor\" : false, \"isInterface\" : false, \"isInstantiable\" : false, \"isAbstract\" : true, \"isFinal\" : true, \"implementedInterfaces\" : [], \"parent\" : \"\", \"genericTypes\" : [], \"isMap\" : false) @int Number, \n"
                   + "      @expressionSupport(\"value\" : SUPPORTED) \"reasonPhrase\" : String, \n"
                   + "      @expressionSupport(\"value\" : SUPPORTED) \"headers\" : @classInformation(\"classname\" : \"org.mule.runtime.api.util.MultiMap\", \"hasDefaultConstructor\" : true, \"isInterface\" : false, \"isInstantiable\" : true, \"isAbstract\" : false, \"isFinal\" : false, \"implementedInterfaces\" : [java.util.Map, java.io.Serializable], \"parent\" : \"\", \"genericTypes\" : [java.lang.String, java.lang.String], \"isMap\" : true) {\n"
                   + "        * : String\n"
                   + "      }\n"
                   + "    }\n"
                   + "  }, \n"
                   + "  \"variables\" : {\n"
                   + "\n"
                   + "  }\n"
                   + "}"));
  }

  @Test
  @Description("Checks resolution of DataSense static on APIKit app")
  public void resolveDataSenseAPIKit() throws Exception {
    newToolingArtifact("applications/datasense-apikit");

    final DataSenseRequest request = getRequest();
    request.setLocation(Location.builder().globalName("get_books").addProcessorsPart().addIndexPart(0).build());
    Optional<DataSenseInfo> result = getToolingArtifact().dataSenseService().resolveDataSense(request);

    assertThat(result.isPresent(), is(true));
    result.ifPresent(dataSenseInfo -> {
      final MetadataTypeWriter metadataTypeWriter = new MetadataTypeWriter();
      assertThat(dataSenseInfo.getIncoming().isPresent(), is(true));
      assertThat(metadataTypeWriter.toString(dataSenseInfo.getIncoming().get()),
                 is("%type _:Java = {\n  \"message\" : @typeId(\"value\" : \"org.mule.runtime.api.message.Message\") {\n" +
                     "    \"payload\" : Any, \n" +
                     "    \"attributes\" : {\n" +
                     "      \"clientCertificate\"? : {\n" +
                     "        \"publicKey\"? : {\n\n" +
                     "        }, \n" +
                     "        \"type\"? : String, \n" +
                     "        \"encoded\"? : Binary\n" +
                     "      }, \n" +
                     "      \"headers\" : {\n" +
                     "        \"X-Amount\" : String\n" +
                     "      }, \n" +
                     "      \"listenerPath\" : String, \n" +
                     "      \"method\" : String, \n" +
                     "      \"queryParams\" : {\n" +
                     "        \"sortAsc\" : Boolean\n" +
                     "      }, \n" +
                     "      \"queryString\" : String, \n" +
                     "      \"relativePath\" : String, \n" +
                     "      \"remoteAddress\" : String, \n" +
                     "      \"requestPath\" : String, \n" +
                     "      \"requestUri\" : String, \n" +
                     "      \"scheme\" : String, \n" +
                     "      \"uriParams\" : {\n\n" +
                     "      }, \n" +
                     "      \"version\" : String, \n" +
                     "      \"localAddress\" : String\n" +
                     "    }\n" +
                     "  }, \n" +
                     "  \"variables\" : {\n\n" +
                     "  }\n" +
                     "}"));
    });
  }

  @Test
  @Description("Checks resolution of DataSense static on SoapKit app")
  @Ignore
  public void resolveDataSenseSoapKit() throws Exception {
    newToolingArtifact("applications/datasense-soapkit");

    final DataSenseRequest request = getRequest();
    request.setLocation(Location.builder().globalName("GetWeather:\\soapkit-config").build());
    Optional<DataSenseInfo> result = getToolingArtifact().dataSenseService().resolveDataSense(request);

    assertThat(result.isPresent(), is(true));
    result.ifPresent(dataSenseInfo -> {
      final MetadataTypeWriter metadataTypeWriter = new MetadataTypeWriter();
      assertThat(dataSenseInfo.getActualOutput().isPresent(), is(true));
      MuleEventMetadataType metadataType = (MuleEventMetadataType) dataSenseInfo.getActualOutput().get();
      Optional<MetadataType> payloadType = metadataType.getMessageType().getPayloadType();
      assertThat(payloadType.isPresent(), is(true));
      assertThat(metadataTypeWriter.toString(((ObjectType) payloadType.get()).getFieldByName("body").get().getValue()),
                 is(""));
    });
  }

  @Test
  public void reuseImportConfigFromJarAsResource() throws Exception {
    newToolingArtifact("applications/import-config-from-jar-as-resource");

    final DataSenseRequest request = getRequest();
    request.setLocation(Location.builder().globalName("mainFlow").addProcessorsPart().addIndexPart(1).build());
    Optional<DataSenseInfo> result = getToolingArtifact().dataSenseService().resolveDataSense(request);

    assertThat(result.isPresent(), is(true));
    result.ifPresent(dataSenseInfo -> {
      final MetadataTypeWriter metadataTypeWriter = new MetadataTypeWriter();
      assertThat(dataSenseInfo.getIncoming().isPresent(), is(true));
      assertThat(metadataTypeWriter.toString(dataSenseInfo.getIncoming().get()),
                 is("%type _:Java = {\n"
                     + "  \"message\" : @typeId(\"value\" : \"org.mule.runtime.api.message.Message\") {\n"
                     + "    \"payload\" : @classInformation(\"classname\" : \"java.io.InputStream\", \"hasDefaultConstructor\" : true, \"isInterface\" : false, \"isInstantiable\" : false, \"isAbstract\" : true, \"isFinal\" : false, \"implementedInterfaces\" : [java.io.Closeable], \"parent\" : \"\", \"genericTypes\" : [], \"isMap\" : false) Binary, \n"
                     + "    \"attributes\" : @typeId(\"value\" : \"org.mule.extension.http.api.HttpRequestAttributes\") @classInformation(\"classname\" : \"org.mule.extension.http.api.HttpRequestAttributes\", \"hasDefaultConstructor\" : false, \"isInterface\" : false, \"isInstantiable\" : false, \"isAbstract\" : false, \"isFinal\" : false, \"implementedInterfaces\" : [], \"parent\" : \"org.mule.extension.http.api.BaseHttpRequestAttributes\", \"genericTypes\" : [], \"isMap\" : false) @typeAlias(\"value\" : \"HttpRequestAttributes\") {\n"
                     + "      @visibility(\"accessibility\" : READ_ONLY) \"clientCertificate\"? : @typeId(\"value\" : \"java.security.cert.Certificate\") @classInformation(\"classname\" : \"java.security.cert.Certificate\", \"hasDefaultConstructor\" : false, \"isInterface\" : false, \"isInstantiable\" : false, \"isAbstract\" : true, \"isFinal\" : false, \"implementedInterfaces\" : [java.io.Serializable], \"parent\" : \"\", \"genericTypes\" : [], \"isMap\" : false) @typeAlias(\"value\" : \"Certificate\") {\n"
                     + "        @visibility(\"accessibility\" : READ_ONLY) \"encoded\"? : @classInformation(\"classname\" : \"byte[]\", \"hasDefaultConstructor\" : false, \"isInterface\" : false, \"isInstantiable\" : false, \"isAbstract\" : true, \"isFinal\" : true, \"implementedInterfaces\" : [java.lang.Cloneable, java.io.Serializable], \"parent\" : \"\", \"genericTypes\" : [], \"isMap\" : false) Binary, \n"
                     + "        @visibility(\"accessibility\" : READ_ONLY) \"publicKey\"? : @typeId(\"value\" : \"java.security.PublicKey\") @classInformation(\"classname\" : \"java.security.PublicKey\", \"hasDefaultConstructor\" : false, \"isInterface\" : true, \"isInstantiable\" : false, \"isAbstract\" : true, \"isFinal\" : false, \"implementedInterfaces\" : [java.security.Key], \"parent\" : \"\", \"genericTypes\" : [], \"isMap\" : false) @typeAlias(\"value\" : \"PublicKey\") {\n"
                     + "\n"
                     + "        }, \n"
                     + "        @visibility(\"accessibility\" : READ_ONLY) \"type\"? : String\n"
                     + "      }, \n"
                     + "      @visibility(\"accessibility\" : READ_ONLY) \"headers\"? : @classInformation(\"classname\" : \"org.mule.runtime.api.util.MultiMap\", \"hasDefaultConstructor\" : true, \"isInterface\" : false, \"isInstantiable\" : true, \"isAbstract\" : false, \"isFinal\" : false, \"implementedInterfaces\" : [java.util.Map, java.io.Serializable], \"parent\" : \"\", \"genericTypes\" : [java.lang.String, java.lang.String], \"isMap\" : true) {\n"
                     + "        * : String\n"
                     + "      }, \n"
                     + "      @visibility(\"accessibility\" : READ_ONLY) \"listenerPath\"? : String, \n"
                     + "      @visibility(\"accessibility\" : READ_ONLY) \"method\"? : String, \n"
                     + "      @visibility(\"accessibility\" : READ_ONLY) \"queryParams\"? : @classInformation(\"classname\" : \"org.mule.runtime.api.util.MultiMap\", \"hasDefaultConstructor\" : true, \"isInterface\" : false, \"isInstantiable\" : true, \"isAbstract\" : false, \"isFinal\" : false, \"implementedInterfaces\" : [java.util.Map, java.io.Serializable], \"parent\" : \"\", \"genericTypes\" : [java.lang.String, java.lang.String], \"isMap\" : true) {\n"
                     + "        * : String\n"
                     + "      }, \n"
                     + "      @visibility(\"accessibility\" : READ_ONLY) \"queryString\"? : String, \n"
                     + "      @visibility(\"accessibility\" : READ_ONLY) \"relativePath\"? : String, \n"
                     + "      @visibility(\"accessibility\" : READ_ONLY) \"remoteAddress\"? : String, \n"
                     + "      @visibility(\"accessibility\" : READ_ONLY) \"requestPath\"? : String, \n"
                     + "      @visibility(\"accessibility\" : READ_ONLY) \"requestUri\"? : String, \n"
                     + "      @visibility(\"accessibility\" : READ_ONLY) \"scheme\"? : String, \n"
                     + "      @visibility(\"accessibility\" : READ_ONLY) \"uriParams\"? : @classInformation(\"classname\" : \"java.util.Map\", \"hasDefaultConstructor\" : false, \"isInterface\" : true, \"isInstantiable\" : false, \"isAbstract\" : true, \"isFinal\" : false, \"implementedInterfaces\" : [], \"parent\" : \"\", \"genericTypes\" : [java.lang.String, java.lang.String], \"isMap\" : true) {\n"
                     + "        * : String\n"
                     + "      }, \n"
                     + "      @visibility(\"accessibility\" : READ_ONLY) \"version\"? : String\n"
                     + "    }\n"
                     + "  }, \n"
                     + "  \"variables\" : {\n"
                     + "    \"X_REQUEST_ID\" : @example(\"value\" : \"{\\\"success\\\":true,\\\"deck_id\\\":\\\"3p40paa87x90\\\",\\\"shuffled\\\":true,\\\"remaining\\\":52}\") @typeAlias(\"value\" : \"ShuffleResult\") {\n"
                     + "      \"success\" : Boolean, \n"
                     + "      \"deck_id\" : String, \n"
                     + "      \"shuffled\" : Boolean, \n"
                     + "      \"remaining\" : @typeId(\"value\" : \"int\") Number\n"
                     + "    }\n"
                     + "  }\n"
                     + "}"));
    });
  }

  @Test
  public void resolveDataSenseApiSyncTestV8() throws Exception {
    newToolingArtifact("applications/test-api-v8");

    final DataSenseRequest request = getRequest();
    request.setLocation(Location.builder().globalName("get:\\test:test-api-config").addProcessorsPart().addIndexPart(0).build());
    Optional<DataSenseInfo> result = getToolingArtifact().dataSenseService().resolveDataSense(request);

    assertThat(result.isPresent(), is(true));
    DataSenseInfo dataSenseInfo = result.get();
    assertThat(dataSenseInfo.getDataSenseNotifications(), hasSize(2));
    assertThat(dataSenseInfo.getDataSenseNotifications(), containsInAnyOrder(
                                                                             allOf(notificationTypeIs(errorDataSenseNotificationType("ERROR")),
                                                                                   notificationMessageContains("Raml resolution error: Error reading RAML document 'resource::com.mycompany:test-api:1.1.0-SNAPSHOT:raml:zip:test-api.raml'. Detail: URI is not hierarchical")),
                                                                             allOf(notificationTypeIs(infoDataSenseNotificationType("INFO")),
                                                                                   notificationMessageContains("Resolving datasense info"))));
  }

  @Test
  public void resolveDataSenseNoConfigurationProvidedForEmailPort() throws Exception {
    newToolingArtifact("applications/datasense-static-no-property");

    final DataSenseRequest request = getRequest();
    request.setLocation(Location.builder().globalName("flow").addProcessorsPart().addIndexPart(0).build());
    Optional<DataSenseInfo> result = getToolingArtifact().dataSenseService().resolveDataSense(request);

    assertThat(result.isPresent(), is(true));
  }

  @Test
  public void resolveDataSenseNoConfigurationProvidedForEmailPortAndNoPropertiesFile() throws Exception {
    newToolingArtifact("applications/datasense-static-no-property-file");

    final DataSenseRequest request = getRequest();
    request.setLocation(Location.builder().globalName("flow").addProcessorsPart().addIndexPart(0).build());
    expectedException
        .expectMessage("Couldn't find configuration properties file properties.yaml neither on classpath or in file system");
    getToolingArtifact().dataSenseService().resolveDataSense(request);
  }

  @Test
  public void resolveDataSenseNoConfigurationProvidedForEmailPortAndPropertiesFileMissingProperty() throws Exception {
    newToolingArtifact("applications/datasense-static-property-file-missing-property");

    final DataSenseRequest request = getRequest();
    request.setLocation(Location.builder().globalName("flow").addProcessorsPart().addIndexPart(0).build());

    Optional<DataSenseInfo> result = getToolingArtifact().dataSenseService().resolveDataSense(request);

    assertThat(result.isPresent(), is(true));
  }

  @Test
  public void resolveDataSenseEEComponent() throws Exception {
    newToolingArtifact();

    final DataSenseRequest request = getRequest();
    request.setLocation(Location.builder().globalName("staticDataSenseFlowEE").addProcessorsPart().addIndexPart(1).build());
    Optional<DataSenseInfo> result = getToolingArtifact().dataSenseService().resolveDataSense(request);

    assertThat(result.isPresent(), is(true));
  }

  @Test
  public void resolveJmsWithRequiredLibraryFromApp() throws Exception {
    newToolingArtifact();

    final DataSenseRequest request = getRequest();
    request.setLocation(Location.builder().globalName("jmsStaticDataSense").addProcessorsPart().addIndexPart(0).build());
    Optional<DataSenseInfo> result = getToolingArtifact().dataSenseService().resolveDataSense(request);

    assertThat(result.isPresent(), is(true));
  }

}
