/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.tests.integration.tooling.client;

import static java.lang.Thread.currentThread;
import static java.util.Collections.emptyMap;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsNot.not;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;
import static org.junit.rules.ExpectedException.none;
import static org.mule.maven.client.test.MavenTestHelper.createDefaultEnterpriseMavenConfiguration;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getMuleVersion;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getTestLog4JConfigurationFile;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getToolingVersion;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrap;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapConfiguration;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapFactory;
import org.mule.tooling.client.tests.integration.category.DoesNotNeedMuleRuntimeTest;

import java.net.URL;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;

@Category(DoesNotNeedMuleRuntimeTest.class)
@Feature("ToolingExceptionSerialization")
@Story("Test for serialization of ToolingException")
public class ToolingExceptionSerializationTestCase {

  private ToolingRuntimeClientBootstrap bootstrap;
  private ToolingRuntimeClient toolingRuntimeClient;
  @Rule
  public ExpectedException expectedException = none();
  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  @Before
  public void setUpToolingRuntimeClient() throws Exception {
    bootstrap = ToolingRuntimeClientBootstrapFactory.newToolingRuntimeClientBootstrap(
                                                                                      ToolingRuntimeClientBootstrapConfiguration
                                                                                          .builder()
                                                                                          .muleVersion(getMuleVersion())
                                                                                          .toolingVersion(getToolingVersion())
                                                                                          .mavenConfiguration(createDefaultEnterpriseMavenConfiguration())
                                                                                          .log4jConfiguration(getTestLog4JConfigurationFile())
                                                                                          .workingFolder(
                                                                                                         temporaryFolder
                                                                                                             .newFolder())
                                                                                          .build());
    toolingRuntimeClient = bootstrap.getToolingRuntimeClientBuilderFactory().create().build();
  }

  @After
  public void disposeBootstrap() throws Exception {
    if (bootstrap != null) {
      bootstrap.dispose();
    }
  }

  @Test
  @Description("Checks the Java exceptions are just passed to clients without serialization")
  public void requiredToolingArtifactUrl() throws Exception {
    expectedException.expect(NullPointerException.class);
    expectedException.expectMessage("artifactUrlContent cannot be null");
    toolingRuntimeClient.newToolingArtifact(null, emptyMap());
  }

  @Test
  @Description("Checks the root cause type and root cause message is captured in a ToolingException")
  public void propagateRootCauseInformation() throws Exception {
    try {
      toolingRuntimeClient.newToolingArtifact(getResource("applications/empty-application"), emptyMap());
      fail();
    } catch (ToolingException e) {
      assertThat(e.getMessage(), containsString("Couldn't load pom model from artifact"));
      assertThat(e.getRootCauseMessage(), containsString("The maven bundle loader requires the file pom.xml"));
      assertThat(e.getRootCauseType(), is("org.mule.maven.client.api.BundleDescriptorCreationException"));
      assertThat(e.getRootCauseStackTrace(), not(nullValue()));
    }
  }

  @Test
  @Description("Checks the when an internal Mule runtime exception is thrown a ToolingException is generated")
  public void wrappedInternalExceptionFromRuntimeWithToolingException() throws Exception {
    expectedException.expect(ToolingException.class);
    expectedException.expectMessage("ModelBuildingException: 1 problem was encountered while building the effective model");
    toolingRuntimeClient.newToolingArtifact(getResource("applications/bad-artifact-descriptor"), emptyMap());
  }

  private URL getResource(String name) {
    URL url = currentThread().getContextClassLoader().getResource(name);
    assertTrue("Missing required resource: " + name, url != null);
    return url;
  }

}
