/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.tests.integration.tooling.client;

import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static org.apache.commons.lang3.tuple.ImmutablePair.of;
import static org.hamcrest.CoreMatchers.hasItems;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.core.Is.is;
import static org.junit.rules.ExpectedException.none;
import static org.mule.tck.junit4.matcher.IsEmptyOptional.empty;
import static org.mule.tooling.client.test.junit4.ValueMatcher.valueWithId;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.component.location.Location;
import org.mule.tooling.client.api.value.ResolvingFailure;
import org.mule.tooling.client.api.value.Value;
import org.mule.tooling.client.api.value.ValueResult;
import org.mule.tooling.client.api.value.provider.ValueProviderRequest;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrap;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapConfiguration;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapFactory;
import org.mule.tooling.client.test.AbstractMuleRuntimeTestCase;
import org.mule.tooling.client.test.RuntimeType;

import java.io.File;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.apache.commons.lang3.tuple.ImmutablePair;
import org.hamcrest.Matchers;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

public class ValueProviderTestCase extends AbstractMuleRuntimeTestCase {

  private static final String INVALID_LOCATION = "INVALID_LOCATION";
  private static final String INVALID_VALUE_RESOLVER_NAME = "INVALID_VALUE_RESOLVER_NAME";

  @Rule
  public ExpectedException expectedException = none();

  @Rule
  public DynamicPort emailServerPort = new DynamicPort("emailServerPort");

  private ToolingRuntimeClientBootstrap bootstrap;
  private ToolingRuntimeClient toolingRuntimeClient;
  private ToolingArtifact toolingArtifact;

  public ValueProviderTestCase(RuntimeType runtimeType) {
    super(runtimeType);
  }

  @Override
  protected List<ImmutablePair<String, String>> getStartupSystemProperties() {
    return singletonList(of("emailServerPort", String.valueOf(emailServerPort.getNumber())));
  }

  @Before
  public void setUpToolingRuntimeClient() throws Exception {
    bootstrap = ToolingRuntimeClientBootstrapFactory.newToolingRuntimeClientBootstrap(
                                                                                      ToolingRuntimeClientBootstrapConfiguration
                                                                                          .builder()
                                                                                          .muleVersion(getMuleVersion())
                                                                                          .toolingVersion(getToolingVersion())
                                                                                          .mavenConfiguration(defaultMavenConfiguration)
                                                                                          .log4jConfiguration(getTestLog4JConfigurationFile())
                                                                                          .workingFolder(temporaryFolder
                                                                                              .newFolder())
                                                                                          .build());

    toolingRuntimeClient = bootstrap.getToolingRuntimeClientBuilderFactory().create()
        .withMavenConfiguration(defaultMavenConfiguration)
        .withRemoteAgentConfiguration(defaultAgentConfiguration)
        .build();
  }

  @After
  public void disposeApplication() throws Exception {
    if (toolingArtifact != null) {
      toolingArtifact.dispose();
    }
  }

  @After
  public void disposeBootstrap() throws Exception {
    if (bootstrap != null) {
      bootstrap.dispose();
    }
  }

  @Test
  public void resolveValuesForEmailContentTransferEncoding() throws Exception {
    newToolingArtifact("value-providers");
    ValueProviderRequest valueProviderRequest = new ValueProviderRequest(Location.builder().globalName("sendEmail")
        .addProcessorsPart().addIndexPart(0).build().toString(), "contentTransferEncoding");
    ValueResult values = toolingArtifact.valueProviderService().getValues(valueProviderRequest);

    assertThat(values.isSuccess(), is(true));
    assertThat(values.getValues(), is(hasItems(valueWithId("7BIT"), valueWithId("8BIT"), valueWithId("Base64"),
                                               valueWithId("Quoted-Printable"), valueWithId("Binary"))));
  }

  @Test
  public void resolveConnectionValuesForWebServiceConsumerWeather() throws Exception {
    newToolingArtifact("value-providers");
    ValueProviderRequest wscRequest =
        new ValueProviderRequest(Location.builder().globalName("weather").addConnectionPart().build().toString(), "Connection");
    ValueResult result = toolingArtifact.valueProviderService().getValues(wscRequest);

    assertThat(result.isSuccess(), is(true));
    assertThat(result.getValues(), hasItems(valueWithId("GlobalWeather").withPartName("service")
        .withChilds(valueWithId("GlobalWeatherHttpGet").withPartName("port")
            .withChilds(valueWithId(Matchers.startsWith("http://www.webservicex.com/globalweather.asmx"))
                .withPartName("address")))));
  }

  @Test
  public void resolveConnectionValuesForWebServiceConsumerWeatherDeclaredOnDomain() throws Exception {
    newToolingArtifactForDomain("value-providers");
    ValueProviderRequest wscRequest =
        new ValueProviderRequest(Location.builder().globalName("weather").addConnectionPart().build().toString(), "Connection");
    ValueResult result = toolingArtifact.valueProviderService().getValues(wscRequest);

    assertThat(result.isSuccess(), is(true));
    assertThat(result.getValues(), hasItems(valueWithId("GlobalWeather").withPartName("service")
        .withChilds(valueWithId("GlobalWeatherHttpGet").withPartName("port")
            .withChilds(valueWithId(Matchers.startsWith("http://www.webservicex.com/globalweather.asmx"))
                .withPartName("address")))));
  }

  @Test
  public void resolveConnectionValuesForWebServiceConsumerHumanResources() throws Exception {
    newToolingArtifact("value-providers");
    ValueProviderRequest wscRequest =
        new ValueProviderRequest(Location.builder().globalName("human").addConnectionPart().build().toString(), "Connection");
    ValueResult result = toolingArtifact.valueProviderService().getValues(wscRequest);

    Set<Value> values = result.getValues();
    assertThat(values, hasItems(valueWithId("Human_ResourcesService").withPartName("service")
        .withChilds(valueWithId("Human_Resources").withPartName("port"))));
  }

  @Test
  public void resolveWithInvalidLocation() throws Exception {
    newToolingArtifact("value-providers");
    ValueProviderRequest valueProviderRequest = new ValueProviderRequest(Location.builder().globalName("unknown")
        .addProcessorsPart().addIndexPart(0).build().toString(), "contentTransferEncoding");
    ValueResult values = toolingArtifact.valueProviderService().getValues(valueProviderRequest);

    assertThat(values.isSuccess(), is(false));
    Optional<ResolvingFailure> optionalFailure = values.getFailure();
    assertThat(optionalFailure, is(not(empty())));

    ResolvingFailure resolvingFailure = optionalFailure.get();
    assertThat(resolvingFailure.getFailureCode(), is(INVALID_LOCATION));
  }

  @Test
  public void resolveWithInvalidProviderName() throws Exception {
    newToolingArtifact("value-providers");
    ValueProviderRequest valueProviderRequest = new ValueProviderRequest(Location.builder().globalName("sendEmail")
        .addProcessorsPart().addIndexPart(0).build().toString(), "invalidName");
    ValueResult values = toolingArtifact.valueProviderService().getValues(valueProviderRequest);

    assertThat(values.isSuccess(), is(false));
    Optional<ResolvingFailure> optionalFailure = values.getFailure();
    assertThat(optionalFailure, is(not(empty())));

    ResolvingFailure resolvingFailure = optionalFailure.get();
    assertThat(resolvingFailure.getFailureCode(), is(INVALID_VALUE_RESOLVER_NAME));
  }

  private void newToolingArtifact(String applicationName) throws Exception {
    newToolingArtifact("applications", applicationName);
  }

  private void newToolingArtifactForDomain(String domainName) throws Exception {
    newToolingArtifact("domains", domainName);
  }

  private void newToolingArtifact(String type, String applicationName) throws Exception {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File applicationLocation = new File(targetTestClassesFolder, type + "/" + applicationName);
    toolingArtifact = newToolingArtifact(applicationLocation);

    assertThat(toolingArtifact, not(nullValue()));
  }

  protected ToolingArtifact newToolingArtifact(File appLocation) throws Exception {
    return toolingRuntimeClient.newToolingArtifact(toUrl(appLocation.toURI()), emptyMap());
  }
}
