/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.tests.junit;

import static org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapFactory.newToolingRuntimeClientBootstrap;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrap;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapConfiguration;

import java.io.File;
import java.lang.annotation.Annotation;
import java.net.URI;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

import org.junit.runner.Runner;
import org.junit.runner.notification.RunNotifier;
import org.junit.runners.BlockJUnit4ClassRunner;
import org.junit.runners.Suite;
import org.junit.runners.model.InitializationError;
import org.junit.runners.model.RunnerScheduler;
import org.junit.runners.model.Statement;

/**
 * Runs JUnit4 tests with one or more Tooling Runtime Client and Mule Runtime versions. The test class must have public constructor with the following parameters:
 * <ul>
 *   <li>{@link MavenConfiguration.MavenConfigurationBuilder mavenConfigurationBuilder}</li>
 *   <li>{@link ToolingRuntimeClientBootstrapProvider toolingRuntimeClientBootstrapProvider}</li>
 *   <li>{@link File muleHome}</li>
 * </ul>
 * <p/>
 * Versions for Tooling Runtime Client and Mule Runtime versions should be defined by {@link ToolingRuntimeClientVersion}.
 */
public class ToolingJUnitTestRunner extends Suite {

  private static final List<Runner> NO_RUNNERS = Collections.<Runner>emptyList();

  private final List<Runner> runners;


  /**
   * {@link BlockJUnit4ClassRunner} implementation that allows to run the same tests using different versions of
   * Mule Runtime and Tooling Runtime Client.
   */
  private static class SingleRunner extends BlockJUnit4ClassRunner {

    private final File muleHome;

    private final String muleVersion;

    private final String toolingRuntimeClientVersion;

    private URI log4jConfiguration;

    private final String name;

    private ToolingRuntimeClientBootstrap toolingRuntimeClientBootstrap;

    private Object testInstance;

    private AtomicBoolean singleTestInstanceCreated = new AtomicBoolean(false);

    SingleRunner(Class<?> klass,
                 File muleHome,
                 String toolingRuntimeClientVersion,
                 URI log4jConfiguration,
                 String name)
        throws InitializationError {
      super(klass);
      if (!muleHome.exists() || !muleHome.isDirectory()) {
        throw new IllegalArgumentException("Invalid Mule Runtime home location");
      }
      this.muleHome = muleHome;
      this.muleVersion = getMuleVersion(muleHome.getName());
      this.toolingRuntimeClientVersion = toolingRuntimeClientVersion;
      this.log4jConfiguration = log4jConfiguration;
      this.name = name;

      setScheduler(new RunnerScheduler() {

        @Override
        public void schedule(Runnable childStatement) {
          childStatement.run();
        }

        @Override
        public void finished() {
          getTestClass().getAnnotatedMethods(AfterTests.class).stream().forEach(afterTests -> {
            try {
              afterTests.invokeExplosively(testInstance);
            } catch (Throwable throwable) {
              throw new RuntimeException(name + " Error while running AfterTests", throwable);
            }
          });

          if (toolingRuntimeClientBootstrap != null) {
            toolingRuntimeClientBootstrap.dispose();
          }
        }

      });
    }

    @Override
    protected Object createTest() throws Exception {
      // Only one instance of the Tests should be created...
      if (singleTestInstanceCreated.compareAndSet(false, true)) {
        ToolingRuntimeClientBootstrapConfiguration.ToolingRuntimeClientBootstrapConfigurationBuilder bootstrapConfigurationBuilder =
            ToolingRuntimeClientBootstrapConfiguration.builder();
        MavenConfiguration.MavenConfigurationBuilder mavenConfigurationBuilder =
            MavenConfiguration.newMavenConfigurationBuilder();

        testInstance = getTestClass().getJavaClass()
            .getConstructor(MavenConfiguration.MavenConfigurationBuilder.class, ToolingRuntimeClientBootstrapProvider.class,
                            File.class)
            .newInstance(mavenConfigurationBuilder, (ToolingRuntimeClientBootstrapProvider) () -> {
              if (toolingRuntimeClientBootstrap == null) {
                throw new IllegalStateException(
                                                name + " Cannot access to the bootstrap instance at this moment, should be accessed once the test class is constructed");
              }
              return toolingRuntimeClientBootstrap;
            }, muleHome);

        bootstrapConfigurationBuilder.muleVersion(muleVersion);
        bootstrapConfigurationBuilder.toolingVersion(toolingRuntimeClientVersion);
        bootstrapConfigurationBuilder.log4jConfiguration(log4jConfiguration);
        bootstrapConfigurationBuilder.mavenConfiguration(mavenConfigurationBuilder.build());
        bootstrapConfigurationBuilder.workingFolder(Files.createTempDirectory("workingDir").toFile());
        this.toolingRuntimeClientBootstrap = newToolingRuntimeClientBootstrap(bootstrapConfigurationBuilder.build());

        // Just run once the beforeTests
        runBeforeTests();
      }
      return testInstance;
    }

    private void runBeforeTests() {
      getTestClass().getAnnotatedMethods(BeforeTests.class).stream().forEach(beforeTests -> {
        try {
          beforeTests.invokeExplosively(testInstance);
        } catch (Throwable throwable) {
          throw new RuntimeException(name + " Error while running BeforeTests", throwable);
        }
      });
    }

    @Override
    protected void validateZeroArgConstructor(List<Throwable> errors) {
      try {
        getTestClass().getJavaClass().getConstructor(MavenConfiguration.MavenConfigurationBuilder.class,
                                                     ToolingRuntimeClientBootstrapProvider.class, File.class);
      } catch (NoSuchMethodException e) {
        errors.add(e);
      }
    }

    @Override
    protected String getName() {
      return "[" + name + "]";
    }

    @Override
    protected Annotation[] getRunnerAnnotations() {
      return new Annotation[0];
    }

    @Override
    protected Statement classBlock(RunNotifier notifier) {
      return childrenInvoker(notifier);
    }

  }

  public ToolingJUnitTestRunner(Class<?> clazz) throws Throwable {
    super(clazz, NO_RUNNERS);
    this.runners = getRunners(clazz);
  }

  private static List<Runner> getRunners(final Class<?> clazz) throws Throwable {
    final List<Throwable> errors = new ArrayList<>();
    final List<Runner> runners = new ArrayList<>();

    ToolingRuntimeClientVersion[] toolingRuntimeClientVersions = clazz.getAnnotationsByType(ToolingRuntimeClientVersion.class);
    if (toolingRuntimeClientVersions.length > 0) {
      for (ToolingRuntimeClientVersion toolingRuntimeClientVersion : toolingRuntimeClientVersions) {
        File targetFolder =
            new File(ToolingJUnitTestRunner.class.getProtectionDomain().getCodeSource().getLocation().toURI()).getParentFile();
        File muleHome = new File(targetFolder, toolingRuntimeClientVersion.muleHome());
        if (muleHome.isDirectory()) {
          final URI log4jConfiguration =
              ToolingJUnitTestRunner.class.getClassLoader().getResource(toolingRuntimeClientVersion.log4Configuration()).toURI();
          String testName = createTestName(muleHome.getName(), toolingRuntimeClientVersion.toolingVersion());
          runners.add(new SingleRunner(clazz, muleHome, toolingRuntimeClientVersion.toolingVersion(),
                                       log4jConfiguration,
                                       testName));
        } else {
          errors.add(new Exception("Invalid mule installation location " + muleHome + ", configuration:"
              + toolingRuntimeClientVersion));
        }
      }
    }
    if (!errors.isEmpty()) {
      throw new InitializationError(errors);
    }

    if (runners.isEmpty()) {
      throw new InitializationError(new Exception("No configured test mule runtime"));
    }

    return runners;
  }

  private static String createTestName(String muleVersion, String toolingVersion) {
    return "Mule-" + muleVersion + "/Tooling-" + toolingVersion;
  }

  private static String getMuleVersion(String muleHomeFolderName) {
    final String delimiter = "-";
    final String[] muleHomeFolderNameSplit = muleHomeFolderName.split(delimiter);
    return String.join(delimiter, Arrays.copyOfRange(muleHomeFolderNameSplit, 3, muleHomeFolderNameSplit.length));
  }

  @Override
  protected List<Runner> getChildren() {
    return runners;
  }

}
