/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.tests.integration.tooling.client;

import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.junit.Assert.assertThat;
import static org.mule.tooling.client.api.el.Severity.ERROR;
import static org.mule.tooling.client.api.el.Severity.WARNING;
import static org.mule.tooling.client.test.RuntimeType.NONE;
import org.mule.metadata.api.model.MetadataType;
import org.mule.tooling.client.api.component.location.Location;
import org.mule.tooling.client.api.datasense.DataSenseInfo;
import org.mule.tooling.client.api.datasense.DataSenseRequest;
import org.mule.tooling.client.api.dataweave.DataWeaveService;
import org.mule.tooling.client.api.dataweave.validation.DataWeaveValidationRequest;
import org.mule.tooling.client.api.el.Position;
import org.mule.tooling.client.api.el.ValidationMessage;
import org.mule.tooling.client.api.el.ValidationResult;
import org.mule.tooling.client.test.RuntimeType;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Optional;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Test;
import org.junit.runners.Parameterized;

@Feature("DataWeaveValidationService")
@Story("Integration tests for DataWeave validation using ToolingBootstrap and ToolingRuntimeClient")
public class DataWeaveValidationTestCase extends DataWeaveTestCase {

  public DataWeaveValidationTestCase(RuntimeType runtimeType) {
    super(runtimeType);
  }

  @Parameterized.Parameters(name = "{0}")
  public static Collection<Object[]> data() {
    return Arrays.asList(new Object[][] {
        {NONE}
    });
  }

  @Test
  @Description("Validation using custom types, globalBindings, functionBindings and eventType")
  public void validateUsingGlobalBindingsFunctionsBindingsAndEventType() throws Exception {
    doWithToolingArtifact(localToolingRuntimeClient(), getApplicationUrlContent("applications/dataweave-validation-bindings"),
                          toolingArtifact -> {
                            final DataSenseRequest dataSenseRequest = new DataSenseRequest();
                            dataSenseRequest.setLocation(Location.builder().globalName("new_flow").addProcessorsPart()
                                .addIndexPart(0).build());
                            final Optional<DataSenseInfo> dataSenseInfoOptional =
                                toolingArtifact.dataSenseService().resolveDataSense(dataSenseRequest);
                            assertThat(dataSenseInfoOptional.isPresent(), is(true));

                            final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

                            final DataSenseInfo dataSenseInfo = dataSenseInfoOptional.get();
                            assertThat(dataSenseInfo.getOutput().isPresent(), is(true));
                            final MetadataType outputType = dataSenseInfo.getOutput().get();
                            final DataWeaveValidationRequest.Builder builder = DataWeaveValidationRequest.builder()
                                .withFunctionBindings(dataSenseInfo.getFunctionBindings())
                                .withGlobalBindings(dataSenseInfo.getGlobalBindings())
                                .withEventType(outputType);

                            // success field from payload (custom type)
                            ValidationResult execute = dataWeaveService.validate(builder
                                .withScript("%dw 2.0\n"
                                    + "---\n"
                                    + "payload.success\n")
                                .build());
                            assertThat(execute.isSuccess(), is(true));
                            assertThat(execute.getMessages(), hasSize(0));

                            // invalid is not a present field on payload (custom type)
                            execute = dataWeaveService.validate(builder
                                .withScript("%dw 2.0\n"
                                    + "---\n"
                                    + "payload.invalid\n")
                                .build());
                            assertThat(execute.isSuccess(), is(true));
                            assertThat(execute.getMessages(), hasSize(1));
                            assertThat(execute.getMessages().get(0).getMessage(),
                                       containsString("Property: `invalid` was not found"));
                            assertThat(execute.getMessages().get(0).getSeverity(), equalTo(WARNING));

                            // globalBindings from runtime, app.name should be valid
                            execute = dataWeaveService.validate(builder
                                .withScript("%dw 2.0\n"
                                    + "---\n"
                                    + "app.name\n")
                                .build());
                            assertThat(execute.isSuccess(), is(true));
                            assertThat(execute.getMessages(), hasSize(0));

                            // runtime function, p("test")
                            execute = dataWeaveService.validate(builder
                                .withScript("%dw 2.0\n"
                                    + "---\n"
                                    + "p(\"test\")\n")
                                .build());

                            assertThat(execute.isSuccess(), is(true));
                            assertThat(execute.getMessages(), hasSize(0));

                            // runtime function, p(100)
                            execute = dataWeaveService.validate(builder
                                .withScript("%dw 2.0\n"
                                    + "---\n"
                                    + "p(100)\n")
                                .build());

                            assertThat(execute.isSuccess(), is(true));
                            assertThat(execute.getMessages(), hasSize(1));
                            assertThat(execute.getMessages().get(0).getMessage(),
                                       containsString("Auto-Coercing type from: Number to: String"));
                            assertThat(execute.getMessages().get(0).getSeverity(), equalTo(WARNING));

                            // wrong runtime function, p1("test")
                            execute = dataWeaveService.validate(builder
                                .withScript("%dw 2.0\n"
                                    + "---\n"
                                    + "p1(\"test\")\n")
                                .build());
                            assertThat(execute.isSuccess(), is(false));
                            assertThat(execute.getMessages(), hasSize(1));
                            assertThat(execute.getMessages().get(0).getMessage(),
                                       containsString("Unable to resolve reference of p1"));
                            assertThat(execute.getMessages().get(0).getSeverity(), equalTo(ERROR));
                          });
  }


  @Test
  @Description("Check simple script execution")
  public void validateDataWeaveSimpleScript() throws Exception {
    doWithToolingArtifact(localToolingRuntimeClient(), getApplicationUrlContent(POJO_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      final String script = "%dw 2.0\n"
          + "output application/json\n"
          + "---\n"
          + "{\n"
          + "a: 1\n"
          + "}";

      final DataWeaveValidationRequest build = DataWeaveValidationRequest.builder().withScript(script).build();

      final ValidationResult execute = dataWeaveService.validate(build);

      assertThat(execute.getMessages(), hasSize(0));
    });
  }

  @Test
  @Description("as Object is not validating the fields when cast is applied")
  public void validateDataWeaveScriptWithPojos() throws Exception {
    doValidateDataWeaveScriptWithPojos(POJO_APP_LOCATION);
  }

  private void doValidateDataWeaveScriptWithPojos(String artifactContent) throws Exception {
    doWithToolingArtifact(localToolingRuntimeClient(), getApplicationUrlContent(artifactContent), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      final String script = "output application/java\n"
          + "---\n"
          + "{\n" +
          "nameInvalidField: 'Matias',\n" +
          "age: 29\n" +
          "} as Object { class: \"pojo.Poyito\" }";

      final DataWeaveValidationRequest build = DataWeaveValidationRequest.builder().withScript(script).build();

      final ValidationResult execute = dataWeaveService.validate(build);

      assertThat(execute.getMessages(), hasSize(0));
    });
  }

  @Test
  @Description("as Object is not validating the fields when cast is applied using Domains")
  public void validateDataWeaveScriptWithPojosOnDomains() throws Exception {
    doValidateDataWeaveScriptWithPojos(POJO_DOMAIN_LOCATION);
  }

  @Test
  @Description("Check simple script execution")
  public void validateDataWeaveScriptWithImports() throws Exception {
    doWithToolingArtifact(localToolingRuntimeClient(), getApplicationUrlContent(IMPORT_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      final String script =
          "output application/json\n"
              + "import mymodule\n"
              + "import doubleNumber from java!my::group::MyStaticClass\n"
              + "---\n"
              + "doubleNumber(mymodule::myvar)";

      final DataWeaveValidationRequest build = DataWeaveValidationRequest.builder().withScript(script).build();

      final ValidationResult execute = dataWeaveService.validate(build);

      assertThat(execute.getMessages(), hasSize(0));
    });
  }

  @Test
  @Description("Check simple script execution")
  public void validateDataWeaveScriptWithImportsInvalidParameters() throws Exception {
    doWithToolingArtifact(localToolingRuntimeClient(), getApplicationUrlContent(IMPORT_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      final String script =
          "output application/json\n"
              + "import java!my::group::MyStaticClass\n"
              + "---\n"
              + "MyStaticClass::doubleNumber('asdf', 'fdsa')";

      final DataWeaveValidationRequest build = DataWeaveValidationRequest.builder().withScript(script).build();

      final ValidationResult execute = dataWeaveService.validate(build);

      final List<ValidationMessage> errors = execute.getMessages();
      assertThat(errors.size(), is(1));
      final Position startPosition = errors.get(0).getLocation().getStartPosition();
      assertThat(startPosition.getLine(), is(4));
      assertThat(startPosition.getColumn(), is(1));
      assertThat(startPosition.getOffset(), is(65));
      final Position endPosition = errors.get(0).getLocation().getEndPosition();
      assertThat(endPosition.getLine(), is(4));
      assertThat(endPosition.getColumn(), is(44));
      assertThat(endPosition.getOffset(), is(108));
      assertThat(errors.get(0).getMessage(), containsString("Too many parameters"));
    });
  }

  @Test
  @Description("Syntax validation")
  public void syntaxtValidation() throws Exception {
    doWithToolingArtifact(remoteToolingRuntimeClient(), getApplicationUrlContent(IMPORT_APP_LOCATION), toolingArtifact -> {
      final DataWeaveService dataWeaveService = toolingArtifact.dataWeaveService();

      final String script =
          "output application/json\n"
              + "import java!my::group::MyStaticClass\n"
              + "---\n"
              + "MyStaticClass::doubleNumber('asdf', 'fdsa'";

      final DataWeaveValidationRequest build = DataWeaveValidationRequest.builder().withScript(script).build();

      final ValidationResult result = dataWeaveService.validate(build);

      final List<ValidationMessage> errors = result.getMessages();
      assertThat(errors.size(), is(1));

      final ValidationMessage validationMessage = result.getMessages().get(0);
      assertThat(validationMessage.getMessage(), containsString("Unexpected end of input"));
      assertThat(validationMessage.getSeverity(), equalTo(ERROR));
    });
  }


}
