/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.tests.integration.tooling.client.metadata;

import static java.lang.System.clearProperty;
import static java.lang.System.setProperty;
import static java.nio.channels.FileChannel.open;
import static java.nio.charset.Charset.defaultCharset;
import static java.nio.file.StandardOpenOption.WRITE;
import static java.util.Optional.empty;
import static org.apache.commons.io.FileUtils.cleanDirectory;
import static org.apache.commons.lang3.tuple.ImmutablePair.of;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.hasItem;
import static org.hamcrest.Matchers.not;
import static org.junit.Assert.assertThat;
import static org.mule.tooling.client.api.component.location.Location.builder;
import static org.mule.tooling.client.api.datasense.DataSenseNotificationType.errorDataSenseNotificationType;
import static org.mule.tooling.client.tests.integration.tooling.client.DataSenseNotificationMatcher.notificationTypeIs;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.component.location.Location;
import org.mule.tooling.client.api.datasense.DataSenseComponentInfo;
import org.mule.tooling.client.api.datasense.DataSenseRequest;
import org.mule.tooling.client.test.RuntimeType;

import com.google.common.collect.ImmutableList;
import com.google.common.io.Files;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Optional;

import org.apache.commons.lang3.tuple.ImmutablePair;
import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class MetadataCacheDisposeTestCase extends AbstractMetadataCacheTestCase {

  @ClassRule
  public static TemporaryFolder testTempFolder = new TemporaryFolder();

  private static final String CACHE_STATS_FILE_PROPERTY_NAME = "cacheStatusFilePath";

  private static File cacheStatsFile;
  private String previousCacheStatsFilePropertyValue;

  public MetadataCacheDisposeTestCase(RuntimeType runtimeType) {
    super(runtimeType);
  }

  @BeforeClass
  public static void createCacheStatsFile() throws IOException {
    cacheStatsFile = testTempFolder.newFile();
  }

  @Before
  public void cleanUpCacheStatsFile() throws IOException {
    previousCacheStatsFilePropertyValue = setProperty(CACHE_STATS_FILE_PROPERTY_NAME, cacheStatsFile.getAbsolutePath());
    open(cacheStatsFile.toPath(), WRITE).truncate(0).close();
  }

  @Before
  public void cleanDataObjectStoreData() throws IOException {
    cleanDirectory(new File(new File(new File(muleDir, ".mule"), "tooling"), "objectstore"));
  }

  @After
  public void resetSystemProperty() {
    if (previousCacheStatsFilePropertyValue == null) {
      clearProperty(CACHE_STATS_FILE_PROPERTY_NAME);
    } else {
      setProperty(CACHE_STATS_FILE_PROPERTY_NAME, previousCacheStatsFilePropertyValue);
    }
  }

  @Override
  protected List<ImmutablePair<String, String>> getStartupSystemProperties() {
    return ImmutableList.<ImmutablePair<String, String>>builder()
        .add(of("cacheStatusFilePath", cacheStatsFile.getAbsolutePath()))
        .build();
  }

  @Test
  public void disposeCacheAtConfigLevel() throws Exception {
    doTest(builder().globalName("config").build());
  }

  @Test
  public void disposeCacheAtComponentModelType() throws Exception {
    doTest(builder().globalName("fetchMetadata").addProcessorsPart().addIndexPart(0).build());
  }

  @Test
  public void reuseCacheBetweenToolingArtifacts() throws Exception {
    // Fist do the call to fetchMetadata for both operations
    doWithToolingArtifact(remoteToolingRuntimeClient(), getApplicationUrlContent("applications/dispose-metadata-cache"),
                          toolingArtifact -> {
                            DataSenseRequest dataSenseRequestAccount = new DataSenseRequest();
                            Location accountLocation =
                                builder().globalName("fetchMetadata").addProcessorsPart().addIndexPart(0).build();
                            dataSenseRequestAccount.setLocation(accountLocation);

                            assertRuntimeMetadataCache(toolingArtifact, dataSenseRequestAccount, "get:outputType_Account",
                                                       "miss:outputType_Account", "put:outputType_Account");
                          });
    // Clean up stats file
    cleanUpCacheStatsFile();
    // Do another call to fetchMetadata
    doWithToolingArtifact(remoteToolingRuntimeClient(), getApplicationUrlContent("applications/dispose-metadata-cache"),
                          toolingArtifact -> {
                            DataSenseRequest dataSenseRequestAccount = new DataSenseRequest();
                            Location accountLocation =
                                builder().globalName("fetchMetadata").addProcessorsPart().addIndexPart(0).build();
                            dataSenseRequestAccount.setLocation(accountLocation);

                            assertRuntimeMetadataCache(toolingArtifact, dataSenseRequestAccount, "get:outputType_Account",
                                                       "hit:outputType_Account");

                          });
  }

  private void doTest(Location componentLocationToBeDisposed) throws Exception {
    doWithToolingArtifact(remoteToolingRuntimeClient(), getApplicationUrlContent("applications/dispose-metadata-cache"),
                          toolingArtifact -> {
                            DataSenseRequest dataSenseRequestAccount = new DataSenseRequest();
                            Location accountLocation =
                                builder().globalName("fetchMetadata").addProcessorsPart().addIndexPart(0).build();
                            dataSenseRequestAccount.setLocation(accountLocation);

                            assertRuntimeMetadataCache(toolingArtifact, dataSenseRequestAccount, "get:outputType_Account",
                                                       "miss:outputType_Account", "put:outputType_Account");
                            assertRuntimeMetadataCache(toolingArtifact, dataSenseRequestAccount, "get:outputType_Account",
                                                       "hit:outputType_Account");

                            // Just another DataSense request that will add Deposit OutputType to the cache
                            DataSenseRequest dataSenseRequestDeposit = new DataSenseRequest();
                            Location depositLocation =
                                builder().globalName("fetchMetadata").addProcessorsPart().addIndexPart(1).build();
                            dataSenseRequestDeposit.setLocation(depositLocation);
                            assertRuntimeMetadataCache(toolingArtifact, dataSenseRequestDeposit, "get:outputType_Deposit",
                                                       "miss:outputType_Deposit", "put:outputType_Deposit");

                            // Dispose MetadataCache
                            toolingArtifact.metadataService().disposeMetadataCache().get().perform(componentLocationToBeDisposed);

                            // After disposed cache, do requests to DataSense
                            assertRuntimeMetadataCache(toolingArtifact, dataSenseRequestAccount, "get:outputType_Account",
                                                       "miss:outputType_Account", "put:outputType_Account");

                            assertRuntimeMetadataCache(toolingArtifact, dataSenseRequestDeposit, "get:outputType_Deposit",
                                                       "miss:outputType_Deposit", "put:outputType_Deposit");
                          });
  }

  private void assertRuntimeMetadataCache(ToolingArtifact toolingArtifact, DataSenseRequest dataSenseRequest,
                                          String... cacheStats)
      throws IOException {
    Optional<DataSenseComponentInfo> dataSenseComponentInfoOptional =
        toolingArtifact.dataSenseService().resolveComponentDataSense(dataSenseRequest);
    assertThat(dataSenseComponentInfoOptional, not(empty()));

    DataSenseComponentInfo dataSenseComponentInfo = dataSenseComponentInfoOptional.get();
    assertThat(dataSenseComponentInfo.getDataSenseNotifications(),
               not(hasItem(notificationTypeIs(errorDataSenseNotificationType("ERROR")))));

    assertThat(Files.readLines(cacheStatsFile, defaultCharset()), contains(cacheStats));
    cleanUpCacheStatsFile();
  }

}
