/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.tests.compatibility;

import static java.net.InetAddress.getLocalHost;
import static java.util.Collections.emptyList;
import static org.mule.maven.client.api.model.RemoteRepository.newRemoteRepositoryBuilder;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.internal.DefaultLocalRepositorySupplierFactory;
import org.mule.maven.client.internal.DefaultSettingsSupplierFactory;
import org.mule.maven.client.internal.MavenEnvironmentVariables;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrap;
import org.mule.tooling.client.test.utils.MuleStandaloneConfiguration;
import org.mule.tooling.client.test.utils.MuleStandaloneController;
import org.mule.tooling.client.tests.integration.category.NeedsJava8Test;
import org.mule.tooling.client.tests.junit.AfterTests;
import org.mule.tooling.client.tests.junit.BeforeTests;
import org.mule.tooling.client.tests.junit.ToolingRuntimeClientBootstrapProvider;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import org.junit.Rule;
import org.junit.experimental.categories.Category;

/**
 * Base class for Tooling test cases.
 */
@Category(NeedsJava8Test.class)
public abstract class AbstractToolingTestCase {

  private static final int DEFAULT_START_TIMEOUT = 50000;
  private static final int DEFAULT_START_POLL_INTERVAL = 500;
  private static final int DEFAULT_START_POLL_DELAY = 300;
  private static final int DEFAULT_CONTROLLER_OPERATION_TIMEOUT = 15000;

  private static final String REST_AGENT_TRANSPORT_PORT_SYS_PROP = "-M-Drest.agent.transport.port=";

  private ToolingRuntimeClientBootstrapProvider toolingRuntimeClientBootstrapProvider;
  private MuleStandaloneController muleStandaloneController;
  private File localMavenRepository;
  @Rule
  public DynamicPort agentPort = new DynamicPort("agentPort");

  public AbstractToolingTestCase(MavenConfiguration.MavenConfigurationBuilder mavenConfigurationBuilder,
                                 ToolingRuntimeClientBootstrapProvider toolingRuntimeClientBootstrapProvider, File muleHome) {
    try {
      localMavenRepository = new DefaultLocalRepositorySupplierFactory().environmentMavenRepositorySupplier().get();
      mavenConfigurationBuilder.localMavenRepositoryLocation(localMavenRepository);
      mavenConfigurationBuilder.remoteRepository(newRemoteRepositoryBuilder()
          .id("mulesoft-public")
          .url(new URL(
                       "https://repository.mulesoft.org/nexus/content/repositories/public"))
          .build());
      mavenConfigurationBuilder.remoteRepository(newRemoteRepositoryBuilder()
          .id("mulesoft-private")
          .url(new URL(
                       "https://repository.mulesoft.org/nexus/content/repositories/private"))
          .build());
      mavenConfigurationBuilder.remoteRepository(newRemoteRepositoryBuilder()
          .id("mule-ee-releases")
          .url(new URL(
                       "https://repository.mulesoft.org/nexus/content/repositories/releases-ee"))
          .build());

      DefaultSettingsSupplierFactory settingsSupplierFactory =
          new DefaultSettingsSupplierFactory(new MavenEnvironmentVariables());
      settingsSupplierFactory.environmentSettingsSecuritySupplier()
          .ifPresent(mavenConfigurationBuilder::settingsSecurityLocation);
      settingsSupplierFactory.environmentUserSettingsSupplier().ifPresent(mavenConfigurationBuilder::userSettingsLocation);
      settingsSupplierFactory.environmentGlobalSettingsSupplier().ifPresent(mavenConfigurationBuilder::globalSettingsLocation);

      this.toolingRuntimeClientBootstrapProvider = toolingRuntimeClientBootstrapProvider;
      this.muleStandaloneController =
          new MuleStandaloneController(muleHome,
                                       new MuleStandaloneConfiguration(DEFAULT_START_TIMEOUT, DEFAULT_START_POLL_INTERVAL,
                                                                       DEFAULT_START_POLL_DELAY,
                                                                       DEFAULT_CONTROLLER_OPERATION_TIMEOUT));
    } catch (MalformedURLException e) {
      throw new IllegalStateException("Couldn't initialize the test", e);
    }
  }

  @BeforeTests
  public void startMule() {
    if (shouldStartMule()) {
      if (muleStandaloneController.isRunning()) {
        throw new IllegalStateException("There is already a mule instance running");
      }
      List<String> defaultStartupSystemProperties = new ArrayList<>();
      defaultStartupSystemProperties
          .add("-M-DmuleRuntimeConfig.maven.repositoryLocation=" + localMavenRepository.getAbsolutePath());
      defaultStartupSystemProperties.add(REST_AGENT_TRANSPORT_PORT_SYS_PROP + agentPort.getValue());
      defaultStartupSystemProperties.addAll(getStartupSystemProperties());
      muleStandaloneController.start(defaultStartupSystemProperties.toArray(new String[] {}), "HTTP");
    }
  }

  protected boolean shouldStartMule() {
    return true;
  }

  protected List<String> getStartupSystemProperties() {
    return emptyList();
  }

  @AfterTests
  public void shutdownMule() {
    if (muleStandaloneController.isRunning()) {
      muleStandaloneController.stop();
    }
  }

  protected URL lookUpApplication(String relativePath) {
    return this.getClass().getClassLoader().getResource(relativePath);
  }

  protected ToolingRuntimeClientBootstrap getToolingRuntimeBootstrap() {
    return toolingRuntimeClientBootstrapProvider.getBootstrap();
  }

  protected AgentConfiguration.Builder newDefaultAgentConfigurationBuilder() throws URISyntaxException, MalformedURLException {
    return AgentConfiguration.builder()
        .withDefaultConnectionTimeout(5000)
        .withDefaultReadTimeout(50000)
        .withMuleVersion(getToolingRuntimeBootstrap().getMuleVersion())
        .withToolingApiUrl(new URI(getAgentToolingUrl()).toURL());
  }

  private String getAgentUrl() {
    try {
      String localhost = getLocalHost().getHostAddress();
      return "http://" + localhost + ":" + agentPort.getNumber();
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  private String getAgentToolingUrl() {
    try {
      return getAgentUrl() + "/mule/tooling";
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

}
