/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.tests.compatibility;

import static java.util.Arrays.stream;
import static java.util.Collections.emptyMap;
import static org.hamcrest.CoreMatchers.allOf;
import static org.hamcrest.CoreMatchers.startsWith;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.junit.Assert.assertThat;
import static org.mule.tooling.client.api.component.location.Location.builder;
import static org.mule.tooling.client.api.component.location.Location.builderFromStringRepresentation;
import static org.mule.tooling.client.api.datasense.DataSenseNotificationType.errorDataSenseNotificationType;
import static org.mule.tooling.client.api.datasense.DataSenseNotificationType.infoDataSenseNotificationType;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.internal.utils.MetadataTypeWriter;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.component.location.Location;
import org.mule.tooling.client.api.datasense.DataSenseComponentInfo;
import org.mule.tooling.client.api.datasense.DataSenseInfo;
import org.mule.tooling.client.api.datasense.DataSenseNotification;
import org.mule.tooling.client.api.datasense.DataSenseNotificationType;
import org.mule.tooling.client.api.datasense.DataSenseRequest;
import org.mule.tooling.client.api.datasense.DataSenseResolutionScope;
import org.mule.tooling.client.api.extension.model.operation.OperationModel;
import org.mule.tooling.client.tests.integration.category.NeedMuleRuntimeTest;
import org.mule.tooling.client.tests.junit.ToolingJUnitTestRunner;
import org.mule.tooling.client.tests.junit.ToolingRuntimeClientBootstrapProvider;
import org.mule.tooling.client.tests.junit.ToolingRuntimeClientVersion;

import java.io.File;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.Statement;
import java.util.Optional;
import java.util.function.Predicate;

import io.qameta.allure.Description;
import org.h2.tools.DeleteDbFiles;
import org.hamcrest.TypeSafeMatcher;
import org.junit.After;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.runner.RunWith;

@Category(NeedMuleRuntimeTest.class)
@RunWith(ToolingJUnitTestRunner.class)
// ********* 4.2 ***********
@ToolingRuntimeClientVersion(muleHome = "runtimes/mule-enterprise-standalone-4.2.0-SNAPSHOT", toolingVersion = "4.2.0-SNAPSHOT")
@ToolingRuntimeClientVersion(muleHome = "runtimes/mule-enterprise-standalone-4.2.0-SNAPSHOT", toolingVersion = "4.1.6-SNAPSHOT")
// ********* 4.1 ***********
@ToolingRuntimeClientVersion(muleHome = "runtimes/mule-enterprise-standalone-4.1.6-SNAPSHOT", toolingVersion = "4.1.6-SNAPSHOT")
// ********* 4.0 ***********
@ToolingRuntimeClientVersion(muleHome = "runtimes/mule-enterprise-standalone-4.0.0", toolingVersion = "4.0.3")
public class DataSenseCompatibilityTestCase extends AbstractToolingTestCase {

  protected DataSenseRequest createDataSenseRequest(Location location) {

    return createDataSenseRequest(location, null);
  }

  private DataSenseRequest createDataSenseRequest(Location location,
                                                  DataSenseResolutionScope dataSenseResolutionScope) {
    DataSenseRequest request = new DataSenseRequest();
    if (location != null) {
      request.setLocation(location);
    } else if (dataSenseResolutionScope != null) {
      request.setDataSenseResolutionScope(dataSenseResolutionScope);
    } else {
      throw new IllegalArgumentException("Neither location nor DataSense resolution scope was specified.");
    }
    return request;
  }

  public DataSenseCompatibilityTestCase(MavenConfiguration.MavenConfigurationBuilder mavenConfigurationBuilder,
                                        ToolingRuntimeClientBootstrapProvider toolingRuntimeClientBootstrapProvider,
                                        File muleHome) {
    super(mavenConfigurationBuilder, toolingRuntimeClientBootstrapProvider, muleHome);
  }

  private static class DataSenseNotificationMatcher extends TypeSafeMatcher<DataSenseNotification> {

    private final Predicate<DataSenseNotification> predicate;
    private final String name;

    DataSenseNotificationMatcher(Predicate<DataSenseNotification> predicate, String name) {
      super(DataSenseNotification.class);
      this.predicate = predicate;
      this.name = name;
    }

    @Override
    protected boolean matchesSafely(DataSenseNotification object) {
      return predicate.test(object);
    }

    @Override
    protected void describeMismatchSafely(DataSenseNotification item, org.hamcrest.Description mismatchDescription) {
      super.describeMismatchSafely(item, mismatchDescription);
    }

    @Override
    public void describeTo(org.hamcrest.Description description) {
      description.appendText(name);
    }
  }

  private static DataSenseNotificationMatcher notification(Predicate<DataSenseNotification> predicate, String name) {
    return new DataSenseNotificationMatcher(predicate, name);
  }

  private static DataSenseNotificationMatcher notificationTypeIs(DataSenseNotificationType dataSenseNotificationType) {
    return notification(n -> dataSenseNotificationType.equals(n.getNotificationType()), "datasense notification");
  }

  private static DataSenseNotificationMatcher notificationMessageContains(String text) {
    return notification(n -> n.getMessage().getMessage().contains(text), "datasense notification message contains");
  }

  private static DataSenseNotificationMatcher notificationMessageOrContains(String... text) {
    return notification(n -> stream(text).filter(value -> n.getMessage().getMessage().contains(value)).findFirst()
        .map(notification -> true).orElse(false), "datasense notification message contains");
  }

  private static final String APP_LOCATION = "applications/datasense-dynamic";
  private static final String DB_APP_LOCATION = "applications/datasense-db-array-type";
  private static final String SIMPLE_DB_APP_LOCATION = "applications/db";
  private static final String BAD_CONFIG_REF_APP_LOCATION = "applications/bad-config-ref";
  public static final String DYNAMIC_DATA_SENSE_FLOW = "dynamicDataSenseFlow";
  public static final String DYNAMIC_DATA_SENSE_FLOW_SOURCE = DYNAMIC_DATA_SENSE_FLOW + "/source";

  private static final String DB_DRIVER = "org.h2.Driver";
  private static final String DB_FOLDER = "./target/tmp/";
  private static final String DB_NAME = "datasenseDB";
  private static final String DB_ENDPOINT = "jdbc:h2:" + DB_FOLDER + DB_NAME;

  private void setupDBServer() throws Exception {
    Class.forName(DB_DRIVER);
    Connection dbConnection = DriverManager.getConnection(DB_ENDPOINT, "",
                                                          "");
    dbConnection.setAutoCommit(false);
    Statement stmt = dbConnection.createStatement();
    stmt.execute("CREATE TABLE TEST(ID int primary key, NAME varchar(255), DESCRIPTION varchar(255))");
    stmt.execute("INSERT INTO TEST(ID, NAME, DESCRIPTION) VALUES(1, 'Hello', 'Good bye')");
    stmt.execute("INSERT INTO TEST(ID, NAME, DESCRIPTION) VALUES(2, 'World', 'Good bye')");
    stmt.close();
    dbConnection.commit();
    dbConnection.close();
  }

  @After
  public void deleteDB() {
    DeleteDbFiles.execute(DB_FOLDER, DB_NAME, true);
  }

  @Test
  @Description("Checks DataSense resolution using DB config")
  public void resolveDataSenseDbConfig() throws Exception {
    setupDBServer();
    try {
      final ToolingRuntimeClient toolingRuntimeClient =
          getToolingRuntimeBootstrap().getToolingRuntimeClientBuilderFactory().create()
              .withRemoteAgentConfiguration(newDefaultAgentConfigurationBuilder().build())
              .build();
      final ToolingArtifact toolingArtifact =
          toolingRuntimeClient.newToolingArtifact(lookUpApplication(DB_APP_LOCATION), emptyMap());

      final DataSenseRequest dataSenseRequest =
          createDataSenseRequest(Location.builder().globalName("flow").addProcessorsPart().addIndexPart(0).build());
      Optional<DataSenseInfo> result = toolingArtifact.dataSenseService().resolveDataSense(dataSenseRequest);
      assertThat(result.isPresent(), is(true));
      result.ifPresent(dataSenseInfo -> {
        final MetadataTypeWriter metadataTypeWriter = new MetadataTypeWriter();
        assertThat(dataSenseInfo.getOutput().isPresent(), is(true));
        final MetadataType outputType =
            dataSenseInfo.getOutput()
                .orElseThrow(() -> new AssertionError("Expected output type not present"));
        assertThat(metadataTypeWriter.toString(outputType), is("%type _:Java = {\n"
            + "  \"message\" : @typeId(\"value\" : \"org.mule.runtime.api.message.Message\") {\n"
            + "    \"payload\" : @classInformation(\"classname\" : \"java.util.Iterator\", \"hasDefaultConstructor\" : false, \"isInterface\" : true, \"isInstantiable\" : false, \"isAbstract\" : true, \"isFinal\" : false, \"implementedInterfaces\" : [], \"parent\" : \"\", \"genericTypes\" : [], \"isMap\" : false) [{\n"
            + "        \"ID\"? : Number\n"
            + "      }], \n"
            + "    \"attributes\" : Void\n"
            + "  }, \n"
            + "  \"variables\" : {\n"
            + "\n"
            + "  }\n"
            + "}"));
      });
    } finally {
      deleteDB();
    }
  }

  @Test
  @Description("Checks that resolving dynamic DataSense requires a configuration for the remote service")
  public void dataSenseRequiresToolingClientWithRuntimeConfiguration() throws Exception {
    final ToolingRuntimeClient toolingRuntimeClient =
        getToolingRuntimeBootstrap().getToolingRuntimeClientBuilderFactory().create().build();
    final ToolingArtifact toolingArtifact = toolingRuntimeClient.newToolingArtifact(lookUpApplication(APP_LOCATION), emptyMap());
    final Location location = builderFromStringRepresentation(DYNAMIC_DATA_SENSE_FLOW_SOURCE).build();
    final DataSenseRequest dataSenseRequest = createDataSenseRequest(location);
    assertDataSenseRequestFailure(toolingArtifact, location, dataSenseRequest);
  }

  private void assertDataSenseRequestFailure(ToolingArtifact toolingArtifact, Location location,
                                             DataSenseRequest dataSenseRequest) {
    final Optional<DataSenseInfo> dataSenseInfoOptional = toolingArtifact.dataSenseService().resolveDataSense(dataSenseRequest);
    assertThat(dataSenseInfoOptional.isPresent(), is(true));
    final DataSenseInfo dataSenseInfo = dataSenseInfoOptional.get();
    assertThat(dataSenseInfo.getDataSenseNotifications(), hasSize(2));
    assertThat(dataSenseInfo.getDataSenseNotifications(),
               containsInAnyOrder(notificationTypeIs(errorDataSenseNotificationType("ERROR")),
                                  notificationTypeIs(infoDataSenseNotificationType("INFO"))));
  }

  @Test
  @Description("Checks DataSense resolution using HTTP")
  public void resolveDataSense() throws Exception {
    final ToolingRuntimeClient toolingRuntimeClient =
        getToolingRuntimeBootstrap().getToolingRuntimeClientBuilderFactory().create()
            .withRemoteAgentConfiguration(newDefaultAgentConfigurationBuilder().build())
            .build();
    final ToolingArtifact toolingArtifact = toolingRuntimeClient.newToolingArtifact(lookUpApplication(APP_LOCATION), emptyMap());

    final DataSenseRequest dataSenseRequest =
        createDataSenseRequest(builder().globalName(DYNAMIC_DATA_SENSE_FLOW).addProcessorsPart()
            .addIndexPart(0).build());
    Optional<DataSenseInfo> result = toolingArtifact.dataSenseService().resolveDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));
    result.ifPresent(dataSenseInfo -> {
      final MetadataTypeWriter metadataTypeWriter = new MetadataTypeWriter();
      assertThat(dataSenseInfo.getOutput().isPresent(), is(true));
      final MetadataType outputType =
          dataSenseInfo.getOutput()
              .orElseThrow(() -> new AssertionError("Expected output type not present"));
      assertThat(metadataTypeWriter.toString(outputType),
                 is("%type _:Java = {\n"
                     + "  \"message\" : @typeId(\"value\" : \"org.mule.runtime.api.message.Message\") {\n"
                     + "    \"payload\" : Binary, \n"
                     + "    \"attributes\" : @typeId(\"value\" : \"org.mule.extension.http.api.HttpResponseAttributes\") @classInformation(\"classname\" : \"org.mule.extension.http.api.HttpResponseAttributes\", \"hasDefaultConstructor\" : false, \"isInterface\" : false, \"isInstantiable\" : false, \"isAbstract\" : false, \"isFinal\" : false, \"implementedInterfaces\" : [], \"parent\" : \"org.mule.extension.http.api.HttpAttributes\", \"genericTypes\" : [], \"isMap\" : false) @typeAlias(\"value\" : \"HttpResponseAttributes\") {\n"
                     + "      @visibility(\"accessibility\" : READ_ONLY) \"headers\"? : @classInformation(\"classname\" : \"org.mule.runtime.api.util.MultiMap\", \"hasDefaultConstructor\" : true, \"isInterface\" : false, \"isInstantiable\" : true, \"isAbstract\" : false, \"isFinal\" : false, \"implementedInterfaces\" : [java.util.Map, java.io.Serializable], \"parent\" : \"\", \"genericTypes\" : [java.lang.String, java.lang.String], \"isMap\" : true) {\n"
                     + "        * : String\n"
                     + "      }, \n"
                     + "      @visibility(\"accessibility\" : READ_ONLY) \"reasonPhrase\"? : String, \n"
                     + "      @visibility(\"accessibility\" : READ_ONLY) \"statusCode\"? : @classInformation(\"classname\" : \"int\", \"hasDefaultConstructor\" : false, \"isInterface\" : false, \"isInstantiable\" : false, \"isAbstract\" : true, \"isFinal\" : true, \"implementedInterfaces\" : [], \"parent\" : \"\", \"genericTypes\" : [], \"isMap\" : false) @int Number\n"
                     + "    }\n"
                     + "  }, \n"
                     + "  \"variables\" : {\n"
                     + "\n"
                     + "  }\n"
                     + "}"));
    });
    result = toolingArtifact.dataSenseService().resolveDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));
    final DataSenseInfo dataSenseInfo = result.get();
    assertThat(dataSenseInfo.getMessages(), hasSize(1));
    assertThat(dataSenseInfo.getMessages().get(0), startsWith("[INFO]"));
  }

  @Test
  @Description("Checks Component DataSense resolution using HTTP")
  public void resolveComponentDataSense() throws Exception {
    final ToolingRuntimeClient toolingRuntimeClient =
        getToolingRuntimeBootstrap().getToolingRuntimeClientBuilderFactory().create()
            .withRemoteAgentConfiguration(newDefaultAgentConfigurationBuilder().build())
            .build();
    final ToolingArtifact toolingArtifact = toolingRuntimeClient.newToolingArtifact(lookUpApplication(APP_LOCATION), emptyMap());
    final DataSenseRequest dataSenseRequest =
        createDataSenseRequest(builder().globalName(DYNAMIC_DATA_SENSE_FLOW).addProcessorsPart()
            .addIndexPart(0).build());
    Optional<DataSenseComponentInfo> result = toolingArtifact.dataSenseService().resolveComponentDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));
    result.ifPresent(dataSenseInfo -> {
      Optional<OperationModel> operationModel = dataSenseInfo.getOperationModel();
      assertThat(operationModel.isPresent(), is(true));
      OperationModel model = operationModel.get();
      assertThat(model.getName(), is("request"));
    });
  }

  @Test
  @Description("Checks DataSense resolution for HTTP listener config is not initialized as part of Application Model")
  public void resolveDataSenseShouldNotInitializeListenerConfiguration() throws Exception {
    final ToolingRuntimeClient toolingRuntimeClient =
        getToolingRuntimeBootstrap().getToolingRuntimeClientBuilderFactory().create()
            .withRemoteAgentConfiguration(newDefaultAgentConfigurationBuilder().build())
            .build();
    ToolingArtifact toolingArtifact = toolingRuntimeClient.newToolingArtifact(lookUpApplication(APP_LOCATION), emptyMap());
    DataSenseRequest dataSenseRequest =
        createDataSenseRequest(builder().globalName(DYNAMIC_DATA_SENSE_FLOW).addProcessorsPart()
            .addIndexPart(0).build());
    Optional<DataSenseComponentInfo> result = toolingArtifact.dataSenseService().resolveComponentDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));
    result.ifPresent(dataSenseInfo -> {
      Optional<OperationModel> operationModel = dataSenseInfo.getOperationModel();
      assertThat(operationModel.isPresent(), is(true));
      OperationModel model = operationModel.get();
      assertThat(model.getName(), is("request"));
    });

    toolingArtifact = toolingRuntimeClient.newToolingArtifact(lookUpApplication(APP_LOCATION), emptyMap());
    result = toolingArtifact.dataSenseService().resolveComponentDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));
    DataSenseComponentInfo dataSenseInfo = result.get();
    Optional<OperationModel> operationModel = dataSenseInfo.getOperationModel();
    assertThat(operationModel.isPresent(), is(true));
    OperationModel model = operationModel.get();
    assertThat(model.getName(), is("request"));
    assertThat(dataSenseInfo.getDataSenseNotifications(),
               containsInAnyOrder(allOf(notificationTypeIs(infoDataSenseNotificationType("INFO")),
                                        notificationMessageContains("Resolving datasense info"))));
  }

  @Test
  @Description("Checks DataSense resolution using HTTP requester referencing to a invalid config")
  public void resolveDataSenseOnHttpRequesterWithBadConfiguration() throws Exception {
    final ToolingRuntimeClient toolingRuntimeClient =
        getToolingRuntimeBootstrap().getToolingRuntimeClientBuilderFactory().create()
            .withRemoteAgentConfiguration(newDefaultAgentConfigurationBuilder().build())
            .build();
    final ToolingArtifact toolingArtifact =
        toolingRuntimeClient.newToolingArtifact(lookUpApplication(BAD_CONFIG_REF_APP_LOCATION), emptyMap());

    final DataSenseRequest dataSenseRequest =
        createDataSenseRequest(builder().globalName("dataFlow").addProcessorsPart()
            .addIndexPart(0).build());
    Optional<DataSenseInfo> result = toolingArtifact.dataSenseService().resolveDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));
    final DataSenseInfo dataSenseInfo = result.get();
    assertThat(dataSenseInfo.getDataSenseNotifications(), hasSize(2));
    assertThat(dataSenseInfo.getDataSenseNotifications(), containsInAnyOrder(
                                                                             allOf(notificationTypeIs(errorDataSenseNotificationType("ERROR")),
                                                                                   notificationMessageContains("'beanName' must not be empty")),
                                                                             allOf(notificationTypeIs(infoDataSenseNotificationType("INFO")),
                                                                                   notificationMessageContains("Resolving datasense info"))));
  }

  @Test
  public void failWhenResolvingMetadata() throws Exception {
    final ToolingRuntimeClient toolingRuntimeClient =
        getToolingRuntimeBootstrap().getToolingRuntimeClientBuilderFactory().create()
            .withRemoteAgentConfiguration(newDefaultAgentConfigurationBuilder().build())
            .build();
    final ToolingArtifact toolingArtifact =
        toolingRuntimeClient.newToolingArtifact(lookUpApplication(SIMPLE_DB_APP_LOCATION), emptyMap());

    final DataSenseRequest dataSenseRequest =
        createDataSenseRequest(builder().globalName("testingFlow").addProcessorsPart()
            .addIndexPart(1).build());
    Optional<DataSenseInfo> result = toolingArtifact.dataSenseService().resolveDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));
    final DataSenseInfo dataSenseInfo = result.get();
    assertThat(dataSenseInfo.getDataSenseNotifications(), hasSize(2));
    assertThat(dataSenseInfo.getDataSenseNotifications(), containsInAnyOrder(
                                                                             allOf(notificationTypeIs(errorDataSenseNotificationType("ERROR")),
                                                                                   notificationMessageOrContains("Dynamic metadata resolution skipped, missing parameters: sql.",
                                                                                                                 "MetadataKey resolved to null",
                                                                                                                 "Unable to resolve value for the parameter: sql.")),
                                                                             allOf(notificationTypeIs(infoDataSenseNotificationType("INFO")),
                                                                                   notificationMessageContains("Resolving datasense info"))));
  }

}
