/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.tests.integration.tooling.client;

import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.startsWith;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;
import static org.mule.tooling.client.tests.integration.tooling.client.DynamicDataSenseTestCase.DYNAMIC_DATA_SENSE_FLOW;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.internal.utils.MetadataTypeWriter;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.component.location.Location;
import org.mule.tooling.client.api.datasense.DataSenseInfo;
import org.mule.tooling.client.api.datasense.DataSenseNotification;
import org.mule.tooling.client.api.datasense.DataSenseRequest;
import org.mule.tooling.client.api.metadata.MetadataFailure;
import org.mule.tooling.client.test.RuntimeType;

import com.google.common.cache.CacheBuilderSpec;
import com.google.common.cache.CacheStats;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Before;
import org.junit.Test;

@Feature("Cache for DataSenseService")
@Story("Integration tests for DataSense resolution using cache")
public class CachedDynamicDataSenseTestCase extends BaseDynamicDataSenseTestCase {

  private static final String APP_LOCATION = "applications/datasense-cached";
  private static final String PROJECT_ID = "projectId";
  private static final String PROJECT_ID_VALUE = "1";
  public static final String DYNAMIC_DATA_SENSE_FLOW_PROCESSORS_1 = "dynamicDataSenseFlow/processors/1";
  public static final String DYNAMIC_DATA_SENSE_FLOW_PROCESSORS_2 = "dynamicDataSenseFlow/processors/2";

  private ToolingMetadataCacheProvider dataSenseMetadataCacheProvider;
  private Map<String, String> toolingArtifactProperties;

  public CachedDynamicDataSenseTestCase(RuntimeType runtimeType) {
    super(runtimeType);
  }

  @Before
  public void init() {
    dataSenseMetadataCacheProvider = new ToolingMetadataCacheProvider(
                                                                      CacheBuilderSpec
                                                                          .parse("expireAfterWrite=50s, maximumSize=100"));
    toolingArtifactProperties = new HashMap<>();
    toolingArtifactProperties.put(PROJECT_ID, PROJECT_ID_VALUE);
  }

  @Test
  @Description("Checks that resolving dynamic DataSense using cache will generate miss, load and hit")
  public void resolveDataSense() throws Exception {
    ToolingRuntimeClient toolingRuntimeClient = createToolingRuntimeBuilder().withMetadataCacheFactory(
                                                                                                       (toolingArtifactId,
                                                                                                        applicationContentUrl,
                                                                                                        toolingArtifactProperties) -> {
                                                                                                         assertThat(toolingArtifactProperties
                                                                                                             .get(PROJECT_ID),
                                                                                                                    is(PROJECT_ID_VALUE));
                                                                                                         return dataSenseMetadataCacheProvider;
                                                                                                       })
        .build();
    final ToolingArtifact toolingArtifact = newToolingArtifact(toolingRuntimeClient, APP_LOCATION, toolingArtifactProperties);

    final DataSenseRequest dataSenseRequest =
        createDataSenseRequest(Location.builder().globalName(DYNAMIC_DATA_SENSE_FLOW).addProcessorsPart()
            .addIndexPart(0).build());

    Optional<DataSenseInfo> result;

    result = toolingArtifact.dataSenseService().resolveDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));

    result = toolingArtifact.dataSenseService().resolveDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));

    result = toolingArtifact.dataSenseService().resolveDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));

    DataSenseInfo dataSenseInfo = result.get();

    assertThat(dataSenseInfo.getOutput().isPresent(), is(true));

    final MetadataType outputType =
        dataSenseInfo.getOutput()
            .orElseThrow(() -> new AssertionError("Expected attribute type not present"));

    final MetadataTypeWriter metadataTypeWriter = new MetadataTypeWriter();
    assertEquals("%type _:Java = {\n"
        + "  \"message\" : @typeId(\"value\" : \"org.mule.runtime.api.message.Message\") {\n"
        + "    \"payload\" : Binary, \n"
        + "    \"attributes\" : @typeId(\"value\" : \"org.mule.extension.http.api.HttpResponseAttributes\") @classInformation(\"classname\" : \"org.mule.extension.http.api.HttpResponseAttributes\", \"hasDefaultConstructor\" : false, \"isInterface\" : false, \"isInstantiable\" : false, \"isAbstract\" : false, \"isFinal\" : false, \"implementedInterfaces\" : [], \"parent\" : \"org.mule.extension.http.api.HttpAttributes\", \"genericTypes\" : [], \"isMap\" : false) @typeAlias(\"value\" : \"HttpResponseAttributes\") {\n"
        + "      \"statusCode\" : @classInformation(\"classname\" : \"int\", \"hasDefaultConstructor\" : false, \"isInterface\" : false, \"isInstantiable\" : false, \"isAbstract\" : true, \"isFinal\" : true, \"implementedInterfaces\" : [], \"parent\" : \"\", \"genericTypes\" : [], \"isMap\" : false) @int Number, \n"
        + "      \"reasonPhrase\" : String, \n"
        + "      \"headers\" : @classInformation(\"classname\" : \"org.mule.runtime.api.util.MultiMap\", \"hasDefaultConstructor\" : true, \"isInterface\" : false, \"isInstantiable\" : true, \"isAbstract\" : false, \"isFinal\" : false, \"implementedInterfaces\" : [java.util.Map, java.io.Serializable], \"parent\" : \"\", \"genericTypes\" : [java.lang.String, java.lang.String], \"isMap\" : true) {\n"
        + "        * : String\n"
        + "      }\n"
        + "    }\n"
        + "  }, \n"
        + "  \"variables\" : {\n"
        + "\n"
        + "  }\n"
        + "}",
                 metadataTypeWriter.toString(outputType));

    assertThat(dataSenseInfo.getMessages(), hasSize(3));
    assertThat(dataSenseInfo.getMessages().get(0), startsWith("[INFO]"));
    DataSenseNotification dataSenseNotification = dataSenseInfo.getDataSenseNotifications().get(1);
    assertThat(dataSenseNotification.getFailureCode().get().isUnknown(), is(true));
    assertThat(dataSenseNotification.getComponentLocation().get().getLocation(), is(DYNAMIC_DATA_SENSE_FLOW_PROCESSORS_1));

    // DS will not trigger dynamic metadata resolution if config-ref is not set or keyId parameters are missing
    dataSenseNotification = dataSenseInfo.getDataSenseNotifications().get(2);
    assertThat(dataSenseNotification.getNotificationType().isError(), is(true));
    assertThat(dataSenseNotification.getComponentLocation().get().getLocation(), is(DYNAMIC_DATA_SENSE_FLOW_PROCESSORS_2));
    assertThat(dataSenseNotification.getMessage().getMessage(),
               is("Dynamic metadata resolution skipped, missing parameters: sql."));

    final Map<String, List<MetadataFailure>> failingComponents = dataSenseMetadataCacheProvider.getFailingComponents();
    assertThat(failingComponents.containsKey(DYNAMIC_DATA_SENSE_FLOW_PROCESSORS_1), is(true));
    final List<MetadataFailure> metadataFailures = failingComponents.get(DYNAMIC_DATA_SENSE_FLOW_PROCESSORS_1);
    assertThat(metadataFailures.size(), is(1));
    assertThat(metadataFailures.get(0).getFailureCode().isUnknown(), is(true));

    final Map<String, CacheStats> stats = dataSenseMetadataCacheProvider.stats();
    final CacheStats cacheStats = stats.get(ToolingMetadataCacheProvider.OPERATION_CACHE);
    assertEquals(1, cacheStats.missCount());
    assertEquals(1, cacheStats.loadCount());
    assertEquals(2, cacheStats.hitCount());
  }
}
