/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.tests.integration.tooling.client;

import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.datasense.DataSenseElementInfo;
import org.mule.tooling.client.api.datasense.DataSenseInfo;
import org.mule.tooling.client.api.datasense.DataSenseRequest;
import org.mule.tooling.client.api.datasense.ConfigResolutionScope;
import org.mule.tooling.client.api.component.location.Location;
import org.mule.tooling.client.test.RuntimeType;

import java.util.Map;
import java.util.Optional;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Test;

@Feature("DataSenseService")
@Story("Integration tests for DataSense resolution on a specific Mule config using ToolingBootstrap and ToolingRuntimeClient")
public class ConfigResolutionDataSenseTestCase extends BaseDynamicDataSenseTestCase {

  private static final String APP_LOCATION = "applications/datasense-configresolution";
  private static final int WORKING_METADATA_TIMEOUT = 5000;
  private static final int FAILING_METADATA_TIMEOUT = 1;

  public ConfigResolutionDataSenseTestCase(RuntimeType runtimeType) {
    super(runtimeType);
  }

  private DataSenseInfo run(int timeout) throws Exception {
    final ToolingRuntimeClient toolingRuntimeClient = createToolingRuntimeClient();
    final ToolingArtifact toolingArtifact = newToolingArtifact(toolingRuntimeClient, APP_LOCATION);

    final ConfigResolutionScope dataSenseResolutionScope =
        new ConfigResolutionScope("aFolder/mule-flows.xml");
    final DataSenseRequest dataSenseRequest =
        createDataSenseRequest(dataSenseResolutionScope);
    if (timeout > 0) {
      dataSenseRequest.setMetadataTimeout(timeout);
    }
    Optional<DataSenseInfo> result = toolingArtifact.dataSenseService().resolveDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));
    return result.get();
  }

  @Test
  @Description("Checks resolution of DataSense static")
  public void resolveDataSense() throws Exception {
    DataSenseInfo dataSenseInfo = run(-1);
    final Map<Location, DataSenseElementInfo> locationDataSenseElementInfoMap = dataSenseInfo.getComponentInfoByComponentPath()
        .orElseThrow(() -> new AssertionError("No datasense component info present."));
    assertNotNull(locationDataSenseElementInfoMap.get(Location.builderFromStringRepresentation("flow2/source").build()));
    assertNull(locationDataSenseElementInfoMap.get(Location.builderFromStringRepresentation("flow1/source").build()));
  }

  private boolean runWithTimeout(int timeout) throws Exception {
    DataSenseInfo dataSenseInfo = run(timeout);

    Boolean hasError = dataSenseInfo.getDataSenseNotifications().stream()
        .reduce(Boolean.FALSE,
                (v,
                 notification) -> new Boolean(v || notification.getNotificationType().isError()),
                (v, notification) -> v);
    return !hasError.booleanValue();
  }

  @Test
  @Description("Checks resolution of DataSense static")
  public void resolveDataSenseWithSmallTimeout() throws Exception {
    assertFalse(runWithTimeout(FAILING_METADATA_TIMEOUT));
  }

  @Test
  @Description("Checks resolution of DataSense static")
  public void resolveDataSenseWithBigTimeout() throws Exception {
    assertTrue(runWithTimeout(WORKING_METADATA_TIMEOUT));
  }
}
