/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.tests.integration.tooling.client;

import static java.lang.String.format;
import static java.lang.String.valueOf;
import static java.net.InetAddress.getLocalHost;
import static java.util.Collections.emptyMap;
import static org.apache.commons.lang3.tuple.ImmutablePair.of;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;
import static org.junit.rules.ExpectedException.none;
import static org.mule.maven.client.api.MavenClientProvider.discoverProvider;
import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;

import com.google.common.collect.ImmutableMap;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.connectivity.ConnectionValidationResult;
import org.mule.tooling.client.api.connectivity.ConnectivityTestingObjectNotFoundException;
import org.mule.tooling.client.api.connectivity.ConnectivityTestingRequest;
import org.mule.tooling.client.api.connectivity.UnsupportedConnectivityTestingObjectException;
import org.mule.tooling.client.api.exception.MissingToolingConfigurationException;
import org.mule.tooling.client.api.exception.TimeoutException;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrap;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapConfiguration;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapFactory;
import org.mule.tooling.client.test.AbstractMuleRuntimeTestCase;
import org.mule.tooling.client.test.RuntimeType;
import org.mule.tooling.client.tests.integration.category.NeedMuleRuntimeTest;

import com.google.common.collect.ImmutableList;
import com.icegreen.greenmail.util.GreenMail;
import com.icegreen.greenmail.util.ServerSetup;

import java.io.File;
import java.net.URL;
import java.net.UnknownHostException;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.List;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.rules.ExpectedException;

@Feature("ConnectivityTestingService")
@Story("Integration tests for ConnectivityTesting that use the ToolingBootstrap and ToolingRuntimeClient")
@Category(NeedMuleRuntimeTest.class)
public class ConnectivityTestCase extends AbstractMuleRuntimeTestCase {

  private static final String APPLICATIONS_LOCATION = "applications/";
  private static final String DOMAINS_LOCATION = "domains/";
  private static final String EMAIL_APP_LOCATION = APPLICATIONS_LOCATION + "email";
  private static final String MULE_4_200_APP_VERSION = APPLICATIONS_LOCATION + "appMinMuleVersionTo4200";
  private static final String DB_APP_LOCATION = APPLICATIONS_LOCATION + "db";
  private static final String DB_APP_DRIVER_FROM_TRANSTIVE_DEP_LOCATION =
      APPLICATIONS_LOCATION + "db-with-driver-from-transitive-dep";
  private static final String DB_ADDITIONAL_DEP_APP_LOCATION = APPLICATIONS_LOCATION + "db-additional-dep";
  private static final String DB_PLUGIN_WITH_ADDITIONAL_DEP = APPLICATIONS_LOCATION + "db-plugin-with-additional-dep";
  private static final String HTTP_LISTENER_DUPLICATE_PORT_APP_LOCATION = APPLICATIONS_LOCATION + "http-listener-duplicated-port";
  private static final String HTTP_LISTENER_CONFIG = "httpListenerConfig";
  private static final String HTTP_BAD_PORT_APP = APPLICATIONS_LOCATION + "http-bad-port";
  private static final String FILE_APP_LOCATION = APPLICATIONS_LOCATION + "file";
  private static final String SC_FILE_APP_LOCATION = APPLICATIONS_LOCATION + "smart-connector-using-file";
  private static final String OTHER_HTTP_LISTENER_CONFIG = "otherHttpListenerConfig";
  private static final String EMAIL_CONFIG = "emailConfig";
  private static final String DB_CONFIG = "dbConfig";
  private static final String UNKNOWN_HOST = "UNKNOWN_HOST";
  private static final String EMAIL = "EMAIL";
  private static final String EMAIL_SERVER_PORT = "emailServerPort";

  @Rule
  public ExpectedException expectedException = none();
  @Rule
  public DynamicPort emailServerPort = new DynamicPort(EMAIL_SERVER_PORT);
  @Rule
  public DynamicPort httpServerPort = new DynamicPort("httpServerPort");

  private ToolingRuntimeClient toolingRuntimeClient;
  private ToolingRuntimeClientBootstrap bootstrap;
  private GreenMail greenMail;

  public ConnectivityTestCase(RuntimeType runtimeType) {
    super(runtimeType);
  }

  @Override
  protected List<ImmutablePair<String, String>> getStartupSystemProperties() {
    try {
      return ImmutableList.<ImmutablePair<String, String>>builder()
          .add(of("dbServerHost", getLocalHost().getHostAddress()))
          .add(of("httpServerPort", valueOf(httpServerPort.getNumber()))).build();
    } catch (UnknownHostException e) {
      throw new RuntimeException("Couldn't get localhost address", e);
    }
  }

  @Before
  public void setUpToolingRuntimeClient() throws Exception {
    bootstrap = ToolingRuntimeClientBootstrapFactory.newToolingRuntimeClientBootstrap(
                                                                                      ToolingRuntimeClientBootstrapConfiguration
                                                                                          .builder()
                                                                                          .muleVersion(getMuleVersion())
                                                                                          .toolingVersion(getToolingVersion())
                                                                                          .mavenConfiguration(defaultMavenConfiguration)
                                                                                          .log4jConfiguration(getTestLog4JConfigurationFile())
                                                                                          .workingFolder(temporaryFolder
                                                                                              .newFolder())
                                                                                          .build());
    toolingRuntimeClient = bootstrap.getToolingRuntimeClientBuilderFactory().create()
        .withMavenConfiguration(defaultMavenConfiguration)
        .withRemoteAgentConfiguration(defaultAgentConfiguration).build();
  }

  @After
  public void stopEmailServerIfStarted() throws Exception {
    if (greenMail != null) {
      greenMail.stop();
    }
  }

  @After
  public void disposeBootstrap() throws Exception {
    if (bootstrap != null) {
      bootstrap.dispose();
    }
  }

  @Test
  @Description("Should throw an error if application declares as minMuleVersion a version not valid for the one it was bootstrapped Tooling Client")
  public void connectivityTestingMinMuleVersionNotValid() {
    expectedException.expect(ToolingException.class);
    expectedException.expectMessage("requires a newest runtime version");
    toolingRuntimeClient.newToolingArtifact(toUrl(MULE_4_200_APP_VERSION), emptyMap());
  }

  @Test
  @Description("Checks that TimeoutException is thrown when resolving a connectivity testing request and timeout is reached for the request")
  public void connectivityTestingTimeout() {
    setUpEmailServer();
    doWithToolingArtifact(toolingRuntimeClient, toUrl(EMAIL_APP_LOCATION), toolingArtifact -> {
      ConnectivityTestingRequest request = new ConnectivityTestingRequest();
      request.setRequestTimeout(10);
      request.setComponentId(EMAIL_CONFIG);

      expectedException.expect(TimeoutException.class);
      toolingArtifact.connectivityTestingService().testConnection(request);
    });
  }

  @Test
  @Description("Checks that connectivity testing should success no matter if there are two http listeners using same port")
  public void connectionTestingTwoHttpListenerUsingSamePort() throws Exception {
    doWithToolingArtifact(toolingRuntimeClient, toUrl(HTTP_LISTENER_DUPLICATE_PORT_APP_LOCATION), toolingArtifact -> {
      ConnectivityTestingRequest request = new ConnectivityTestingRequest();
      request.setComponentId(HTTP_LISTENER_CONFIG);
      ConnectionValidationResult connectionValidationResult =
          toolingArtifact.connectivityTestingService().testConnection(request);

      assertThat(connectionValidationResult.isValid(), is(true));

      doWithFetchedToolingArtifact(toolingRuntimeClient, toolingArtifact.getId(), fetchedToolingArtifact -> {
        request.setComponentId(OTHER_HTTP_LISTENER_CONFIG);
        assertThat(toolingArtifact.connectivityTestingService().testConnection(request).isValid(), is(true));
      });
    });
  }

  @Test
  @Description("Checks that connectivity testing should success no matter if there are two http listeners using same port")
  public void connectionTestingWithBadPort() throws Exception {
    doWithToolingArtifact(toolingRuntimeClient, toUrl(HTTP_BAD_PORT_APP), toolingArtifact -> {
      ConnectivityTestingRequest request = new ConnectivityTestingRequest();
      request.setComponentId(HTTP_LISTENER_CONFIG);
      ConnectionValidationResult connectionValidationResult =
          toolingArtifact.connectivityTestingService().testConnection(request);

      assertThat(connectionValidationResult.isValid(), is(false));
      assertThat(connectionValidationResult.getMessage(), is("port out of range:99999"));
    });
  }

  @Test
  @Description("Checks that connectivity testing using tlsContext")
  public void connectionTestingHTTPWithTlsContext() throws Exception {
    doWithToolingArtifact(toolingRuntimeClient, toUrl(APPLICATIONS_LOCATION + "http-tls-context"), toolingArtifact -> {
      ConnectivityTestingRequest request = new ConnectivityTestingRequest();
      request.setComponentId(HTTP_LISTENER_CONFIG);
      ConnectionValidationResult connectionValidationResult =
          toolingArtifact.connectivityTestingService().testConnection(request);

      assertThat(connectionValidationResult.isValid(), is(true));
    });
  }

  @Test
  @Description("Checks that connectivity testing works on domains")
  public void connectionTestingDomainWithHttpListener() throws Exception {
    doWithToolingArtifact(toolingRuntimeClient, toUrl(DOMAINS_LOCATION + "http-listener-domain"), toolingArtifact -> {
      ConnectivityTestingRequest request = new ConnectivityTestingRequest();
      request.setComponentId(HTTP_LISTENER_CONFIG);
      ConnectionValidationResult connectionValidationResult =
          toolingArtifact.connectivityTestingService().testConnection(request);

      assertThat(connectionValidationResult.isValid(), is(true));
    });
  }

  @Test
  @Description("Checks that connectivity testing should fail when email server is down")
  public void connectionTestingWhileEmailServerIsDown() throws Exception {
    doWithToolingArtifact(toolingRuntimeClient, toUrl(EMAIL_APP_LOCATION),
                          ImmutableMap.of(EMAIL_SERVER_PORT, emailServerPort.getValue()), toolingArtifact -> {
                            ConnectivityTestingRequest request = new ConnectivityTestingRequest();
                            request.setComponentId(EMAIL_CONFIG);

                            final ConnectionValidationResult connectionValidationResult =
                                toolingArtifact.connectivityTestingService().testConnection(request);

                            assertThat(connectionValidationResult.getErrorType().getIdentifier(), equalTo(UNKNOWN_HOST));
                            assertThat(connectionValidationResult.getErrorType().getNamespace(), equalTo(EMAIL));
                            assertThat(connectionValidationResult.isValid(), is(false));
                            assertThat(connectionValidationResult.getException(), notNullValue());
                          });
  }

  @Test
  @Description("Checks connectivity testing using an extension that depends on a plugin")
  public void connectionTestingWithPluginDependingOnCommonPlugin() throws Exception {
    doWithToolingArtifact(toolingRuntimeClient, toUrl(FILE_APP_LOCATION), toolingArtifact -> {
      ConnectivityTestingRequest request = new ConnectivityTestingRequest();
      request.setComponentId("fileConfig");

      ConnectionValidationResult connectionValidationResult =
          toolingArtifact.connectivityTestingService().testConnection(request);

      assertThat(connectionValidationResult.isValid(), is(true));
    });
  }

  @Test
  @Description("Checks connectivity testing success scenario")
  public void connectionTesting() throws Exception {
    setUpEmailServer();
    doWithToolingArtifact(toolingRuntimeClient, toUrl(EMAIL_APP_LOCATION),
                          ImmutableMap.of(EMAIL_SERVER_PORT, emailServerPort.getValue()), toolingArtifact -> {
                            ConnectivityTestingRequest request = new ConnectivityTestingRequest();
                            request.setComponentId(EMAIL_CONFIG);

                            final ConnectionValidationResult connectionValidationResult =
                                toolingArtifact.connectivityTestingService().testConnection(request);
                            assertThat(connectionValidationResult.isValid(), is(true));
                          });
  }

  @Test
  @Description("Checks connectivity testing for config declared on domain")
  public void connectionTestingConfigOnDomainArtifact() throws Exception {
    setUpEmailServer();
    doWithToolingArtifact(toolingRuntimeClient, toUrl("domains/email-domain"),
                          ImmutableMap.of(EMAIL_SERVER_PORT, emailServerPort.getValue()), toolingArtifact -> {
                            ConnectivityTestingRequest request = new ConnectivityTestingRequest();
                            request.setComponentId(EMAIL_CONFIG);

                            final ConnectionValidationResult connectionValidationResult =
                                toolingArtifact.connectivityTestingService().testConnection(request);
                            assertThat(connectionValidationResult.isValid(), is(true));
                          });
  }

  @Test
  @Description("Checks connectivity testing for config declared on domain but reusing a Tooling Artifact for domain")
  public void connectionTestingConfigOnDomainArtifactUsingAlreadyCreatedToolingArtifact() throws Exception {
    setUpEmailServer();
    doWithToolingArtifact(toolingRuntimeClient, toUrl("domains/email-domain"),
                          ImmutableMap.of(EMAIL_SERVER_PORT, emailServerPort.getValue()), domainToolingArtifact -> {
                            ConnectivityTestingRequest domainRequest = new ConnectivityTestingRequest();
                            domainRequest.setComponentId(EMAIL_CONFIG);

                            assertThat(domainToolingArtifact.connectivityTestingService().testConnection(domainRequest).isValid(),
                                       is(true));
                            doWithToolingArtifact(toolingRuntimeClient, toUrl("applications/email-with-domain-exploded"),
                                                  ImmutableMap.of(EMAIL_SERVER_PORT, emailServerPort.getValue()),
                                                  domainToolingArtifact,
                                                  applicationToolingArtifact -> {
                                                    ConnectivityTestingRequest applicationRequest =
                                                        new ConnectivityTestingRequest();
                                                    applicationRequest.setComponentId("anotherEmailConfig");

                                                    assertThat(applicationToolingArtifact.connectivityTestingService()
                                                        .testConnection(applicationRequest).isValid(), is(true));
                                                  });
                          });
  }

  @Test
  @Description("Checks connectivity testing success scenario")
  public void connectionTestingDoesNotCreateLogFile() {
    setUpEmailServer();
    doWithToolingArtifact(toolingRuntimeClient, toUrl(EMAIL_APP_LOCATION),
                          ImmutableMap.of(EMAIL_SERVER_PORT, emailServerPort.getValue()), toolingArtifact -> {
                            ConnectivityTestingRequest request = new ConnectivityTestingRequest();
                            request.setComponentId(EMAIL_CONFIG);
                            toolingArtifact.connectivityTestingService().testConnection(request);
                          });

    File logs = new File(new File(muleDir), "/logs");
    Boolean noToolingLogs = Arrays.stream(logs.listFiles()).allMatch(file -> !file.getPath().contains("mule-app-tooling-"));

    assertThat(noToolingLogs, is(true));
  }

  @Test
  @Description("Checks connectivity testing using an smart connector that depends on a plugin with success")
  public void scUsingFileConnectionTestingSuccess() throws Exception {
    doWithToolingArtifact(toolingRuntimeClient, toUrl(SC_FILE_APP_LOCATION), toolingArtifact -> {
      ConnectivityTestingRequest request = new ConnectivityTestingRequest();
      request.setComponentId("fileConfigThatWorks");

      ConnectionValidationResult connectionValidationResult =
          toolingArtifact.connectivityTestingService().testConnection(request);
      assertThat(connectionValidationResult.isValid(), is(true));
    });
  }

  @Test
  @Description("Checks connectivity testing using an smart connector that depends on a plugin with failure")
  public void scUsingFileConnectionTestingFailure() throws Exception {
    doWithToolingArtifact(toolingRuntimeClient, toUrl(SC_FILE_APP_LOCATION), toolingArtifact -> {
      ConnectivityTestingRequest request = new ConnectivityTestingRequest();
      request.setComponentId("fileConfigThatDoesntWork");

      ConnectionValidationResult connectionValidationResult =
          toolingArtifact.connectivityTestingService().testConnection(request);
      assertThat(connectionValidationResult.isValid(), is(false));
      assertThat(connectionValidationResult.getMessage(),
                 containsString(Paths.get("PATH", "THAT", "DOES", "NOT", "EXIST").toString()));
      assertThat(connectionValidationResult.getErrorType().getNamespace(), is("FILE"));
      assertThat(connectionValidationResult.getErrorType().getIdentifier(), is("FILE_DOESNT_EXIST"));
    });
  }

  @Test
  @Description("Checks connectivity testing without setting the remote Tooling API and after error is obtained sets the configuration and executes the request")
  public void tryConnectionTestingWithClientWithoutConfiguration() throws Exception {
    bootstrap = ToolingRuntimeClientBootstrapFactory.newToolingRuntimeClientBootstrap(
                                                                                      ToolingRuntimeClientBootstrapConfiguration
                                                                                          .builder()
                                                                                          .muleVersion(getMuleVersion())
                                                                                          .toolingVersion(getToolingVersion())
                                                                                          .mavenConfiguration(newMavenConfigurationBuilder()
                                                                                              .localMavenRepositoryLocation(discoverProvider(this
                                                                                                  .getClass().getClassLoader())
                                                                                                      .getLocalRepositorySuppliers()
                                                                                                      .environmentMavenRepositorySupplier()
                                                                                                      .get())
                                                                                              .build())
                                                                                          .log4jConfiguration(getTestLog4JConfigurationFile())
                                                                                          .workingFolder(temporaryFolder
                                                                                              .newFolder())
                                                                                          .build());
    ToolingRuntimeClient toolingRuntimeClient = bootstrap.getToolingRuntimeClientBuilderFactory().create().build();

    setUpEmailServer();
    doWithToolingArtifact(toolingRuntimeClient, toUrl(EMAIL_APP_LOCATION),
                          ImmutableMap.of(EMAIL_SERVER_PORT, emailServerPort.getValue()), toolingArtifact -> {
                            ConnectivityTestingRequest request = new ConnectivityTestingRequest();
                            request.setComponentId(EMAIL_CONFIG);

                            try {
                              toolingArtifact.connectivityTestingService().testConnection(request);
                              fail("Should have thrown a MissingToolingConfigurationException");
                            } catch (MissingToolingConfigurationException e) {
                              ToolingRuntimeClient toolingRuntimeClientWithAgentConfiguration =
                                  bootstrap.getToolingRuntimeClientBuilderFactory().create()
                                      .withRemoteAgentConfiguration(defaultAgentConfiguration)
                                      .build();
                              doWithToolingArtifact(toolingRuntimeClientWithAgentConfiguration, toUrl(EMAIL_APP_LOCATION),
                                                    ImmutableMap.of(EMAIL_SERVER_PORT, emailServerPort.getValue()),
                                                    toolingArtifactWithAgent -> {
                                                      ConnectionValidationResult connectionValidationResult =
                                                          toolingArtifactWithAgent.connectivityTestingService()
                                                              .testConnection(request);
                                                      assertThat(connectionValidationResult.isValid(), is(true));
                                                    });
                            }
                          });
  }

  @Test
  @Description("Checks connectivity testing for a componentId that is not present in config")
  public void connectionTestingNoFoundComponentId() throws Exception {
    setUpEmailServer();
    doWithToolingArtifact(toolingRuntimeClient, toUrl(EMAIL_APP_LOCATION),
                          ImmutableMap.of(EMAIL_SERVER_PORT, emailServerPort.getValue()), toolingArtifact -> {
                            ConnectivityTestingRequest request = new ConnectivityTestingRequest();
                            request.setComponentId("emailConfigNonExistent");
                            expectedException.expect(ConnectivityTestingObjectNotFoundException.class);
                            toolingArtifact.connectivityTestingService().testConnection(request);
                          });
  }

  @Test
  @Description("Checks connectivity testing for an unsupported component")
  public void connectionTestingForUnsupportedComponent() throws Exception {
    setUpEmailServer();
    doWithToolingArtifact(toolingRuntimeClient, toUrl(EMAIL_APP_LOCATION),
                          ImmutableMap.of(EMAIL_SERVER_PORT, emailServerPort.getValue()), toolingArtifact -> {
                            ConnectivityTestingRequest request = new ConnectivityTestingRequest();
                            request.setComponentId("emailFlow");
                            expectedException.expect(UnsupportedConnectivityTestingObjectException.class);
                            toolingArtifact.connectivityTestingService().testConnection(request);
                          });
  }

  @Test
  @Description("Checks connectivity testing for an application that has a shared lib dependency")
  public void pluginSharedLibs() {
    testDbConnectivityTesting(DB_APP_LOCATION);
  }

  @Test
  @Description("Checks connectivity testing for an application that has shared lib dependency which has a transitive dependency to the JDBC driver")
  public void driverFromTransitiveDependencyInSharedLibrary() {
    testDbConnectivityTesting(DB_APP_DRIVER_FROM_TRANSTIVE_DEP_LOCATION);
  }

  private void testDbConnectivityTesting(String appLocation) {
    doWithToolingArtifact(toolingRuntimeClient, toUrl(appLocation), toolingArtifact -> {
      ConnectivityTestingRequest request = new ConnectivityTestingRequest();
      request.setComponentId(DB_CONFIG);

      ConnectionValidationResult connectionValidationResult =
          toolingArtifact.connectivityTestingService().testConnection(request);
      assertThat(connectionValidationResult.isValid(), is(true));
    });
  }

  @Test
  @Description("Checks connectivity testing for an application that has additional libraries defined for a plugin")
  public void applicationAdditionalLibraries() throws Exception {
    doWithToolingArtifact(toolingRuntimeClient, toUrl(DB_ADDITIONAL_DEP_APP_LOCATION), toolingArtifact -> {
      ConnectivityTestingRequest request = new ConnectivityTestingRequest();
      request.setComponentId(DB_CONFIG);

      ConnectionValidationResult connectionValidationResult =
          toolingArtifact.connectivityTestingService().testConnection(request);
      assertThat(connectionValidationResult.isValid(), is(true));
    });
  }

  @Test
  @Description("Checks connectivity testing for an application that has additional libraries defined in a plugin")
  public void pluginAdditionalLibraries() throws Exception {
    doWithToolingArtifact(toolingRuntimeClient, toUrl(DB_PLUGIN_WITH_ADDITIONAL_DEP), toolingArtifact -> {
      ConnectivityTestingRequest request = new ConnectivityTestingRequest();
      request.setComponentId(DB_CONFIG);

      ConnectionValidationResult connectionValidationResult =
          toolingArtifact.connectivityTestingService().testConnection(request);
      assertThat(connectionValidationResult.isValid(), is(true));
    });
  }

  @Test
  @Description("Checks connectivity testing using Derby to validate that Driver releases the connection")
  public void derbyIsNotShutdownWhenDispose() throws Exception {
    pluginSharedLibs();
    // Call two times to check the driver releases resources when using derby
    pluginSharedLibs();
  }

  private URL toUrl(String appFolder) {
    try {
      File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
      return new File(targetTestClassesFolder, appFolder).toURI().toURL();
    } catch (Exception e) {
      throw new IllegalStateException(format("Couldn't get URL for application %s", appFolder));
    }
  }

  private void setUpEmailServer() {
    ServerSetup serverSetup = new ServerSetup(emailServerPort.getNumber(), null, "pop3");
    greenMail = new GreenMail(serverSetup);
    greenMail.setUser("foo", "pwd");
    greenMail.start();
  }

}
