/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.tests.integration.tooling.client;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.arrayWithSize;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.emptyCollectionOf;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.not;
import static org.mule.maven.client.test.MavenTestHelper.createDefaultEnterpriseMavenConfiguration;
import static org.mule.maven.client.test.MavenTestUtils.getMavenProperty;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.MULE_DB_CONNECTOR_VERSION;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.MULE_OBJECTSTORE_CONNECTOR_VERSION;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.MULE_VALIDATION_MODULE_VERSION;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.MULE_WSC_CONNECTOR_VERSION;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.POM_FOLDER_FINDER;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getMuleVersion;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getToolingVersion;
import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;
import org.mule.metadata.java.api.JavaTypeLoader;
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.artifact.dsl.DslElementSyntax;
import org.mule.tooling.client.api.artifact.dsl.DslSyntaxResolver;
import org.mule.tooling.client.api.artifact.dsl.DslSyntaxResolverFactory;
import org.mule.tooling.client.api.artifact.dsl.request.DslSyntaxResolverFactoryRequest;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import org.mule.tooling.client.api.extension.ExtensionModelService;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrap;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapConfiguration;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapFactory;
import org.mule.tooling.client.internal.persistence.Reference;
import org.mule.tooling.client.tests.integration.category.DoesNotNeedMuleRuntimeTest;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.rules.TemporaryFolder;

@Category(DoesNotNeedMuleRuntimeTest.class)
@Feature("DslSyntaxResolverService")
@Story("Integration tests for DslSyntaxResolverService using ToolingBootstrap and ToolingRuntimeClient")
public class DslSyntaxResolverServiceTestCase {

  private ToolingRuntimeClientBootstrap bootstrap;
  private ToolingRuntimeClient toolingRuntimeClient;
  private Map<String, ExtensionModel> extensionModels;
  private DslSyntaxResolverFactory resolverFactory;
  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  @Before
  public void setUpToolingRuntimeClient() throws Exception {
    bootstrap = ToolingRuntimeClientBootstrapFactory.newToolingRuntimeClientBootstrap(
                                                                                      ToolingRuntimeClientBootstrapConfiguration
                                                                                          .builder()
                                                                                          .muleVersion(getMuleVersion())
                                                                                          .toolingVersion(getToolingVersion())
                                                                                          .mavenConfiguration(createDefaultEnterpriseMavenConfiguration())
                                                                                          .workingFolder(temporaryFolder
                                                                                              .newFolder())
                                                                                          .build());
    toolingRuntimeClient = bootstrap.getToolingRuntimeClientBuilderFactory().create().build();

    List<ArtifactDescriptor> pluginArtifactDescriptors = new ArrayList<>();
    pluginArtifactDescriptors.add(buildPluginArtifactFixedVersion("mule-http-connector", "1.3.2"));
    pluginArtifactDescriptors.add(buildPluginArtifact("mule-db-connector", MULE_DB_CONNECTOR_VERSION));
    pluginArtifactDescriptors.add(buildPluginArtifact("mule-wsc-connector", MULE_WSC_CONNECTOR_VERSION));
    pluginArtifactDescriptors.add(buildPluginArtifact("mule-objectstore-connector", MULE_OBJECTSTORE_CONNECTOR_VERSION));
    pluginArtifactDescriptors.add(buildPluginModuleArtifact("mule-validation-module", MULE_VALIDATION_MODULE_VERSION));

    ExtensionModelService extensionModelService = toolingRuntimeClient.extensionModelService();
    extensionModels =
        pluginArtifactDescriptors.stream().map(p -> extensionModelService.loadExtensionModel(p)
            .orElseThrow(() -> new RuntimeException("Failed to load extension model for plugin: " + p.getArtifactId())))
            .collect(toMap(ExtensionModel::getName, e -> e));

    extensionModels.putAll(extensionModelService.loadMuleExtensionModels().stream()
        .collect(toMap(ExtensionModel::getName, e -> e)));

    resolverFactory = toolingRuntimeClient.dslSyntaxResolverService()
        .getDslSyntaxResolverFactory(new DslSyntaxResolverFactoryRequest(pluginArtifactDescriptors));
  }

  private ArtifactDescriptor buildPluginArtifact(String artifactId, String pomPropertyWithVersion) {
    return buildPluginArtifact("org.mule.connectors", artifactId, pomPropertyWithVersion);
  }

  private ArtifactDescriptor buildPluginArtifactFixedVersion(String artifactId, String version) {
    return doBuildPluginArtifact("org.mule.connectors", artifactId, version);
  }

  private ArtifactDescriptor buildPluginModuleArtifact(String artifactId, String pomPropertyWithVersion) {
    return buildPluginArtifact("org.mule.modules", artifactId, pomPropertyWithVersion);
  }

  private ArtifactDescriptor buildPluginArtifact(String groupId, String artifactId, String pomPropertyWithVersion) {
    return doBuildPluginArtifact(groupId, artifactId, getMavenProperty(pomPropertyWithVersion, POM_FOLDER_FINDER));
  }

  private ArtifactDescriptor doBuildPluginArtifact(String groupId, String artifactId, String version) {
    return ArtifactDescriptor.newBuilder()
        .withGroupId(groupId)
        .withArtifactId(artifactId)
        .withVersion(version)
        .withExtension("jar")
        .withClassifier("mule-plugin")
        .build();
  }

  @After
  public void disposeBootstrap() throws Exception {
    if (bootstrap != null) {
      bootstrap.dispose();
    }
  }

  @Test
  @Description("Checks the resolution of DSLSyntax for configs")
  public void configDsl() throws Exception {
    DslSyntaxResolver httpDsl = resolverFactory.createDslResolver(extensionModels.get("HTTP"));
    DslSyntaxResolver dbDsl = resolverFactory.createDslResolver(extensionModels.get("Database"));
    DslSyntaxResolver wscDsl = resolverFactory.createDslResolver(extensionModels.get("Web Service Consumer"));

    // Validate each extension resolves it's own config
    DslElementSyntax httpListener = httpDsl.resolve(extensionModels.get("HTTP").getConfigurationModels().get(0));
    assertThat(httpListener.getElementName(), is(equalTo("listener-config")));
    assertThat(httpListener.getContainedElement("basePath").get().getAttributeName(), is(equalTo("basePath")));
    DslElementSyntax httpConnection = httpListener.getContainedElement("listener").get();
    assertThat(httpConnection.getElementName(), is(equalTo("listener-connection")));
    assertThat(httpConnection.getContainedElement("protocol").get().supportsAttributeDeclaration(), is(true));
    assertThat(httpConnection.getContainedElement("tlsContext").get().supportsChildDeclaration(), is(true));

    DslElementSyntax dbConfig = dbDsl.resolve(extensionModels.get("Database").getConfigurationModels().get(0));
    DslElementSyntax dbConnection = dbConfig.getContainedElement("derby").get();
    assertThat(dbConnection.getElementName(), is(equalTo("derby-connection")));
    assertThat(dbConnection.getContainedElement("database").get().getAttributeName(), is(equalTo("database")));
    assertThat(dbConnection.getContainedElement("connectionProperties").get().supportsChildDeclaration(), is(true));
    assertThat(dbConnection.getContainedElement("poolingProfile").get().supportsChildDeclaration(), is(true));

    DslElementSyntax wscConfig = wscDsl.resolve(extensionModels.get("Web Service Consumer").getConfigurationModels().get(0));
    assertThat(wscConfig.getElementName(), is(equalTo("config")));
    DslElementSyntax expirationPolicy = wscConfig.getContainedElement("expirationPolicy").get();
    assertThat(expirationPolicy.supportsChildDeclaration(), is(true));
    assertThat(expirationPolicy.getContainedElement("maxIdleTime").get().getAttributeName(), is(equalTo("maxIdleTime")));
    DslElementSyntax wscConnection = wscConfig.getContainedElement("connection").get();
    assertThat(wscConnection.getElementName(), is(equalTo("connection")));
    assertThat(wscConnection.getContainedElement("address").get().supportsAttributeDeclaration(), is(true));
    assertThat(wscConnection.getContainedElement("mtomEnabled").get().supportsChildDeclaration(), is(false));

    httpDsl.dispose();
    dbDsl.dispose();

    resolverFactory.dispose();
  }

  @Test
  @Description("Checks the resolution of DslSyntax for connection is the same no matter which resolution path is used")
  public void connectionDsl() throws Exception {
    ExtensionModel httpExtension = extensionModels.get("HTTP");
    DslSyntaxResolver httpDsl = resolverFactory.createDslResolver(httpExtension);

    DslElementSyntax httpListenerConfig = httpDsl.resolve(httpExtension.getConfigurationModels().get(0));

    DslElementSyntax connectionDsl = httpDsl.resolve(httpExtension.getConfigurationModels().get(0)
        .getConnectionProviders().get(0));

    assertThat(connectionDsl, is(equalTo(httpListenerConfig.getContainedElement("listener").get())));
  }

  @Test
  @Description("Checks the resolution of DslSyntax for operations")
  public void operationDsl() throws Exception {
    ExtensionModel dbExtension = extensionModels.get("Database");
    DslSyntaxResolver dbDslResolver = resolverFactory.createDslResolver(dbExtension);

    DslElementSyntax bulkInsertDsl =
        dbDslResolver.resolve(dbExtension.getConfigurationModels().get(0).getOperationModels().get(1));
    assertThat(bulkInsertDsl.getElementName(), is(equalTo("bulk-insert")));
    assertThat(bulkInsertDsl.getContainedElement("sql").get().supportsAttributeDeclaration(), is(false));
    assertThat(bulkInsertDsl.getContainedElement("sql").get().supportsChildDeclaration(), is(true));

    DslElementSyntax parameterTypes = bulkInsertDsl.getContainedElement("parameterTypes").get();
    assertThat(parameterTypes.supportsChildDeclaration(), is(true));
    assertThat(parameterTypes.getElementName(), is(equalTo("parameter-types")));
    assertThat(parameterTypes.getGenerics().size(), is(1));
  }

  @Test
  @Description("Checks the resolution of DslSyntax for Sources")
  public void sourceDsl() throws Exception {
    ExtensionModel httpExtension = extensionModels.get("HTTP");
    DslSyntaxResolver httpDslResolver = resolverFactory.createDslResolver(httpExtension);

    DslElementSyntax listenerDsl =
        httpDslResolver.resolve(httpExtension.getConfigurationModels().get(0).getSourceModels().get(0));
    assertThat(listenerDsl.getElementName(), is(equalTo("listener")));
    assertThat(listenerDsl.getContainedElement("path").get().supportsAttributeDeclaration(), is(true));
    assertThat(listenerDsl.getContainedElement("reconnectionStrategy").get().supportsChildDeclaration(), is(true));
    assertThat(listenerDsl.getContainedElement("Response").get()
        .getContainedElement("body").get().supportsChildDeclaration(), is(true));
  }

  @Test
  @Description("Checks the resolution of DslSyntax for Constructs")
  public void constructDsl() throws Exception {
    resolverFactory = toolingRuntimeClient.dslSyntaxResolverService()
        .getDslSyntaxResolverFactory(new DslSyntaxResolverFactoryRequest(emptyList()));

    ExtensionModel muleExtension = extensionModels.get("mule");
    DslSyntaxResolver muleDslResolver = resolverFactory.createDslResolver(muleExtension);

    DslElementSyntax choiceDsl = muleDslResolver.resolve(muleExtension.getConstructModels().get(1));
    assertThat(choiceDsl.getElementName(), is(equalTo("choice")));
    DslElementSyntax whenDsl = choiceDsl.getContainedElement("when").get();
    assertThat(whenDsl.supportsChildDeclaration(), is(true));
    assertThat(whenDsl.supportsAttributeDeclaration(), is(false));
    assertThat(whenDsl.getContainedElement("expression").get().getAttributeName(), is("expression"));
    DslElementSyntax otherwiseDsl = choiceDsl.getContainedElement("otherwise").get();
    assertThat(otherwiseDsl.supportsChildDeclaration(), is(true));
    assertThat(otherwiseDsl.supportsAttributeDeclaration(), is(false));
    assertThat(otherwiseDsl.getContainedElements(), is(emptyCollectionOf(DslElementSyntax.class)));
  }

  @Test
  @Description("Checks the resolution of DslSyntax for Types")
  public void typesDsl() throws Exception {
    DslSyntaxResolver dslResolver = resolverFactory.createDslResolver(extensionModels.get("Database"));
    MetadataType type = new JavaTypeLoader(getClass().getClassLoader()).load(SimplePojo.class);

    DslElementSyntax typeDsl = dslResolver.resolve(type).get();
    assertThat(typeDsl.getNamespace(), is(equalTo("http://www.mulesoft.org/schema/mule/db")));
    assertThat(typeDsl.getElementName(), is(equalTo("simple-pojo")));
    assertThat(typeDsl.supportsChildDeclaration(), is(true));
    assertThat(typeDsl.supportsTopLevelDeclaration(), is(false));
    assertThat(typeDsl.getContainedElements().toArray(), is(arrayWithSize(2)));
  }

  @Test
  @Description("Checks the resolution of DslSyntax for Types that support top level declaration")
  public void typesDslSupportingTopLevelDeclarations() throws Exception {
    final ExtensionModel extensionModel = extensionModels.get("Validation");
    DslSyntaxResolver dslResolver = resolverFactory.createDslResolver(extensionModel);
    Reference<MetadataType> metadataTypeReference = new Reference<>();
    MetadataTypeVisitor metadataTypeVisitor = new MetadataTypeVisitor() {

      @Override
      public void visitObject(ObjectType objectType) {
        objectType.getAnnotation(ClassInformationAnnotation.class).ifPresent(annotation -> {
          if (annotation.getClassname().equals("org.mule.extension.validation.api.I18NConfig")) {
            metadataTypeReference.set(objectType);
          }
        });
      }
    };

    extensionModel.getTypes().stream().forEach(type -> type.accept(metadataTypeVisitor));
    assertThat(metadataTypeReference.get(), not(nullValue()));

    DslElementSyntax typeDsl = dslResolver.resolve(metadataTypeReference.get()).get();
    assertThat(typeDsl.getNamespace(), is(equalTo("http://www.mulesoft.org/schema/mule/validation")));
    assertThat(typeDsl.getElementName(), is(equalTo("i18n")));
    assertThat(typeDsl.supportsChildDeclaration(), is(true));
    assertThat(typeDsl.supportsTopLevelDeclaration(), is(true));
    assertThat(typeDsl.getContainedElements().toArray(), is(arrayWithSize(2)));
  }

  @Test
  @Description("Validates the typeDslAnnotation is considered")
  public void dslChildDeclarationShouldMatchExtensionModelTypeDsl() {
    final ExtensionModel extensionModel = extensionModels.get("ObjectStore");
    DslSyntaxResolver dslResolver = resolverFactory.createDslResolver(extensionModel);
    MetadataType topLevelElement =
        extensionModel.getTypes().stream().filter(type -> type.getAnnotation(ClassInformationAnnotation.class).get()
            .getClassname().equals("org.mule.extension.objectstore.api.TopLevelObjectStore")).findFirst().get();
    Optional<DslElementSyntax> syntaxOptional = dslResolver.resolve(topLevelElement);
    assertThat(syntaxOptional.isPresent(), is(true));
    DslElementSyntax syntax = syntaxOptional.get();
    assertThat(syntax.supportsChildDeclaration(), is(false));
    assertThat(syntax.supportsTopLevelDeclaration(), is(true));
  }

  @Test
  @Description("Validates the typeDslAnnotation is considered")
  public void shouldProvideAllSubtypesInContext() {
    ObjectType objectStore = findType("mule", "org.mule.runtime.api.store.ObjectStore");
    ObjectType scheduler = findType("mule", "org.mule.runtime.core.api.source.scheduler.Scheduler");
    ObjectType wscTransport =
        findType("Web Service Consumer", "org.mule.extension.ws.api.transport.CustomTransportConfiguration");

    DslSyntaxResolver dslResolver = resolverFactory.createDslResolver(extensionModels.get("Validation"));
    assertThat(getIds(dslResolver.getSubTypes(objectStore)),
               containsInAnyOrder("org.mule.extension.objectstore.api.TopLevelObjectStore",
                                  "org.mule.extension.objectstore.api.PrivateObjectStore"));

    assertThat(getIds(dslResolver.getSubTypes(scheduler)),
               containsInAnyOrder("org.mule.runtime.core.api.source.scheduler.FixedFrequencyScheduler",
                                  "org.mule.runtime.core.api.source.scheduler.CronScheduler"));

    assertThat(getIds(dslResolver.getSubTypes(wscTransport)),
               containsInAnyOrder("org.mule.extension.ws.api.transport.CustomHttpTransportConfiguration",
                                  "org.mule.extension.ws.api.transport.DefaultHttpTransportConfiguration"));
  }

  private List<String> getIds(Set<ObjectType> subTypes) {
    return subTypes.stream()
        .map(t -> t.getAnnotation(TypeIdAnnotation.class).get().getValue())
        .collect(toList());
  }

  private ObjectType findType(String model, String typedId) {
    return extensionModels.get(model).getTypes().stream()
        .filter(t -> t.getAnnotation(TypeIdAnnotation.class).map(id -> id.getValue().equals(typedId)).orElse(false))
        .findFirst()
        .orElseThrow(() -> new IllegalStateException("Missing Type: " + typedId));
  }

  public static final class SimplePojo {

    private String string;
    private Integer integer;

    public String getString() {
      return string;
    }

    public void setString(String string) {
      this.string = string;
    }

    public Integer getInteger() {
      return integer;
    }

    public void setInteger(Integer integer) {
      this.integer = integer;
    }
  }

}
