/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.tests.integration.tooling.client;

import static java.lang.String.valueOf;
import static org.apache.commons.lang3.tuple.ImmutablePair.of;
import static org.hamcrest.CoreMatchers.allOf;
import static org.hamcrest.CoreMatchers.startsWith;
import static org.hamcrest.Matchers.arrayContaining;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.assertThat;
import static org.mule.tooling.client.api.component.location.Location.builder;
import static org.mule.tooling.client.api.component.location.Location.builderFromStringRepresentation;
import static org.mule.tooling.client.api.datasense.DataSenseNotificationType.errorDataSenseNotificationType;
import static org.mule.tooling.client.api.datasense.DataSenseNotificationType.infoDataSenseNotificationType;
import static org.mule.tooling.client.tests.integration.tooling.client.DataSenseNotificationMatcher.notificationMessageContains;
import static org.mule.tooling.client.tests.integration.tooling.client.DataSenseNotificationMatcher.notificationTypeIs;
import org.mule.metadata.api.annotation.EnumAnnotation;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.internal.utils.MetadataTypeWriter;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.component.location.Location;
import org.mule.tooling.client.api.datasense.ComponentResolutionScope;
import org.mule.tooling.client.api.datasense.DataSenseComponentInfo;
import org.mule.tooling.client.api.datasense.DataSenseInfo;
import org.mule.tooling.client.api.datasense.DataSenseRequest;
import org.mule.tooling.client.api.extension.model.operation.OperationModel;
import org.mule.tooling.client.api.extension.model.source.SourceModel;
import org.mule.tooling.client.test.RuntimeType;

import com.google.common.collect.ImmutableList;
import com.icegreen.greenmail.util.GreenMail;
import com.icegreen.greenmail.util.ServerSetup;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.Statement;
import java.util.List;
import java.util.Optional;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.h2.tools.DeleteDbFiles;
import org.junit.After;
import org.junit.Rule;
import org.junit.Test;

@Feature("DataSenseService")
@Story("Integration tests for dynamic DataSense resolution using ToolingBootstrap and ToolingRuntimeClient")
public class DynamicDataSenseTestCase extends BaseDynamicDataSenseTestCase {

  private static final String APP_LOCATION = "applications/datasense-dynamic";
  private static final String SLACK_CONNECTOR_APP_LOCATION = "applications/slack-connector";
  private static final String DATA_SENSE_SUB_FLOW_APP = "applications/datasense-sub-flow";
  private static final String CHOICE_DOMAIN_LOCATION = "domains/datasense-dynamic-choice-domain";
  private static final String CHOICE_APP_LOCATION = "applications/datasense-dynamic-choice";
  private static final String DB_APP_LOCATION = "applications/datasense-db-array-type";
  private static final String BAD_CONFIG_REF_APP_LOCATION = "applications/bad-config-ref";
  private static final String SOAPKIT_APP_LOCATION = "applications/soapkit-metadata";
  public static final String DYNAMIC_DATA_SENSE_FLOW = "dynamicDataSenseFlow";
  public static final String DYNAMIC_DATA_SENSE_FLOW_SOURCE = DYNAMIC_DATA_SENSE_FLOW + "/source";

  private static final String DB_DRIVER = "org.h2.Driver";
  private static final String DB_FOLDER = "./target/tmp/";
  private static final String DB_NAME = "datasenseDB";
  private static final String DB_ENDPOINT = "jdbc:h2:" + DB_FOLDER + DB_NAME;

  @Rule
  public DynamicPort emailServerPort = new DynamicPort("emailServerPort");

  private GreenMail greenMail;

  public DynamicDataSenseTestCase(RuntimeType runtimeType) {
    super(runtimeType);
  }

  private void setupDBServer() throws Exception {
    Class.forName(DB_DRIVER);
    Connection dbConnection = DriverManager.getConnection(DB_ENDPOINT, "",
                                                          "");
    dbConnection.setAutoCommit(false);
    Statement stmt = dbConnection.createStatement();
    stmt.execute("CREATE TABLE TEST(ID int primary key, NAME varchar(255), DESCRIPTION varchar(255))");
    stmt.execute("INSERT INTO TEST(ID, NAME, DESCRIPTION) VALUES(1, 'Hello', 'Good bye')");
    stmt.execute("INSERT INTO TEST(ID, NAME, DESCRIPTION) VALUES(2, 'World', 'Good bye')");
    stmt.close();
    dbConnection.commit();
    dbConnection.close();
  }

  @After
  public void deleteDB() {
    DeleteDbFiles.execute(DB_FOLDER, DB_NAME, true);
  }

  @Override
  protected boolean isStartMuleBeforeEachTest() {
    return false;
  }

  @Override
  protected List<ImmutablePair<String, String>> getStartupSystemProperties() {
    return ImmutableList.<ImmutablePair<String, String>>builder()
        .add(of("emailServerPort", valueOf(emailServerPort.getNumber())))
        .build();
  }

  private void setUpEmailServer() {
    ServerSetup serverSetup = new ServerSetup(emailServerPort.getNumber(), null, "imap");
    greenMail = new GreenMail(serverSetup);
    greenMail.setUser("foo", "pwd");
    greenMail.start();
  }

  @After
  public void stopEmailServer() {
    if (greenMail != null) {
      greenMail.stop();
    }
  }

  @Test
  @Description("Checks DataSense resolution using DB config")
  public void resolveDataSenseDbConfig() throws Exception {
    startMuleRuntime();
    setupDBServer();
    try {
      final ToolingRuntimeClient toolingRuntimeClient = createToolingRuntimeClient();
      final ToolingArtifact toolingArtifact = newToolingArtifact(toolingRuntimeClient, DB_APP_LOCATION);

      final DataSenseRequest dataSenseRequest =
          createDataSenseRequest(Location.builder().globalName("flow").addProcessorsPart().addIndexPart(0).build());
      Optional<DataSenseInfo> result = toolingArtifact.dataSenseService().resolveDataSense(dataSenseRequest);
      assertThat(result.isPresent(), is(true));
      result.ifPresent(dataSenseInfo -> {
        final MetadataTypeWriter metadataTypeWriter = new MetadataTypeWriter();
        assertThat(dataSenseInfo.getOutput().isPresent(), is(true));
        final MetadataType outputType =
            dataSenseInfo.getOutput()
                .orElseThrow(() -> new AssertionError("Expected output type not present"));
        assertThat(metadataTypeWriter.toString(outputType), is("%type _:Java = {\n"
            + "  \"message\" : @typeId(\"value\" : \"org.mule.runtime.api.message.Message\") {\n"
            + "    \"payload\" : @classInformation(\"classname\" : \"java.util.Iterator\", \"hasDefaultConstructor\" : false, \"isInterface\" : true, \"isInstantiable\" : false, \"isAbstract\" : true, \"isFinal\" : false, \"implementedInterfaces\" : [], \"parent\" : \"\", \"genericTypes\" : [], \"isMap\" : false) [{\n"
            + "        \"ID\"? : Number\n"
            + "      }], \n"
            + "    \"attributes\" : Void\n"
            + "  }, \n"
            + "  \"variables\" : {\n"
            + "\n"
            + "  }\n"
            + "}"));
      });
    } finally {
      deleteDB();
    }
  }

  @Test
  @Description("Checks that resolving dynamic DataSense requires a configuration for the remote service")
  public void dataSenseRequiresToolingClientWithRuntimeConfiguration() throws Exception {
    final ToolingRuntimeClient toolingRuntimeClient = getBootstrap().getToolingRuntimeClientBuilderFactory().create().build();
    final ToolingArtifact toolingArtifact = newToolingArtifact(toolingRuntimeClient, APP_LOCATION);
    final Location location = builderFromStringRepresentation(DYNAMIC_DATA_SENSE_FLOW_SOURCE).build();
    final DataSenseRequest dataSenseRequest = createDataSenseRequest(location);
    assertDataSenseRequestFailure(toolingArtifact, location, dataSenseRequest, "Tooling Client has to be configured");
  }

  @Test
  @Description("Checks that resolving dynamic DataSense requires a Mule Runtime up and running")
  public void dataSenseRequiresRuntimeUpAndRunning() throws Exception {
    ToolingRuntimeClient toolingRuntimeClient = createToolingRuntimeClient();
    final ToolingArtifact toolingArtifact = newToolingArtifact(toolingRuntimeClient, APP_LOCATION);
    final Location location = builderFromStringRepresentation(DYNAMIC_DATA_SENSE_FLOW_SOURCE).build();
    final DataSenseRequest dataSenseRequest = createDataSenseRequest(location);
    assertDataSenseRequestFailure(toolingArtifact, location, dataSenseRequest, "Mule Agent REST service is not available");
  }

  private void assertDataSenseRequestFailure(ToolingArtifact toolingArtifact, Location location,
                                             DataSenseRequest dataSenseRequest, String errorMessage) {
    final Optional<DataSenseInfo> dataSenseInfoOptional = toolingArtifact.dataSenseService().resolveDataSense(dataSenseRequest);
    assertThat(dataSenseInfoOptional.isPresent(), is(true));
    final DataSenseInfo dataSenseInfo = dataSenseInfoOptional.get();
    assertThat(dataSenseInfo.getDataSenseNotifications(), hasSize(2));
    assertThat(dataSenseInfo.getDataSenseNotifications(), containsInAnyOrder(
                                                                             allOf(notificationTypeIs(errorDataSenseNotificationType("ERROR")),
                                                                                   notificationMessageContains(errorMessage)),
                                                                             allOf(notificationTypeIs(infoDataSenseNotificationType("INFO")),
                                                                                   notificationMessageContains("Resolving datasense info"))));
  }

  @Test
  @Description("Checks DataSense resolution using HTTP")
  public void resolveDataSense() throws Exception {
    startMuleRuntime();
    final ToolingRuntimeClient toolingRuntimeClient = createToolingRuntimeClient();
    final ToolingArtifact toolingArtifact = newToolingArtifact(toolingRuntimeClient, APP_LOCATION);

    final DataSenseRequest dataSenseRequest =
        createDataSenseRequest(builder().globalName(DYNAMIC_DATA_SENSE_FLOW).addProcessorsPart()
            .addIndexPart(0).build());
    Optional<DataSenseInfo> result = toolingArtifact.dataSenseService().resolveDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));
    result.ifPresent(dataSenseInfo -> {
      final MetadataTypeWriter metadataTypeWriter = new MetadataTypeWriter();
      assertThat(dataSenseInfo.getOutput().isPresent(), is(true));
      // TODO https://www.mulesoft.org/jira/browse/MULE-11410
      final MetadataType outputType =
          dataSenseInfo.getOutput()
              .orElseThrow(() -> new AssertionError("Expected output type not present"));
      assertThat(metadataTypeWriter.toString(outputType),
                 is("%type _:Java = {\n"
                     + "  \"message\" : @typeId(\"value\" : \"org.mule.runtime.api.message.Message\") {\n"
                     + "    \"payload\" : Binary, \n"
                     + "    \"attributes\" : @typeId(\"value\" : \"org.mule.extension.http.api.HttpResponseAttributes\") @classInformation(\"classname\" : \"org.mule.extension.http.api.HttpResponseAttributes\", \"hasDefaultConstructor\" : false, \"isInterface\" : false, \"isInstantiable\" : false, \"isAbstract\" : false, \"isFinal\" : false, \"implementedInterfaces\" : [], \"parent\" : \"org.mule.extension.http.api.HttpAttributes\", \"genericTypes\" : [], \"isMap\" : false) @typeAlias(\"value\" : \"HttpResponseAttributes\") {\n"
                     + "      @visibility(\"accessibility\" : READ_ONLY) \"headers\"? : @classInformation(\"classname\" : \"org.mule.runtime.api.util.MultiMap\", \"hasDefaultConstructor\" : true, \"isInterface\" : false, \"isInstantiable\" : true, \"isAbstract\" : false, \"isFinal\" : false, \"implementedInterfaces\" : [java.util.Map, java.io.Serializable], \"parent\" : \"\", \"genericTypes\" : [java.lang.String, java.lang.String], \"isMap\" : true) {\n"
                     + "        * : String\n"
                     + "      }, \n"
                     + "      @visibility(\"accessibility\" : READ_ONLY) \"reasonPhrase\"? : String, \n"
                     + "      @visibility(\"accessibility\" : READ_ONLY) \"statusCode\"? : @classInformation(\"classname\" : \"int\", \"hasDefaultConstructor\" : false, \"isInterface\" : false, \"isInstantiable\" : false, \"isAbstract\" : true, \"isFinal\" : true, \"implementedInterfaces\" : [], \"parent\" : \"\", \"genericTypes\" : [], \"isMap\" : false) @int Number\n"
                     + "    }\n"
                     + "  }, \n"
                     + "  \"variables\" : {\n"
                     + "\n"
                     + "  }\n"
                     + "}"));
    });
    result = toolingArtifact.dataSenseService().resolveDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));
    final DataSenseInfo dataSenseInfo = result.get();
    // TODO https://www.mulesoft.org/jira/browse/MULE-11410
    assertThat(dataSenseInfo.getMessages(), hasSize(1));
    assertThat(dataSenseInfo.getMessages().get(0), startsWith("[INFO]"));
  }

  @Test
  public void resolveSoapKitMetadata() throws Exception {
    startMuleRuntime();
    final ToolingRuntimeClient toolingRuntimeClient = createToolingRuntimeClient();
    final ToolingArtifact toolingArtifact = newToolingArtifact(toolingRuntimeClient, SOAPKIT_APP_LOCATION);

    DataSenseRequest dataSenseRequest =
        createDataSenseRequest(builder().globalName("ListInventory:\\config").build());
    Optional<DataSenseInfo> result = toolingArtifact.dataSenseService().resolveDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));
    final Optional<MetadataType> output =
        result.get().getComponentInfoByComponentPath().get().get(
                                                                 builderFromStringRepresentation("ListInventory:\\config/processors/0")
                                                                     .build())
            .getOutput();
    final MetadataTypeWriter metadataTypeWriter = new MetadataTypeWriter();
    final MetadataType outputType = output.orElseThrow(() -> new AssertionError("Expected output type is not present"));
    assertThat(metadataTypeWriter.toString(outputType),
               is("%type _:Java = {\n  \"message\" : @typeId(\"value\" : \"org.mule.runtime.api.message.Message\") {\n    \"payload\" : @typeId(\"value\" : \"#root:{http://mulesoft.org/tshirt-service}TshirtFault\") {\n      \"{http://mulesoft.org/tshirt-service}TshirtFault\" : @typeId(\"value\" : \"{http://mulesoft.org/tshirt-service}#AnonType_TshirtFault\") {\n        \"errorMessage\" : String\n      }\n    }, \n    \"attributes\" : Void\n  }, \n  \"variables\" : {\n\n  }\n}"));
  }

  @Test
  @Description("Checks DataSense resolution using slack connector with a repeatable stream config with default values. That causes the default value to be loaded from the ComponentBuildingDefinition instead of the XSD. This test case prevents an incompatibility between the default value provided and the input expected by the type converter.")
  public void resolveDataSenseWithRepeatableStreamConfig() throws Exception {
    startMuleRuntime();
    final ToolingRuntimeClient toolingRuntimeClient = createToolingRuntimeClient();
    final ToolingArtifact toolingArtifact = newToolingArtifact(toolingRuntimeClient, SLACK_CONNECTOR_APP_LOCATION);

    DataSenseRequest dataSenseRequest =
        createDataSenseRequest(builder().globalName("slackFlow").addProcessorsPart()
            .addIndexPart(0).build());
    Optional<DataSenseInfo> result = toolingArtifact.dataSenseService().resolveDataSense(dataSenseRequest);
    assertNoErrors(result);
    dataSenseRequest =
        createDataSenseRequest(builder().globalName("slackFlow").addProcessorsPart()
            .addIndexPart(1).build());
    result = toolingArtifact.dataSenseService().resolveDataSense(dataSenseRequest);
    assertNoErrors(result);
  }

  private void assertNoErrors(Optional<DataSenseInfo> result) {
    assertThat(result.isPresent(), is(true));
    assertThat(result.get().getMessages(), hasSize(1));
    assertThat(result.get().getMessages().get(0), startsWith("[INFO]"));
  }

  @Test
  @Description("Checks DataSense resolution using choice component and a complex flow to validate MinimalApplicationGeneration logic")
  public void resolveDataSenseChoiceComponent() throws Exception {
    setUpEmailServer();
    startMuleRuntime();
    final ToolingRuntimeClient toolingRuntimeClient = createToolingRuntimeClient();
    final ToolingArtifact domainToolingArtifact = newToolingArtifact(toolingRuntimeClient, CHOICE_DOMAIN_LOCATION);
    final ToolingArtifact toolingArtifact = newToolingArtifact(toolingRuntimeClient, CHOICE_APP_LOCATION, domainToolingArtifact);

    DataSenseRequest dataSenseRequest =
        createDataSenseRequest(builder().globalName(DYNAMIC_DATA_SENSE_FLOW).addProcessorsPart()
            .addIndexPart(1).addPart("route").addIndexPart(0).addProcessorsPart().addIndexPart(0).build());
    Optional<DataSenseInfo> result = toolingArtifact.dataSenseService().resolveDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));
    result.ifPresent(dataSenseInfo -> {
      final MetadataTypeWriter metadataTypeWriter = new MetadataTypeWriter();
      final MetadataType outputType =
          dataSenseInfo.getOutput().orElseThrow(() -> new AssertionError("Expected output type not present"));
      assertThat(metadataTypeWriter.toString(outputType),
                 is("%type _:Java = {\n"
                     + "  \"message\" : @typeId(\"value\" : \"org.mule.runtime.api.message.Message\") {\n"
                     + "    \"payload\" : @classInformation(\"classname\" : \"java.io.InputStream\", \"hasDefaultConstructor\" : true, \"isInterface\" : false, \"isInstantiable\" : false, \"isAbstract\" : true, \"isFinal\" : false, \"implementedInterfaces\" : [java.io.Closeable], \"parent\" : \"\", \"genericTypes\" : [], \"isMap\" : false) Any, \n"
                     + "    \"attributes\" : @typeId(\"value\" : \"org.mule.extension.http.api.HttpResponseAttributes\") @classInformation(\"classname\" : \"org.mule.extension.http.api.HttpResponseAttributes\", \"hasDefaultConstructor\" : false, \"isInterface\" : false, \"isInstantiable\" : false, \"isAbstract\" : false, \"isFinal\" : false, \"implementedInterfaces\" : [], \"parent\" : \"org.mule.extension.http.api.HttpAttributes\", \"genericTypes\" : [], \"isMap\" : false) @typeAlias(\"value\" : \"HttpResponseAttributes\") {\n"
                     + "      @expressionSupport(\"value\" : SUPPORTED) \"statusCode\" : @classInformation(\"classname\" : \"int\", \"hasDefaultConstructor\" : false, \"isInterface\" : false, \"isInstantiable\" : false, \"isAbstract\" : true, \"isFinal\" : true, \"implementedInterfaces\" : [], \"parent\" : \"\", \"genericTypes\" : [], \"isMap\" : false) @int Number, \n"
                     + "      @expressionSupport(\"value\" : SUPPORTED) \"reasonPhrase\" : String, \n"
                     + "      @expressionSupport(\"value\" : SUPPORTED) \"headers\" : @classInformation(\"classname\" : \"org.mule.runtime.api.util.MultiMap\", \"hasDefaultConstructor\" : true, \"isInterface\" : false, \"isInstantiable\" : true, \"isAbstract\" : false, \"isFinal\" : false, \"implementedInterfaces\" : [java.util.Map, java.io.Serializable], \"parent\" : \"\", \"genericTypes\" : [java.lang.String, java.lang.String], \"isMap\" : true) {\n"
                     + "        * : String\n"
                     + "      }\n"
                     + "    }\n"
                     + "  }, \n"
                     + "  \"variables\" : {\n"
                     + "\n"
                     + "  }\n"
                     + "}"));
    });
    dataSenseRequest =
        createDataSenseRequest(builder().globalName(DYNAMIC_DATA_SENSE_FLOW).addProcessorsPart()
            .addIndexPart(1).addPart("route").addIndexPart(1).addProcessorsPart().addIndexPart(0).build());
    result = toolingArtifact.dataSenseService().resolveDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));
    result.ifPresent(dataSenseInfo -> {
      final MetadataTypeWriter metadataTypeWriter = new MetadataTypeWriter();
      final MetadataType outputType =
          dataSenseInfo.getOutput().orElseThrow(() -> new AssertionError("Expected output type not present"));
      assertThat(metadataTypeWriter.toString(outputType),
                 is("%type _:Java = {\n  "
                     + "\"message\" : @typeId(\"value\" : \"org.mule.runtime.api.message.Message\") {\n"
                     + "    \"payload\" : @classInformation(\"classname\" : \"org.mule.runtime.extension.api.runtime.streaming.PagingProvider\", \"hasDefaultConstructor\" : false, \"isInterface\" : true, \"isInstantiable\" : false, \"isAbstract\" : true, \"isFinal\" : false, \"implementedInterfaces\" : [], \"parent\" : \"\", \"genericTypes\" : [org.mule.extension.email.internal.mailbox.MailboxConnection, org.mule.runtime.extension.api.runtime.operation.Result], \"isMap\" : false) [@typeId(\"value\" : \"org.mule.runtime.api.message.Message\") {\n"
                     + "        \"payload\" : @typeId(\"value\" : \"org.mule.extension.email.api.StoredEmailContent\") @classInformation(\"classname\" : \"org.mule.extension.email.api.StoredEmailContent\", \"hasDefaultConstructor\" : false, \"isInterface\" : true, \"isInstantiable\" : false, \"isAbstract\" : true, \"isFinal\" : false, \"implementedInterfaces\" : [], \"parent\" : \"\", \"genericTypes\" : [], \"isMap\" : false) @typeAlias(\"value\" : \"StoredEmailContent\") {\n\n"
                     + "        }, \n"
                     + "        \"attributes\" : @typeId(\"value\" : \"org.mule.extension.email.api.attributes.IMAPEmailAttributes\") @classInformation(\"classname\" : \"org.mule.extension.email.api.attributes.IMAPEmailAttributes\", \"hasDefaultConstructor\" : false, \"isInterface\" : false, \"isInstantiable\" : false, \"isAbstract\" : false, \"isFinal\" : false, \"implementedInterfaces\" : [], \"parent\" : \"org.mule.extension.email.api.attributes.BaseEmailAttributes\", \"genericTypes\" : [], \"isMap\" : false) @typeAlias(\"value\" : \"IMAPEmailAttributes\") {\n"
                     + "          @expressionSupport(\"value\" : SUPPORTED) \"flags\" : @typeId(\"value\" : \"org.mule.extension.email.api.EmailFlags\") @classInformation(\"classname\" : \"org.mule.extension.email.api.EmailFlags\", \"hasDefaultConstructor\" : false, \"isInterface\" : false, \"isInstantiable\" : false, \"isAbstract\" : false, \"isFinal\" : false, \"implementedInterfaces\" : [], \"parent\" : \"\", \"genericTypes\" : [], \"isMap\" : false) @typeAlias(\"value\" : \"EmailFlags\") {\n"
                     + "            @default(\"value\" : \"false\") @expressionSupport(\"value\" : SUPPORTED) \"answered\"? : @typeId(\"value\" : \"boolean\") Boolean, \n"
                     + "            @default(\"value\" : \"false\") @expressionSupport(\"value\" : SUPPORTED) \"deleted\"? : @typeId(\"value\" : \"boolean\") Boolean, \n"
                     + "            @default(\"value\" : \"false\") @expressionSupport(\"value\" : SUPPORTED) \"draft\"? : @typeId(\"value\" : \"boolean\") Boolean, \n"
                     + "            @default(\"value\" : \"false\") @expressionSupport(\"value\" : SUPPORTED) \"recent\"? : @typeId(\"value\" : \"boolean\") Boolean, \n"
                     + "            @default(\"value\" : \"false\") @expressionSupport(\"value\" : SUPPORTED) \"seen\"? : @typeId(\"value\" : \"boolean\") Boolean\n"
                     + "          }, \n"
                     + "          @expressionSupport(\"value\" : SUPPORTED) \"number\" : @classInformation(\"classname\" : \"int\", \"hasDefaultConstructor\" : false, \"isInterface\" : false, \"isInstantiable\" : false, \"isAbstract\" : true, \"isFinal\" : true, \"implementedInterfaces\" : [], \"parent\" : \"\", \"genericTypes\" : [], \"isMap\" : false) @int Number, \n"
                     + "          @expressionSupport(\"value\" : SUPPORTED) \"fromAddresses\" : @classInformation(\"classname\" : \"java.util.List\", \"hasDefaultConstructor\" : false, \"isInterface\" : true, \"isInstantiable\" : false, \"isAbstract\" : true, \"isFinal\" : false, \"implementedInterfaces\" : [java.util.Collection], \"parent\" : \"\", \"genericTypes\" : [java.lang.String], \"isMap\" : false) [String], \n"
                     + "          @expressionSupport(\"value\" : SUPPORTED) \"toAddresses\" : @classInformation(\"classname\" : \"java.util.List\", \"hasDefaultConstructor\" : false, \"isInterface\" : true, \"isInstantiable\" : false, \"isAbstract\" : true, \"isFinal\" : false, \"implementedInterfaces\" : [java.util.Collection], \"parent\" : \"\", \"genericTypes\" : [java.lang.String], \"isMap\" : false) [String], \n"
                     + "          @expressionSupport(\"value\" : SUPPORTED) \"ccAddresses\" : @classInformation(\"classname\" : \"java.util.List\", \"hasDefaultConstructor\" : false, \"isInterface\" : true, \"isInstantiable\" : false, \"isAbstract\" : true, \"isFinal\" : false, \"implementedInterfaces\" : [java.util.Collection], \"parent\" : \"\", \"genericTypes\" : [java.lang.String], \"isMap\" : false) [String], \n"
                     + "          @expressionSupport(\"value\" : SUPPORTED) \"bccAddresses\" : @classInformation(\"classname\" : \"java.util.List\", \"hasDefaultConstructor\" : false, \"isInterface\" : true, \"isInstantiable\" : false, \"isAbstract\" : true, \"isFinal\" : false, \"implementedInterfaces\" : [java.util.Collection], \"parent\" : \"\", \"genericTypes\" : [java.lang.String], \"isMap\" : false) [String], \n"
                     + "          @expressionSupport(\"value\" : SUPPORTED) \"replyToAddresses\" : @classInformation(\"classname\" : \"java.util.List\", \"hasDefaultConstructor\" : false, \"isInterface\" : true, \"isInstantiable\" : false, \"isAbstract\" : true, \"isFinal\" : false, \"implementedInterfaces\" : [java.util.Collection], \"parent\" : \"\", \"genericTypes\" : [java.lang.String], \"isMap\" : false) [String], \n"
                     + "          @expressionSupport(\"value\" : SUPPORTED) \"headers\" : @classInformation(\"classname\" : \"java.util.Map\", \"hasDefaultConstructor\" : false, \"isInterface\" : true, \"isInstantiable\" : false, \"isAbstract\" : true, \"isFinal\" : false, \"implementedInterfaces\" : [], \"parent\" : \"\", \"genericTypes\" : [java.lang.String, java.lang.String], \"isMap\" : true) {\n"
                     + "            * : String\n"
                     + "          }, \n"
                     + "          @expressionSupport(\"value\" : SUPPORTED) \"subject\" : String, \n"
                     + "          @expressionSupport(\"value\" : SUPPORTED) \"receivedDate\"? : @typeId(\"value\" : \"java.time.LocalDateTime\") DateTime, \n"
                     + "          @expressionSupport(\"value\" : SUPPORTED) \"sentDate\"? : @typeId(\"value\" : \"java.time.LocalDateTime\") DateTime\n"
                     + "        }\n"
                     + "      }], \n"
                     + "    \"attributes\" : Void\n"
                     + "  }, \n"
                     + "  \"variables\" : {\n"
                     + "\n"
                     + "  }\n"
                     + "}"));
    });
  }

  @Test
  @Description("Checks Component DataSense resolution using HTTP")
  public void resolveComponentDataSense() throws Exception {
    startMuleRuntime();
    ToolingRuntimeClient toolingRuntimeClient = createToolingRuntimeClient();
    final ToolingArtifact toolingArtifact = newToolingArtifact(toolingRuntimeClient, APP_LOCATION);
    final DataSenseRequest dataSenseRequest =
        createDataSenseRequest(builder().globalName(DYNAMIC_DATA_SENSE_FLOW).addProcessorsPart()
            .addIndexPart(0).build());
    Optional<DataSenseComponentInfo> result = toolingArtifact.dataSenseService().resolveComponentDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));
    result.ifPresent(dataSenseInfo -> {
      Optional<OperationModel> operationModel = dataSenseInfo.getOperationModel();
      assertThat(operationModel.isPresent(), is(true));
      OperationModel model = operationModel.get();
      assertThat(model.getName(), is("request"));
    });
  }

  @Test
  public void resolveComponentDataSenseForSource() throws Exception {
    ToolingRuntimeClient toolingRuntimeClient = createToolingRuntimeClient();
    final ToolingArtifact toolingArtifact = newToolingArtifact(toolingRuntimeClient,
                                                               "applications/app-using-extension-with-only-sources");
    final DataSenseRequest dataSenseRequest =
        createDataSenseRequest(builder().globalName("flow").addSourcePart().build());
    Optional<DataSenseComponentInfo> result = toolingArtifact.dataSenseService().resolveComponentDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));
    result.ifPresent(dataSenseInfo -> {
      Optional<SourceModel> sourceModel = dataSenseInfo.getSourceModel();
      assertThat(sourceModel.isPresent(), is(true));
      SourceModel model = sourceModel.get();
      assertThat(model.getName(), is("BasicSource"));
    });
  }

  @Test
  public void resolveComponentDataSenseWithoutKeyEnrichment() throws Exception {
    startMuleRuntime();
    ToolingRuntimeClient toolingRuntimeClient = createToolingRuntimeClient();
    final ToolingArtifact toolingArtifact = newToolingArtifact(toolingRuntimeClient, DATA_SENSE_SUB_FLOW_APP);
    final DataSenseRequest dataSenseRequest =
        createDataSenseRequest(new ComponentResolutionScope(builder().globalName("ws_consumer").addProcessorsPart()
            .addIndexPart(0).build(), false));
    Optional<DataSenseInfo> result = toolingArtifact.dataSenseService().resolveDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));
    DataSenseInfo dataSenseInfo = result.get();

    assertThat(dataSenseInfo.getOperationModel().get().getParameterGroupModel("General").get().getParameterModel("operation")
        .get().getType().getAnnotation(
                                       EnumAnnotation.class)
        .isPresent(), is(false));
  }

  @Test
  @Description("Checks DataSense can be resolved for sub-flow components")
  public void resolveDataSenseOnSubFlowComponents() throws Exception {
    startMuleRuntime();
    ToolingRuntimeClient toolingRuntimeClient = createToolingRuntimeClient();
    final ToolingArtifact toolingArtifact = newToolingArtifact(toolingRuntimeClient, DATA_SENSE_SUB_FLOW_APP);
    final DataSenseRequest dataSenseRequest =
        createDataSenseRequest(builder().globalName("ws_consumer").addProcessorsPart()
            .addIndexPart(0).build());
    Optional<DataSenseInfo> result = toolingArtifact.dataSenseService().resolveDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));
    DataSenseInfo dataSenseInfo = result.get();
    assertThat(dataSenseInfo.getOutput().isPresent(), is(true));

    final MetadataType outputType =
        dataSenseInfo.getOutput()
            .orElseThrow(() -> new AssertionError("Expected attribute type not present"));

    assertThat(
               dataSenseInfo.getOperationModel().get().getParameterGroupModel("General").get().getParameterModel("operation")
                   .get().getType().getAnnotation(
                                                  EnumAnnotation.class)
                   .get().getValues(),
               arrayContaining("GetWeather", "GetCitiesByCountry"));

    final MetadataTypeWriter metadataTypeWriter = new MetadataTypeWriter();
    assertThat(metadataTypeWriter.toString(outputType), equalTo("%type _:Java = {\n"
        + "  \"message\" : @typeId(\"value\" : \"org.mule.runtime.api.message.Message\") {\n"
        + "\n"
        + "  }, \n"
        + "  \"variables\" : {\n"
        + "    \"wsc_result\" : {\n"
        + "      \"body\"? : @typeId(\"value\" : \"#root:{http://www.webserviceX.NET}GetWeatherResponse\") {\n"
        + "        \"{http://www.webserviceX.NET}GetWeatherResponse\" : @typeId(\"value\" : \"{http://www.webserviceX.NET}#AnonType_GetWeatherResponse\") {{\n"
        + "          \"{http://www.webserviceX.NET}GetWeatherResult\"? : String\n"
        + "        }}\n"
        + "      }\n"
        + "    }\n"
        + "  }\n"
        + "}"));

    assertThat(dataSenseInfo.getMessages(), hasSize(1));
  }

  @Test
  public void unableToLocateDocWscAppWithDomain() throws Exception {
    startMuleRuntime();
    ToolingRuntimeClient toolingRuntimeClient = createToolingRuntimeClient();
    final ToolingArtifact toolingArtifact = newToolingArtifact(toolingRuntimeClient, "applications/wsc-unable-to-locate-doc");
    final DataSenseRequest dataSenseRequest =
        createDataSenseRequest(builder().globalName("ws_consumer").addProcessorsPart()
            .addIndexPart(0).build());
    Optional<DataSenseInfo> result = toolingArtifact.dataSenseService().resolveDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));
    DataSenseInfo dataSenseInfo = result.get();
    assertThat(dataSenseInfo.getOutput().isPresent(), is(true));

    final MetadataType outputType =
        dataSenseInfo.getOutput()
            .orElseThrow(() -> new AssertionError("Expected attribute type not present"));

    assertThat(
               dataSenseInfo.getOperationModel().get().getParameterGroupModel("General").get().getParameterModel("operation")
                   .get().getType().getAnnotation(
                                                  EnumAnnotation.class)
                   .get().getValues(),
               arrayContaining("GetWeather", "GetCitiesByCountry"));

    final MetadataTypeWriter metadataTypeWriter = new MetadataTypeWriter();
    assertThat(metadataTypeWriter.toString(outputType), equalTo("%type _:Java = {\n"
        + "  \"message\" : @typeId(\"value\" : \"org.mule.runtime.api.message.Message\") {\n"
        + "\n"
        + "  }, \n"
        + "  \"variables\" : {\n"
        + "    \"wsc_result\" : {\n"
        + "      \"body\"? : @typeId(\"value\" : \"#root:{http://www.webserviceX.NET}GetWeatherResponse\") {\n"
        + "        \"{http://www.webserviceX.NET}GetWeatherResponse\" : @typeId(\"value\" : \"{http://www.webserviceX.NET}#AnonType_GetWeatherResponse\") {{\n"
        + "          \"{http://www.webserviceX.NET}GetWeatherResult\"? : String\n"
        + "        }}\n"
        + "      }\n"
        + "    }\n"
        + "  }\n"
        + "}"));

    assertThat(dataSenseInfo.getMessages(), hasSize(1));
  }

  @Test
  @Description("Checks DataSense resolution for HTTP listener config is not initialized as part of Application Model")
  public void resolveDataSenseShouldNotInitializeListenerConfiguration() throws Exception {
    startMuleRuntime();
    ToolingRuntimeClient toolingRuntimeClient = createToolingRuntimeClient();
    ToolingArtifact toolingArtifact = newToolingArtifact(toolingRuntimeClient, APP_LOCATION);
    DataSenseRequest dataSenseRequest =
        createDataSenseRequest(builder().globalName(DYNAMIC_DATA_SENSE_FLOW).addProcessorsPart()
            .addIndexPart(0).build());
    Optional<DataSenseComponentInfo> result = toolingArtifact.dataSenseService().resolveComponentDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));
    result.ifPresent(dataSenseInfo -> {
      Optional<OperationModel> operationModel = dataSenseInfo.getOperationModel();
      assertThat(operationModel.isPresent(), is(true));
      OperationModel model = operationModel.get();
      assertThat(model.getName(), is("request"));
    });

    toolingArtifact = newToolingArtifact(toolingRuntimeClient, APP_LOCATION);
    result = toolingArtifact.dataSenseService().resolveComponentDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));
    DataSenseComponentInfo dataSenseInfo = result.get();
    Optional<OperationModel> operationModel = dataSenseInfo.getOperationModel();
    assertThat(operationModel.isPresent(), is(true));
    OperationModel model = operationModel.get();
    assertThat(model.getName(), is("request"));
    assertThat(dataSenseInfo.getDataSenseNotifications(),
               containsInAnyOrder(allOf(notificationTypeIs(infoDataSenseNotificationType("INFO")),
                                        notificationMessageContains("Resolving datasense info"))));
  }

  @Test
  @Description("Checks DataSense resolution using HTTP requester referencing to a invalid config")
  public void resolveDataSenseOnHttpRequesterWithBadConfiguration() throws Exception {
    startMuleRuntime();
    final ToolingRuntimeClient toolingRuntimeClient = createToolingRuntimeClient();
    final ToolingArtifact toolingArtifact = newToolingArtifact(toolingRuntimeClient, BAD_CONFIG_REF_APP_LOCATION);

    final DataSenseRequest dataSenseRequest =
        createDataSenseRequest(builder().globalName("dataFlow").addProcessorsPart()
            .addIndexPart(0).build());
    Optional<DataSenseInfo> result = toolingArtifact.dataSenseService().resolveDataSense(dataSenseRequest);
    assertThat(result.isPresent(), is(true));
    final DataSenseInfo dataSenseInfo = result.get();
    assertThat(dataSenseInfo.getDataSenseNotifications(), hasSize(2));
    assertThat(dataSenseInfo.getDataSenseNotifications(), containsInAnyOrder(
                                                                             allOf(notificationTypeIs(errorDataSenseNotificationType("ERROR")),
                                                                                   notificationMessageContains("'beanName' must not be empty")),
                                                                             allOf(notificationTypeIs(infoDataSenseNotificationType("INFO")),
                                                                                   notificationMessageContains("Resolving datasense info"))));
  }

}
