/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.tests.integration.tooling.client;

import static java.util.Optional.empty;
import static org.hamcrest.CoreMatchers.not;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.Matchers.startsWith;
import static org.mule.maven.client.test.MavenTestHelper.createDefaultEnterpriseMavenConfiguration;
import static org.mule.maven.client.test.MavenTestUtils.getMavenProperty;
import static org.mule.tooling.client.api.descriptors.ArtifactDescriptor.newBuilder;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.EXTENSION_WITH_METADATA_CONNECTOR_VERSION;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.POM_FOLDER_FINDER;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getMuleVersion;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getTestLog4JConfigurationFile;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getToolingVersion;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import org.mule.tooling.client.api.extension.model.MuleVersion;
import org.mule.tooling.client.api.extension.model.parameter.ParameterModel;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrap;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapConfiguration;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapFactory;
import org.mule.tooling.client.tests.integration.category.DoesNotNeedMuleRuntimeTest;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.rules.TemporaryFolder;

@Category(DoesNotNeedMuleRuntimeTest.class)
@Feature("ExtensionModelFilterService")
@Story("Integration tests for ExtensionModelService to validate loading an extension model for a previous Runtime version")
public class ExtensionModelFilterTestCase {

  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  @Test
  @Description("Checks loading extension model for Mule components using latest Tooling Runtime Client version but targetRuntime set to 4.1.0")
  public void loadMuleCoreExtensionModel41() {
    withToolingRuntimeClient("4.1.0", toolingRuntimeClient -> {
      List<ExtensionModel> muleExtensionModels = toolingRuntimeClient.extensionModelService().loadMuleExtensionModels();
      Optional<ExtensionModel> coreExtensionModel =
          muleExtensionModels.stream().filter(e -> e.getName().equals("mule")).findFirst();
      assertThat(coreExtensionModel.isPresent(), is(true));
      assertThat(coreExtensionModel.get().getDescription(), containsString(": Core components"));
      assertThat(coreExtensionModel.get().getVersion(), startsWith("4.1."));
      assertThat(coreExtensionModel.get().getMinMuleVersion(), nullValue());

      Optional<ExtensionModel> eeExtensionModel =
          muleExtensionModels.stream().filter(e -> e.getName().equals("ee")).findFirst();
      assertThat(eeExtensionModel.isPresent(), is(true));
      assertThat(eeExtensionModel.get().getDescription(), containsString(": Core EE components"));
      assertThat(coreExtensionModel.get().getVersion(), startsWith("4.1."));
      assertThat(eeExtensionModel.get().getMinMuleVersion(), nullValue());
    });
  }

  @Test
  public void filterMimeTypeParameters() {
    withToolingRuntimeClient(getMuleVersionWithoutSnapshot(), toolingRuntimeClient -> {
      final ExtensionModel extensionModel = loadExtensionModel(toolingRuntimeClient);
      assertThat(getOutputMimeTypeParameterModel(extensionModel), not(empty()));
    });

    withToolingRuntimeClient("4.1.0", toolingRuntimeClient -> {
      final ExtensionModel extensionModel = loadExtensionModel(toolingRuntimeClient);
      assertThat(getOutputMimeTypeParameterModel(extensionModel), is(empty()));
    });
  }

  private static String getMuleVersionWithoutSnapshot() {
    return new MuleVersion(getMuleVersion()).toCompleteNumericVersion();
  }

  @Test
  public void filterExternalLibAttributes() {
    withToolingRuntimeClient(getMuleVersionWithoutSnapshot(), toolingRuntimeClient -> {
      final ExtensionModel extensionModel = loadDBExtensionModel(toolingRuntimeClient);
      extensionModel.getConnectionProviders().forEach(cp -> cp.getExternalLibraryModels()
          .forEach(elm -> {
            assertThat(elm.allowsDeclarationAsAdditionalDependency().isEnabled(), is(true));
            assertThat(elm.allowsDeclarationAsAdditionalDependency().get(), is(true));
          }));
    });

    withToolingRuntimeClient("4.1.6", toolingRuntimeClient -> {
      final ExtensionModel extensionModel = loadDBExtensionModel(toolingRuntimeClient);
      extensionModel.getConnectionProviders().forEach(cp -> cp.getExternalLibraryModels()
          .forEach(elm -> {
            assertThat(elm.allowsDeclarationAsAdditionalDependency().isEnabled(), is(true));
            assertThat(elm.allowsDeclarationAsAdditionalDependency().get(), is(false));
          }));
    });
  }

  private Optional<ParameterModel> getOutputMimeTypeParameterModel(ExtensionModel extensionModel) {
    return extensionModel.getConfigurationModel("config")
        .flatMap(configurationModel -> configurationModel.getOperationModel("operationOne")
            .flatMap(operationModel -> operationModel.getParameterGroupModel("General")
                .flatMap(parameterGroupModel -> parameterGroupModel.getParameterModel("outputMimeType"))));
  }

  private ExtensionModel loadExtensionModel(ToolingRuntimeClient toolingRuntimeClient2) {
    Optional<ExtensionModel> extensionModelOptional = toolingRuntimeClient2.extensionModelService()
        .loadExtensionModel(newBuilder()
            .withGroupId("org.mule.tooling")
            .withArtifactId("extension-with-metadata")
            .withClassifier("mule-plugin")
            .withVersion(getMavenProperty(EXTENSION_WITH_METADATA_CONNECTOR_VERSION, POM_FOLDER_FINDER))
            .build());

    assertThat(extensionModelOptional.isPresent(), is(true));
    return extensionModelOptional.get();
  }

  private ExtensionModel loadDBExtensionModel(ToolingRuntimeClient toolingRuntimeClient) {
    Optional<ExtensionModel> extensionModelOptional = toolingRuntimeClient.extensionModelService()
        .loadExtensionModel(newBuilder()
            .withGroupId("org.mule.connectors")
            .withArtifactId("mule-db-connector")
            .withVersion("1.0.0")
            .withClassifier("mule-plugin")
            .build());

    assertThat(extensionModelOptional.isPresent(), is(true));
    return extensionModelOptional.get();
  }

  private void withToolingRuntimeClient(String muleVersion, Consumer<ToolingRuntimeClient> task) {
    ToolingRuntimeClientBootstrap bootstrap = null;
    try {
      bootstrap = getToolingRuntimeClientBootstrap();
      ToolingRuntimeClient.Builder builder = bootstrap.getToolingRuntimeClientBuilderFactory().create();
      builder.withTargetRuntimeVersion().ifEnabled(action -> action.perform(muleVersion));
      task.accept(builder.build());
    } finally {
      if (bootstrap != null) {
        bootstrap.dispose();
      }
    }
  }

  private ToolingRuntimeClientBootstrap getToolingRuntimeClientBootstrap() throws UncheckedIOException {
    try {
      return ToolingRuntimeClientBootstrapFactory.newToolingRuntimeClientBootstrap(
                                                                                   ToolingRuntimeClientBootstrapConfiguration
                                                                                       .builder()
                                                                                       .toolingVersion(getToolingVersion())
                                                                                       .mavenConfiguration(createDefaultEnterpriseMavenConfiguration())
                                                                                       .log4jConfiguration(getTestLog4JConfigurationFile())
                                                                                       .workingFolder(temporaryFolder
                                                                                           .newFolder())
                                                                                       .build());
    } catch (IOException e) {
      throw new UncheckedIOException(e);
    }
  }

}
