/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.tests.integration.tooling.client;

import static java.lang.String.format;
import static java.lang.System.clearProperty;
import static java.lang.System.setProperty;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.apache.commons.io.FileUtils.toFile;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThanOrEqualTo;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.Matchers.sameInstance;
import static org.hamcrest.Matchers.startsWith;
import static org.hamcrest.collection.IsArrayWithSize.arrayWithSize;
import static org.junit.rules.ExpectedException.none;
import static org.mule.maven.client.test.MavenTestHelper.createDefaultEnterpriseMavenConfiguration;
import static org.mule.maven.client.test.MavenTestUtils.getMavenProperty;
import static org.mule.tooling.client.api.descriptors.ArtifactDescriptor.newBuilder;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.MULE_DB_CONNECTOR_VERSION;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.MULE_EMAIL_CONNECTOR_VERSION;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.MULE_JMS_CONNECTOR_VERSION;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.MULE_SPRING_MODULE_VERSION;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.MULE_VERSION;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.MULE_WSC_CONNECTOR_VERSION;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.MUNIT_RUNNER_PLUGIN_VERSION;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.MUNIT_TOOLS_PLUGIN_VERSION;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.POM_FOLDER_FINDER;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getMuleVersion;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getTestLog4JConfigurationFile;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getToolingVersion;
import org.mule.maven.client.api.BundleDependenciesResolutionException;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.declaration.type.annotation.StereotypeTypeAnnotation;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import org.mule.tooling.client.api.extension.model.MuleVersion;
import org.mule.tooling.client.api.extension.model.StereotypeModel;
import org.mule.tooling.client.api.extension.model.config.ConfigurationModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterGroupModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterModel;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrap;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapConfiguration;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapFactory;
import org.mule.tooling.client.tests.integration.category.DoesNotNeedMuleRuntimeTest;

import com.google.common.io.Files;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.apache.commons.io.filefilter.AndFileFilter;
import org.apache.commons.io.filefilter.PrefixFileFilter;
import org.apache.commons.io.filefilter.SuffixFileFilter;
import org.hamcrest.Matcher;
import org.junit.After;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Category(DoesNotNeedMuleRuntimeTest.class)
@Feature("ExtensionModelService")
@Story("Integration tests for ExtensionModelService using ToolingBootstrap and ToolingRuntimeClient")
public class ExtensionModelTestCase {

  private final Logger logger = LoggerFactory.getLogger(this.getClass());

  private static final String EXTENSIONS_FOLDER = "extensions";
  public static final String JAR = ".jar";
  private ToolingRuntimeClientBootstrap bootstrap;
  private ToolingRuntimeClient toolingRuntimeClient;

  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  @Rule
  public ExpectedException expectedException = none();

  @Before
  public void setUpToolingRuntimeClient() throws Exception {
    bootstrap = ToolingRuntimeClientBootstrapFactory.newToolingRuntimeClientBootstrap(
                                                                                      ToolingRuntimeClientBootstrapConfiguration
                                                                                          .builder()
                                                                                          .muleVersion(getMuleVersion())
                                                                                          .toolingVersion(getToolingVersion())
                                                                                          .mavenConfiguration(createDefaultEnterpriseMavenConfiguration())
                                                                                          .log4jConfiguration(getTestLog4JConfigurationFile())
                                                                                          .workingFolder(temporaryFolder
                                                                                              .newFolder())
                                                                                          .build());
    toolingRuntimeClient = bootstrap.getToolingRuntimeClientBuilderFactory().create().build();
  }

  @After
  public void disposeBootstrap() throws Exception {
    if (bootstrap != null) {
      bootstrap.dispose();
    }
  }

  @Test
  @Description("Checks that ToolingRuntimeClient can be used with a different MavenConfiguration instead of the one set to the Bootstrap")
  public void useDifferentMavenClientForToolingClient() throws Exception {
    bootstrap = ToolingRuntimeClientBootstrapFactory.newToolingRuntimeClientBootstrap(
                                                                                      ToolingRuntimeClientBootstrapConfiguration
                                                                                          .builder()
                                                                                          .muleVersion(getMuleVersion())
                                                                                          .toolingVersion(getToolingVersion())
                                                                                          .mavenConfiguration(createDefaultEnterpriseMavenConfiguration())
                                                                                          .log4jConfiguration(getTestLog4JConfigurationFile())
                                                                                          .workingFolder(temporaryFolder
                                                                                              .newFolder())
                                                                                          .build());
    MavenConfiguration.MavenConfigurationBuilder otherMavenConfigurationBuilder =
        new MavenConfiguration.MavenConfigurationBuilder();
    otherMavenConfigurationBuilder.localMavenRepositoryLocation(temporaryFolder.newFolder());
    toolingRuntimeClient =
        bootstrap.getToolingRuntimeClientBuilderFactory().create()
            .withMavenConfiguration(otherMavenConfigurationBuilder.build())
            .build();

    expectedException.expect(BundleDependenciesResolutionException.class);
    expectedException
        .expectMessage(containsString("Could not find artifact org.mule.connectors:mule-email-connector:jar:mule-plugin"));

    toolingRuntimeClient.extensionModelService()
        .loadExtensionModel(newBuilder()
            .withGroupId("org.mule.connectors")
            .withArtifactId("mule-email-connector")
            .withClassifier("mule-plugin")
            .withVersion(getMavenProperty(MULE_EMAIL_CONNECTOR_VERSION, POM_FOLDER_FINDER))
            .build());

  }

  @Test
  @Description("Checks loading extension model for Mule components")
  public void loadMuleCoreExtensionModelCurrentVersion() {
    List<ExtensionModel> muleExtensionModels = toolingRuntimeClient.extensionModelService().loadMuleExtensionModels();
    assertMuleRuntimeExtensionModels(muleExtensionModels, empty());
  }

  private void assertMuleRuntimeExtensionModels(List<ExtensionModel> muleExtensionModels, Optional<Matcher> muleVersionOptional) {
    Optional<ExtensionModel> coreExtensionModel =
        muleExtensionModels.stream().filter(e -> e.getName().equals("mule")).findFirst();
    assertThat(coreExtensionModel.isPresent(), is(true));
    assertThat(coreExtensionModel.get().getDescription(), containsString(": Core components"));
    assertThat(coreExtensionModel.get().getVersion(),
               muleVersionOptional.orElse(is(getMavenProperty(MULE_VERSION, POM_FOLDER_FINDER))));
    assertThat(coreExtensionModel.get().getMinMuleVersion(), nullValue());

    Optional<ExtensionModel> eeExtensionModel =
        muleExtensionModels.stream().filter(e -> e.getName().equals("ee")).findFirst();
    assertThat(eeExtensionModel.isPresent(), is(true));
    assertThat(eeExtensionModel.get().getDescription(), containsString(": Core EE components"));
    assertThat(eeExtensionModel.get().getVersion(),
               muleVersionOptional.orElse(is(getMavenProperty(MULE_VERSION, POM_FOLDER_FINDER))));
    assertThat(eeExtensionModel.get().getMinMuleVersion(), nullValue());
  }

  @Test
  @Description("Checks loading extension model for Mule components")
  public void loadMuleCoreExtensionModel42() {
    ToolingRuntimeClient.Builder toolingRuntimeClientBuilder = bootstrap.getToolingRuntimeClientBuilderFactory().create();
    toolingRuntimeClientBuilder.withTargetRuntimeVersion().ifEnabled(action -> action.perform("4.2.0-hf1"));
    ToolingRuntimeClient toolingRuntimeClient = toolingRuntimeClientBuilder.build();
    List<ExtensionModel> muleExtensionModels = toolingRuntimeClient.extensionModelService().loadMuleExtensionModels();
    assertMuleRuntimeExtensionModels(muleExtensionModels, empty());
  }

  @Test
  @Description("Checks loading extension model for Mule components")
  public void loadMuleCoreExtensionModel41() throws IOException {
    ToolingRuntimeClient.Builder toolingRuntimeClientBuilder = bootstrap.getToolingRuntimeClientBuilderFactory().create();
    toolingRuntimeClientBuilder.withTargetRuntimeVersion().ifEnabled(action -> action.perform("4.1.0"));
    ToolingRuntimeClient toolingRuntimeClient = toolingRuntimeClientBuilder.build();

    List<ExtensionModel> muleExtensionModels = toolingRuntimeClient.extensionModelService().loadMuleExtensionModels();
    assertMuleRuntimeExtensionModels(muleExtensionModels, of(startsWith("4.1.")));
  }

  @Test
  @Description("Checks loading extension model for file common plugin")
  public void getExtensionModelFileCommonPlugin() throws Exception {
    File pluginJarFile = findExtensionModelJarFile(this.getClass(), "mule-module-file-extension-common");
    Optional<ExtensionModel> extensionModelOptional =
        toolingRuntimeClient.extensionModelService().loadExtensionModel(pluginJarFile);

    assertThat(extensionModelOptional.isPresent(), is(false));
  }

  @Test
  @Description("Should resolve extension model from a file without trying to resolve it using Maven")
  public void resolveExtensionModelFromFile() throws Exception {
    File pluginJarFile = toFile(this.getClass().getClassLoader().getResource("extensions/americanapi-1.0.1-mule-plugin.jar"));
    Optional<ExtensionModel> extensionModelOptional =
        toolingRuntimeClient.extensionModelService().loadExtensionModel(pluginJarFile);

    assertThat(extensionModelOptional.isPresent(), is(true));
    final ExtensionModel extensionModel = extensionModelOptional.get();
    assertThat(extensionModel.getName(), equalTo("Z My American Flights API"));
  }

  @Test
  @Description("Checks loading extension model for a plugin that requires JMS API to be shipped with Mule Container")
  public void getExtensionModelFromPluginWithRequiredLibrary() throws Exception {
    Optional<ExtensionModel> extensionModelOptional = toolingRuntimeClient.extensionModelService()
        .loadExtensionModel(newBuilder()
            .withGroupId("org.mule.connectors")
            .withArtifactId("mule-jms-connector")
            .withClassifier("mule-plugin")
            .withVersion(getMavenProperty(MULE_JMS_CONNECTOR_VERSION, POM_FOLDER_FINDER))
            .build());

    assertThat(extensionModelOptional.isPresent(), is(true));
    final ExtensionModel extensionModel = extensionModelOptional.get();
    assertThat(extensionModel.getName(), equalTo("JMS"));
  }

  @Test
  public void getExtensionModelForMuleHttpProxyExtension() {
    File pluginJarFile = findExtensionModelJarFile(this.getClass(), "mule-http-proxy-extension");
    Optional<ExtensionModel> extensionModelOptional =
        toolingRuntimeClient.extensionModelService().loadExtensionModel(pluginJarFile);

    assertThat(extensionModelOptional.isPresent(), is(true));
    final ExtensionModel extensionModel = extensionModelOptional.get();
    assertThat(extensionModel.getName(), equalTo("proxy"));
  }

  @Test
  @Description("Checks loading extension model for MUnit Tools")
  public void loadExtensionModelMUnitTools() throws Exception {
    Optional<ExtensionModel> extensionModelOptional = toolingRuntimeClient.extensionModelService()
        .loadExtensionModel(newBuilder()
            .withGroupId("com.mulesoft.munit")
            .withArtifactId("munit-tools")
            .withClassifier("mule-plugin")
            .withVersion(getMavenProperty(MUNIT_TOOLS_PLUGIN_VERSION, POM_FOLDER_FINDER))
            .build());

    assertThat(extensionModelOptional.isPresent(), is(true));
    final ExtensionModel extensionModel = extensionModelOptional.get();
    assertThat(extensionModel.getName(), equalTo("MUnit Tools"));
  }

  @Test
  @Description("Checks loading extension model for MUnit Runner")
  public void loadExtensionModelMUnitRunner() throws Exception {
    Optional<ExtensionModel> extensionModelOptional = toolingRuntimeClient.extensionModelService()
        .loadExtensionModel(newBuilder()
            .withGroupId("com.mulesoft.munit")
            .withArtifactId("munit-runner")
            .withClassifier("mule-plugin")
            .withVersion(getMavenProperty(MUNIT_RUNNER_PLUGIN_VERSION, POM_FOLDER_FINDER))
            .build());

    assertThat(extensionModelOptional.isPresent(), is(true));
    final ExtensionModel extensionModel = extensionModelOptional.get();
    assertThat(extensionModel.getName(), equalTo("MUnit"));
  }

  @Test
  @Description("Checks loading extension model for email extension as jar")
  public void getExtensionModelForSimpleJarPlugin() {
    File pluginJarFile = findExtensionModelJarFile(this.getClass(), "mule-email-connector");
    getExtensionModelFromSimplePlugin(pluginJarFile);
  }

  @Test
  @Description("Checks loading extension model for email extension as artifact declaration")
  public void getExtensionModelForArtifact() {
    Optional<ExtensionModel> extensionModelOptional = toolingRuntimeClient.extensionModelService()
        .loadExtensionModel(newBuilder()
            .withGroupId("org.mule.connectors")
            .withArtifactId("mule-email-connector")
            .withClassifier("mule-plugin")
            .withVersion(getMavenProperty(MULE_EMAIL_CONNECTOR_VERSION, POM_FOLDER_FINDER))
            .build());

    assertThat(extensionModelOptional.isPresent(), is(true));
    final ExtensionModel extensionModel = extensionModelOptional.get();
    assertThat(extensionModel.getName(), equalTo("Email"));
  }

  @Test
  @Description("Checks loading extension model for db extension as artifact declaration")
  public void getExtensionModelForDbArtifact() {
    Optional<ExtensionModel> extensionModelOptional = toolingRuntimeClient.extensionModelService()
        .loadExtensionModel(newBuilder()
            .withGroupId("org.mule.connectors")
            .withArtifactId("mule-db-connector")
            .withClassifier("mule-plugin")
            .withVersion(getMavenProperty(MULE_DB_CONNECTOR_VERSION, POM_FOLDER_FINDER))
            .build());

    assertThat(extensionModelOptional.isPresent(), is(true));
    final ExtensionModel extensionModel = extensionModelOptional.get();
    assertThat(extensionModel.getName(), equalTo("Database"));
  }

  @Test
  @Description("Checks loading extension model for wsc extension as artifact declaration")
  public void getExtensionModelForWscPlugin() {
    Optional<ExtensionModel> extensionModelOptional = toolingRuntimeClient.extensionModelService()
        .loadExtensionModel(newBuilder()
            .withGroupId("org.mule.connectors")
            .withArtifactId("mule-wsc-connector")
            .withClassifier("mule-plugin")
            .withVersion(getMavenProperty(MULE_WSC_CONNECTOR_VERSION, POM_FOLDER_FINDER))
            .build());

    assertThat(extensionModelOptional.isPresent(), is(true));
    final ExtensionModel extensionModel = extensionModelOptional.get();
    assertThat(extensionModel.getName(), equalTo("Web Service Consumer"));
    checkStereotypeFromType(extensionModel, "HTTP", "org.mule.extension.ws.api.transport.CustomHttpTransportConfiguration",
                            "REQUEST_CONFIG");
  }

  @Test
  @Description("Checks loading extension model for extension that has a reference to an object store")
  public void getExtensionModelForObjectStoreReferencePlugin() {
    Optional<ExtensionModel> extensionModelOptional = toolingRuntimeClient.extensionModelService()
        .loadExtensionModel(newBuilder()
            .withGroupId("org.mule.modules")
            .withArtifactId("extension-with-objectstore-ref")
            .withClassifier("mule-plugin")
            .withVersion("4.0.0")
            .build());

    assertThat(extensionModelOptional.isPresent(), is(true));
    final ExtensionModel extensionModel = extensionModelOptional.get();
    assertThat(extensionModel.getName(), equalTo("ObjectStoreRef"));
    checkStereotypeFromType(extensionModel, "OS", "org.mule.modules.basic.PojoWithRef", "OBJECT_STORE");
  }

  @Test
  @Description("Checks loading extension model for test extension that has a POJO with a flowRef")
  public void extensionWithFlowRef() {
    Optional<ExtensionModel> extensionModelOptional = toolingRuntimeClient.extensionModelService()
        .loadExtensionModel(newBuilder()
            .withGroupId("org.mule.modules")
            .withArtifactId("extension-with-flow-ref")
            .withClassifier("mule-plugin")
            .withVersion("4.0.0")
            .build());

    assertThat(extensionModelOptional.isPresent(), is(true));
    final ExtensionModel extensionModel = extensionModelOptional.get();
    assertThat(extensionModel.getName(), equalTo("FlowRef"));
    checkStereotypeFromType(extensionModel, "MULE", "org.mule.modules.basic.PojoWithRef", "FLOW");
  }

  private void checkStereotypeFromType(ExtensionModel extensionModel, String allowedStereotypeNamespace,
                                       String className, String allowedStereotypeName) {
    assertThat(extensionModel.getTypes().size(), greaterThanOrEqualTo(1));
    final ObjectType objectType = extensionModel.getTypes().stream()
        .filter(type -> type.getAnnotation(ClassInformationAnnotation.class).map(classInformation -> classInformation
            .getClassname().equals(className)).orElse(false))
        .findFirst().orElseThrow(() -> new AssertionError("Missing type"));
    assertThat(objectType.getFields().size(), greaterThanOrEqualTo(1));
    ObjectFieldType fieldType = objectType.getFields().iterator().next();
    final Optional<StereotypeTypeAnnotation> stereotypeTypeAnnotationOptional =
        fieldType.getAnnotation(StereotypeTypeAnnotation.class);
    assertThat(stereotypeTypeAnnotationOptional.isPresent(), is(true));
    assertThat(stereotypeTypeAnnotationOptional.get().getStereotypeModel().size(), greaterThanOrEqualTo(1));
    final StereotypeModel stereotypeModel = stereotypeTypeAnnotationOptional.get().getStereotypeModel().iterator().next();
    assertThat(stereotypeModel.getNamespace(), equalTo(allowedStereotypeNamespace));
    assertThat(stereotypeModel.getType(), equalTo(allowedStereotypeName));
  }

  @Test
  @Description("Checks loading extension model for email extension from the exploded folder")
  public void getExtensionModelForSimpleExplodedPlugin() {
    File pluginFile = findExtensionModelJarFile(this.getClass(), "mule-email-connector");
    getExtensionModelFromSimplePlugin(pluginFile);
  }

  @Test
  public void getExtensionModelForSpringModulePlugin() {
    Optional<ExtensionModel> extensionModelOptional = toolingRuntimeClient.extensionModelService()
        .loadExtensionModel(newBuilder()
            .withGroupId("org.mule.modules")
            .withArtifactId("mule-spring-module")
            .withClassifier("mule-plugin")
            .withVersion(getMavenProperty(MULE_SPRING_MODULE_VERSION, POM_FOLDER_FINDER))
            .build());

    assertThat(extensionModelOptional.isPresent(), is(true));
    final ExtensionModel extensionModel = extensionModelOptional.get();
    assertThat(extensionModel.getName(), equalTo("Spring"));
    ConfigurationModel configurationModel = extensionModel.getConfigurationModels().get(0);
    assertThat(configurationModel.getName(), equalTo("config"));
    List<ParameterGroupModel> parameterGroupModels = configurationModel.getParameterGroupModels();
    assertThat(parameterGroupModels.size(), is(1));
    ParameterGroupModel parameterGroupModel = parameterGroupModels.get(0);
    List<ParameterModel> parameterModels = parameterGroupModel.getParameterModels();
    assertThat(parameterModels.size(), is(1));
    ParameterModel nameParameterModel = parameterModels.get(0);
    assertThat(nameParameterModel.getName(), is("files"));
    assertThat(nameParameterModel.getExpressionSupport().isNotSupported(), is(true));
  }

  private void getExtensionModelFromSimplePlugin(File pluginJarFile) {
    Optional<ExtensionModel> extensionModelOptional =
        toolingRuntimeClient.extensionModelService().loadExtensionModel(pluginJarFile);

    assertThat(extensionModelOptional.isPresent(), is(true));
    final ExtensionModel extensionModel = extensionModelOptional.get();
    assertThat(extensionModel.getName(), equalTo("Email"));
    assertThat(extensionModel.getMinMuleVersion(), equalTo("4.0.0"));
  }

  @Test
  @Description("Checks loading extension model for extension with dependencies on another extension")
  public void getExtensionModelForJarPluginWithDependency() {
    File pluginJarFile = findExtensionModelJarFile(this.getClass(), "mule-http-connector");
    assertExtensionModelGeneration(pluginJarFile);
  }

  @Test
  public void loadingExtensionModelWithDiskStoreEnabledShouldWriteTheFile() throws Exception {
    final File persistenceFolder = temporaryFolder.newFolder();
    withPersistence(persistenceFolder, (toolingRuntimeClient) -> {
      Optional<ExtensionModel> extensionModelOptional = loadHttpExtensionModel(toolingRuntimeClient);
      assertThat(extensionModelOptional.isPresent(), is(true));
      ExtensionModel extensionModel = extensionModelOptional.get();
      assertThat(extensionModel.getName(), equalTo("HTTP"));
      assertThat(extensionModel.getMinMuleVersion(), equalTo("4.1.0"));

      MuleVersion toolingVersion = new MuleVersion(getToolingVersion());
      final Path basePath =
          persistenceFolder.toPath().resolve(Paths.get(toolingVersion.toString(),
                                                       "org",
                                                       "mule", "connectors", "mule-http-connector", "1.1.0"));
      File extensionModelJsonFile = basePath.resolve("mule-http-connector-1.1.0-mule-plugin.json").toFile();
      assertThat(extensionModelJsonFile.exists(), is(true));

      File extensionSchemaFile = basePath.resolve("mule-http-connector-1.1.0-mule-plugin.xsd").toFile();
      assertThat(extensionSchemaFile.exists(), is(true));
    });
  }

  private Optional<ExtensionModel> loadHttpExtensionModel(ToolingRuntimeClient toolingRuntimeClient) {
    return toolingRuntimeClient.extensionModelService()
        .loadExtensionModel(newBuilder()
            .withGroupId("org.mule.connectors")
            .withArtifactId("mule-http-connector")
            .withClassifier("mule-plugin")
            .withVersion("1.1.0")
            .build());
  }

  @Test
  public void loadExtensionModelFromCacheFile() throws Exception {
    final File persistenceFolder = temporaryFolder.newFolder();
    MuleVersion muleVersion = new MuleVersion(getMuleVersion());
    File extensionModelJsonFile =
        persistenceFolder.toPath().resolve(Paths.get(format("%s.%s", muleVersion.getMajor(), muleVersion.getMinor()),
                                                     "org",
                                                     "mule", "connectors", "mule-http-connector", "1.1.0",
                                                     "mule-http-connector-1.1.0-mule-plugin.json"))
            .toFile();
    extensionModelJsonFile.getParentFile().mkdirs();
    extensionModelJsonFile.createNewFile();

    Files
        .copy(toFile(this.getClass().getClassLoader().getResource("extension-models/mule-http-connector-1.1.0-mule-plugin.json")),
              extensionModelJsonFile);

    withPersistence(persistenceFolder, (toolingRuntimeClient) -> {
      Optional<ExtensionModel> extensionModelOptional = loadHttpExtensionModel(toolingRuntimeClient);
      assertThat(extensionModelOptional.isPresent(), is(true));
      final ExtensionModel extensionModel = extensionModelOptional.get();
      assertThat(extensionModel.getName(), equalTo("HTTP"));
      assertThat(extensionModel.getMinMuleVersion(), equalTo("4.1.0"));

      assertThat(extensionModelJsonFile.exists(), is(true));
    });
  }

  private void withPersistence(File persistenceFolder, Consumer<ToolingRuntimeClient> consumer) throws Exception {
    final String persistenceCacheSystemPropertyName = "tooling.client.ExtensionModelServiceCache.diskStore.path";

    final String previous = setProperty(persistenceCacheSystemPropertyName, persistenceFolder.getAbsolutePath());
    ToolingRuntimeClientBootstrap bootstrap =
        ToolingRuntimeClientBootstrapFactory.newToolingRuntimeClientBootstrap(
                                                                              ToolingRuntimeClientBootstrapConfiguration
                                                                                  .builder()
                                                                                  .muleVersion(getMuleVersion())
                                                                                  .toolingVersion(getToolingVersion())
                                                                                  .mavenConfiguration(createDefaultEnterpriseMavenConfiguration())
                                                                                  .log4jConfiguration(getTestLog4JConfigurationFile())
                                                                                  .workingFolder(temporaryFolder.newFolder())
                                                                                  .build());
    ToolingRuntimeClient toolingRuntimeClient = bootstrap.getToolingRuntimeClientBuilderFactory().create().build();
    try {
      consumer.accept(toolingRuntimeClient);
    } finally {
      if (previous != null) {
        setProperty(persistenceCacheSystemPropertyName, persistenceFolder.getAbsolutePath());
      } else {
        clearProperty(persistenceCacheSystemPropertyName);
      }
      if (bootstrap != null) {
        bootstrap.dispose();
      }
    }
  }

  @Test
  @Description("Checks loading extension model for extension with dependencies on a plugin")
  public void getExtensionModelForExplodedPluginWithDependency() {
    File pluginJarFile = findExtensionModelJarFile(this.getClass(), "mule-http-connector");
    assertExtensionModelGeneration(pluginJarFile);
  }

  @Test
  @Ignore("TODO gfernandes: move this test to unit test for DefaultToolingClient")
  @Description("Checks loading extension model for extension is cached at the Bootstrap level")
  public void cacheExtensionModelAtBootstrapLevel() {
    File pluginJarFile = findExtensionModelJarFile(this.getClass(), "mule-http-connector");
    ExtensionModel httpExtensionModel = assertExtensionModelGeneration(pluginJarFile);

    toolingRuntimeClient = bootstrap.getToolingRuntimeClientBuilderFactory().create().build();
    ExtensionModel otherHttpExtensionModel = assertExtensionModelGeneration(pluginJarFile);

    assertThat(otherHttpExtensionModel, sameInstance(httpExtensionModel));
  }

  @Test
  @Description("Checks loading extension model for extension with dependencies on a file common plugin")
  public void getExtensionModelForPluginDependingOnCommonPlugin() {
    File pluginFile = findExtensionModelJarFile(this.getClass(), "mule-file-connector");
    Optional<ExtensionModel> fileExtensionModel = getExtensionModelForPluginWithDependencies(pluginFile);
    assertThat(fileExtensionModel.isPresent(), is(true));
    assertThat(fileExtensionModel.get().getName(), is("File"));
  }

  @Test
  @Description("Checks loading extension model for extension with dependencies on a file common plugin")
  public void getExtensionModelForCraftedExtension() {
    File pluginFile = findExtensionModelJarFile(this.getClass(), "test-crafted-extension");
    Optional<ExtensionModel> fileExtensionModel = getExtensionModelForPluginWithDependencies(pluginFile);
    assertThat(fileExtensionModel.isPresent(), is(true));
    assertThat(fileExtensionModel.get().getName(), is("crafted-extension"));
  }

  private ExtensionModel assertExtensionModelGeneration(File pluginJarFile) {
    Optional<ExtensionModel> extensionModelOptional =
        getExtensionModelForPluginWithDependencies(pluginJarFile);
    assertThat(extensionModelOptional.isPresent(), is(true));
    final ExtensionModel extensionModel = extensionModelOptional.get();
    assertThat(extensionModel.getName(), equalTo("HTTP"));
    return extensionModel;
  }

  @Test
  @Description("Checks loading of all the extension model to measure the resolution time")
  public void benchmarkToGetAllTheExtensionModel() throws Exception {
    if (logger.isDebugEnabled()) {
      logger.debug("Starting to get extension models for all extensions");
    }
    File[] allExtensions = findExtensionFiles(this.getClass(), new SuffixFileFilter(JAR));
    if (logger.isDebugEnabled()) {
      logger.debug("Found {} extensions to load", allExtensions.length);
    }

    for (File extension : allExtensions) {
      if (extension.getName().contains("jms")) {
        continue;
      }
      if (logger.isDebugEnabled()) {
        logger.debug("Getting ExtensionModel for: {}", extension);
      }
      toolingRuntimeClient = bootstrap.getToolingRuntimeClientBuilderFactory().create().build();
      Optional<ExtensionModel> extensionModelOptional = toolingRuntimeClient.extensionModelService()
          .loadExtensionModel(extension);

      if (extension.getName().startsWith("mule-module-file-extension-common")) {
        if (logger.isDebugEnabled()) {
          logger.debug("File commons has no ExtensionModel");
        }
        continue;
      }
      assertThat("ExtensionModel for " + extension + " couldn't be loaded", extensionModelOptional.isPresent(), is(true));
      if (logger.isDebugEnabled()) {
        logger.debug("ExtensionModel loaded: {}", extensionModelOptional.get().getName());
      }
    }
  }


  @Test
  @Description("Checks loading extension model for smart connector using core components")
  public void getExtensionModelForSmartConnectorPluginUsingCoreComponents() {
    assertExtensionModelFrom("smart-connector-using-core", "module-using-core");
  }

  @Test
  @Description("Checks loading extension model for smart connector using custom types")
  public void getExtensionModelForSmartConnectorPluginUsingCustomTypes() {
    assertExtensionModelFrom("smart-connector-using-custom-types", "module-using-custom-types");
  }

  @Test
  @Description("Checks loading extension model for smart connector using DW <ee:transform../> MP")
  public void getExtensionModelForSmartConnectorPluginUsingDW() {
    assertExtensionModelFrom("smart-connector-using-dw", "module-using-dw");
  }

  @Test
  @Description("Checks loading extension model for smart connector using file")
  public void getExtensionModelForSmartConnectorPluginUsingFile() {
    assertExtensionModelFrom("smart-connector-using-file", "module-using-file");
  }

  @Test
  @Description("Checks loading extension model for smart connector using http")
  public void getExtensionModelForSmartConnectorPluginUsingHttp() {
    assertExtensionModelFrom("smart-connector-using-http", "module-using-http");
  }

  @Test
  @Description("Checks loading extension model for smart connector using smart connector")
  public void getExtensionModelForSmartConnectorPluginUsingSmartConnector() {
    assertExtensionModelFrom("smart-connector-using-smart-connector", "module-using-sc");
  }

  private void assertExtensionModelFrom(String prefixArtifactPluginId, String expectedExtensionModelName) {
    final File pluginJarFile = findExtensionModelJarFile(this.getClass(), prefixArtifactPluginId);
    final Optional<ExtensionModel> extensionModelOptional =
        toolingRuntimeClient.extensionModelService().loadExtensionModel(pluginJarFile);
    assertThat(extensionModelOptional.isPresent(), is(true));
    final ExtensionModel extensionModel = extensionModelOptional.get();
    assertThat(extensionModel.getName(), equalTo(expectedExtensionModelName));
  }

  private Optional<ExtensionModel> getExtensionModelForPluginWithDependencies(File pluginJarFile) {
    return toolingRuntimeClient.extensionModelService().loadExtensionModel(pluginJarFile);
  }

  public static File findExtensionModelJarFile(Class<?> testClass, String prefixFileFilter) {
    final AndFileFilter filter = new AndFileFilter(new PrefixFileFilter(prefixFileFilter), new SuffixFileFilter(JAR));
    return findExtensionFile(testClass, filter);
  }

  public static File findExtensionFile(Class<?> testClass, FileFilter filter) {
    File[] extensionsFound = findExtensionFiles(testClass, filter);
    assertThat(extensionsFound, arrayWithSize(1));
    return extensionsFound[0];

  }

  private static File[] findExtensionFiles(Class<?> testClass, FileFilter filter) {
    File extensionsFolder = getExtensionsFolder(testClass);

    return extensionsFolder.listFiles(filter);
  }

  private static File getExtensionsFolder(Class<?> testClass) {
    try {
      return new File(new File(testClass.getProtectionDomain().getCodeSource().getLocation().toURI()).getParentFile(),
                      EXTENSIONS_FOLDER);
    } catch (URISyntaxException e) {
      throw new ToolingException("Error while getting testClass location", e);
    }
  }

}
