/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.tests.integration.tooling.client;

import static java.net.InetAddress.getLocalHost;
import static java.nio.charset.Charset.defaultCharset;
import static java.util.stream.Collectors.toList;
import static javax.mail.Message.RecipientType.TO;
import static org.apache.commons.io.IOUtils.readLines;
import static org.apache.commons.lang3.tuple.ImmutablePair.of;
import static org.apache.http.entity.ContentType.APPLICATION_JSON;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.emptyCollectionOf;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.equalToIgnoringCase;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.Matchers.startsWith;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.junit.Assert.assertThat;
import static org.junit.rules.ExpectedException.none;
import static org.mule.tck.junit4.matcher.StringContainsIgnoringLineBreaks.containsStringIgnoringLineBreaks;
import static org.mule.tooling.client.test.utils.ZipUtils.compress;
import static org.mule.tooling.client.tests.hamcrest.IsEquivalentTo.isSimilarTo;

import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.exception.NoSuchApplicationException;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.message.history.MessageHistory;
import org.mule.tooling.client.api.types.Transaction;
import org.mule.tooling.client.api.types.TransactionStackEntry;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrap;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapConfiguration;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapFactory;
import org.mule.tooling.client.test.AbstractMuleRuntimeTestCase;
import org.mule.tooling.client.test.RuntimeType;
import org.mule.tooling.client.test.utils.probe.JUnitLambdaProbe;
import org.mule.tooling.client.test.utils.probe.PollingProber;
import org.mule.tooling.client.tests.integration.category.NeedMuleRuntimeTest;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringReader;
import java.net.URL;
import java.nio.charset.Charset;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Properties;

import javax.mail.Session;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;

import com.google.common.collect.ImmutableList;
import com.google.common.net.MediaType;
import com.icegreen.greenmail.user.GreenMailUser;
import com.icegreen.greenmail.util.GreenMail;
import com.icegreen.greenmail.util.ServerSetup;
import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.HttpClientBuilder;
import org.h2.tools.DeleteDbFiles;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.rules.ExpectedException;
import org.junit.runners.Parameterized;
import org.mockftpserver.fake.FakeFtpServer;
import org.mockftpserver.fake.UserAccount;
import org.mockftpserver.fake.filesystem.FileEntry;
import org.mockftpserver.fake.filesystem.FileSystem;
import org.mockftpserver.fake.filesystem.UnixFakeFileSystem;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Category(NeedMuleRuntimeTest.class)
@Feature("MessageHistoryService")
@Story("Integration tests for MessageHistoryService using ToolingBootstrap and ToolingRuntimeClient")
public class MessageHistoryTestCase extends AbstractMuleRuntimeTestCase {

  static final transient Logger logger = LoggerFactory.getLogger(MessageHistoryTestCase.class);

  private static final String SCHEDULER_APP_LOCATION = "applications/scheduler";
  private static final String SCHEDULER_APP_NAME = "scheduler";
  private static final String FLOW_NAME = "sampleFlow";
  private static final String APP_LOCATION = "applications/http";
  private static final String APP_NAME = "http-app";
  private static final String SMART_CONNECTOR_APP_NAME = "smart-connector-app";
  private static final String SMART_CONNECTOR_APP_LOCATION = "applications/smart-connector";
  private static final String SMART_CONNECTOR_FAILING_APP_NAME = "smart-connector-failing";
  private static final String SMART_CONNECTOR_FAILING_APP_LOCATION = "applications/smart-connector-failing";
  private static final String DB_APP_LOCATION = "applications/streaming-db";
  private static final String DB_APP_NAME = "db-app";
  private static final String DB_DRIVER = "org.h2.Driver";
  private static final String DB_FOLDER = "./target/tmp/";
  private static final String DB_NAME = "streamingDB";
  private static final String DB_ENDPOINT = "jdbc:h2:" + DB_FOLDER + DB_NAME;
  private static final String FTP_APP_NAME = "ftp";
  private static final String FTP_APP_LOCATION = "applications/ftp";
  private static final String HTTP_LISTENER_SOURCE_RESPONSE_ERROR_APP_NAME = "http-listener-source-response-error";
  private static final String HTTP_LISTENER_SOURCE_RESPONSE_ERROR_APP_LOCATION =
      "applications/" + HTTP_LISTENER_SOURCE_RESPONSE_ERROR_APP_NAME;
  private static final String STREAMING_APP_LOCATION = "applications/streaming";
  private static final String STREAMING_APP_NAME = "streaming-app";
  private static final String APP_ERROR_LOCATION = "applications/http-error";
  private static final String APP_ERROR_NAME = "http-error";
  private static final String INVALID_APP_LOCATION = "applications/invalid";
  private static final String INVALID_APP_NAME = "invalid-app";
  private static final String DATAWEAVE_COLLECTION_APP_NAME = "dataweave-collection";
  private static final String DATAWEAVE_COLLECTION_APP_LOCATION = "applications/dataweave-collection";
  private static final String ASYNC_APP_NAME = "async";
  private static final String ASYNC_APP_LOCATION = "applications/async";
  private static final String TRY_APP_NAME = "try";
  private static final String TRY_ON_ERROR_CONTINUE_APP_NAME = "tryOnErrorContinue";
  private static final String TRY_APP_LOCATION = "applications/try";
  private static final String TRY_ON_ERROR_CONTINUE_APP_LOCATION = "applications/try-on-error-continue";
  private static final String WSC_APP_NAME = "wsc";
  private static final String WSC_APP_LOCATION = "applications/wsc";
  private static final String FOR_EACH_LOCATION = "applications/foreach";
  private static final String FOR_EACH_NAME = "foreach";
  private static final String FOR_EACH_ERROR_LOCATION = "applications/foreach-error";
  private static final String FOR_EACH_ERROR_NAME = "foreach-error";
  private static final String DATAWEAVE_COLLECTION_FIRST_TRANSFORMATION =
      "[\n"
          + "  {\n"
          + "    value1: 10,\n"
          + "    value2: 5\n"
          + "  }, \n"
          + "  {\n"
          + "    value1: 7,\n"
          + "    value2: 9\n"
          + "  }\n"
          + "]";
  private static final String DATAWEAVE_COLLECTION_SECOND_TRANSFORMATION =
      "[\n"
          + "  {\n"
          + "    PHONE: 10,\n"
          + "    GENDER: 5\n"
          + "  }, \n"
          + "  {\n"
          + "    PHONE: 7,\n"
          + "    GENDER: 9\n"
          + "  }\n"
          + "]";
  private static final String SAMPLE_FLOW_PROCESSORS_PATH = "sampleFlow/processors/";
  private static final String PAYLOAD_EXPECTED = "content";
  private static final String INPUT_XML =
      "<?xml version=\"1.0\" encoding=\"UTF-16\"?><listAllEmployees><return><firstName>John</firstName><lastName>Doe</lastName></return><return><firstName>Jean</firstName><lastName>Doe</lastName></return></listAllEmployees>";
  private static final String CSV_PAYLOAD = "firstName,lastName\nJohn,Doe\nJean,Doe\n";
  private static final String JSON_PAYLOAD = "[\n"
      + "  {\n"
      + "    NAME: \"Hello\",\n"
      + "    ID: 1\n"
      + "  }, \n"
      + "  {\n"
      + "    NAME: \"World\",\n"
      + "    ID: 2\n"
      + "  }\n"
      + "]";
  private static final Charset inputCharset = Charset.forName("UTF-16");
  private static final String APP_XML_MEDIA_TYPE = MediaType.create("application", "xml").withCharset(inputCharset).toString();
  public static final String SET_PAYLOAD_HARDCODED = "set-payload-hardcoded";
  public static final String WUBBA_LUBBA_DUB_DUB = "Wubba Lubba Dub Dub";
  private static final String SC_LIST = "list";
  private static final String FILE_CONNECTIVITY_ERROR = "FILE:CONNECTIVITY";

  private GreenMail greenMail;
  private GreenMailUser user;
  public static final Session testSession = Session.getDefaultInstance(new Properties());
  private static final String MARTIN_MAIL = "martin.buchwald@mulesoft.com";
  private static final String GUILLE_MAIL = "guillermo.fernandes@mulesoft.com";
  private static final String EMAIL_SUBJECT = "Some message";
  private static final String EMAIL_CONTENT = "Hello world!";

  private FakeFtpServer ftpServer;

  @Rule
  public ExpectedException expectedException = none();
  private ToolingRuntimeClient toolingRuntimeClient;
  private ToolingRuntimeClientBootstrap bootstrap;

  @Rule
  public DynamicPort httpServerPort = new DynamicPort("httpServerPort");
  @Rule
  public DynamicPort emailServerPort = new DynamicPort("emailServerPort");
  @Rule
  public DynamicPort ftpServerPort = new DynamicPort("ftpServerPort");

  public MessageHistoryTestCase(RuntimeType runtimeType) {
    super(runtimeType);
  }

  @Parameterized.Parameters(name = "{0}")
  public static Collection<Object[]> data() {
    return Arrays.asList(new Object[][] {
        {RuntimeType.REMOTE}
    });
  }


  @Override
  protected List<ImmutablePair<String, String>> getStartupSystemProperties() {
    try {
      File file = temporaryFolder.newFile();
      PrintWriter out = new PrintWriter(file);
      out.write("content");
      out.close();
      return ImmutableList.<ImmutablePair<String, String>>builder()
          .add(of("httpServerPort", String.valueOf(httpServerPort.getNumber())))
          .add(of("mule.agent.tracking.handler.messageHistory.enabled", "true"))
          .add(of("mule.agent.tracking.service.globalUseDataWeaveFormatForPojo", "true"))
          .add(of("path", file.getAbsolutePath()))
          .add(of("dbServerHost", getLocalHost().getHostAddress()))
          .add(of("emailServerPort", String.valueOf(emailServerPort.getNumber())))
          .add(of("ftpServerPort", String.valueOf(ftpServerPort.getNumber())))
          .add(of("fileEncoding", defaultCharset().toString())).build();
    } catch (IOException e) {
      throw new RuntimeException("Couldn't set up environment for test", e);
    }
  }

  @Before
  public void setUpToolingRuntimeClient() throws Exception {
    bootstrap = ToolingRuntimeClientBootstrapFactory.newToolingRuntimeClientBootstrap(
                                                                                      ToolingRuntimeClientBootstrapConfiguration
                                                                                          .builder()
                                                                                          .muleVersion(getMuleVersion())
                                                                                          .toolingVersion(getToolingVersion())
                                                                                          .mavenConfiguration(defaultMavenConfiguration)
                                                                                          .log4jConfiguration(getTestLog4JConfigurationFile())
                                                                                          .workingFolder(temporaryFolder
                                                                                              .newFolder())
                                                                                          .build());
    toolingRuntimeClient = bootstrap.getToolingRuntimeClientBuilderFactory().create()
        .withMavenConfiguration(defaultMavenConfiguration)
        .withRemoteAgentConfiguration(defaultAgentConfiguration)
        .build();
  }

  @After
  public void disposeBootstrap() throws Exception {
    disposeApplication(APP_NAME);
    disposeApplication(INVALID_APP_NAME);
    disposeApplication(APP_ERROR_NAME);
    disposeApplication(SCHEDULER_APP_NAME);
    disposeApplication(STREAMING_APP_NAME);
    disposeApplication(DB_APP_NAME);
    disposeApplication(SMART_CONNECTOR_APP_NAME);
    disposeApplication(SMART_CONNECTOR_FAILING_APP_NAME);
    disposeApplication(FTP_APP_NAME);
    disposeApplication(DATAWEAVE_COLLECTION_APP_NAME);
    disposeApplication(TRY_APP_NAME);
    disposeApplication(TRY_ON_ERROR_CONTINUE_APP_NAME);
    disposeApplication(WSC_APP_NAME);
    disposeApplication(FOR_EACH_ERROR_NAME);
    disposeApplication(FOR_EACH_NAME);
    disposeApplication(ASYNC_APP_NAME);
    disposeApplication(HTTP_LISTENER_SOURCE_RESPONSE_ERROR_APP_NAME);
    if (bootstrap != null) {
      bootstrap.dispose();
    }
  }

  private void setUpEmailServer() throws Exception {
    ServerSetup serverSetup = new ServerSetup(emailServerPort.getNumber(), null, "imap");
    greenMail = new GreenMail(serverSetup);
    user = greenMail.setUser(MARTIN_MAIL, "password");
    greenMail.start();
    sendEmails();
  }

  private void setupDBServer() throws Exception {
    Class.forName(DB_DRIVER);
    Connection dbConnection = DriverManager.getConnection(DB_ENDPOINT, "",
                                                          "");
    dbConnection.setAutoCommit(false);
    Statement stmt = dbConnection.createStatement();
    stmt.execute("CREATE TABLE TEST(ID int primary key, NAME varchar(255))");
    stmt.execute("INSERT INTO TEST(ID, NAME) VALUES(1, 'Hello')");
    stmt.execute("INSERT INTO TEST(ID, NAME) VALUES(2, 'World')");
    stmt.close();
    dbConnection.commit();
    dbConnection.close();
  }

  @After
  public void stopEmailServer() {
    if (greenMail != null) {
      greenMail.stop();
    }
  }

  @After
  public void deleteDB() {
    DeleteDbFiles.execute(DB_FOLDER, DB_NAME, true);
  }

  private void disposeApplication(String appName) throws Exception {
    if (toolingRuntimeClient != null) {
      toolingRuntimeClient.messageHistoryService().disable(appName);
      unDeployApplication(appName, false);
    }
  }

  @Test
  @Description("Checks enabling tryIt should fail when the application was not deployed")
  public void enableTryItOnApplicationNotDeployed() {
    expectedException.expect(NoSuchApplicationException.class);
    toolingRuntimeClient.messageHistoryService().enable(APP_NAME);
  }

  @Test
  @Description("Checks message history transactions for a valid application using HTTP listener")
  public void collectNotificationsHttpListener() throws Exception {
    deployApplication(APP_NAME, APP_LOCATION);
    toolingRuntimeClient.messageHistoryService().enable(APP_NAME);
    sendRequest();
    assertNotifications(APP_NAME);
  }

  @Test
  @Description("Checks message history transactions for a valid application using HTTP listener")
  public void collectDataWeaveCollectionNotifications() throws Exception {
    deployApplication(DATAWEAVE_COLLECTION_APP_NAME, DATAWEAVE_COLLECTION_APP_LOCATION);
    toolingRuntimeClient.messageHistoryService().enable(DATAWEAVE_COLLECTION_APP_NAME);
    sendRequest();
    assertDWNotifications(DATAWEAVE_COLLECTION_APP_NAME);
  }

  @Test
  @Description("Checks message history transactions for an async scope")
  public void messageHistoryAsyncScope() throws Exception {
    deployApplication(ASYNC_APP_NAME, ASYNC_APP_LOCATION);
    toolingRuntimeClient.messageHistoryService().enable(ASYNC_APP_NAME);
    sendRequest();
    assertAsyncNotifications(ASYNC_APP_NAME);
  }

  @Test
  @Description("Checks message history transactions for a valid application using Smart Connectors")
  public void collectNotificationsSmartConnectors() throws Exception {
    deployApplication(SMART_CONNECTOR_APP_NAME, SMART_CONNECTOR_APP_LOCATION);
    toolingRuntimeClient.messageHistoryService().enable(SMART_CONNECTOR_APP_NAME);
    sendRequest();
    assertSmartConnectorNotifications(SMART_CONNECTOR_APP_NAME);
  }

  @Test
  @Description("Checks message history transactions for a valid application using Smart Connectors")
  public void collectNotificationsSmartConnectorsThatFails() throws Exception {
    deployApplication(SMART_CONNECTOR_FAILING_APP_NAME, SMART_CONNECTOR_FAILING_APP_LOCATION);
    toolingRuntimeClient.messageHistoryService().enable(SMART_CONNECTOR_FAILING_APP_NAME);
    sendRequest(false);
    assertSmartConnectorFailingNotifications(SMART_CONNECTOR_FAILING_APP_NAME);
  }

  @Test
  @Description("Checks message history transactions for a valid application using a Scheduler")
  public void collectNotificationsScheduler() throws Exception {
    deployApplication(SCHEDULER_APP_NAME, SCHEDULER_APP_LOCATION);
    toolingRuntimeClient.messageHistoryService().enable(SCHEDULER_APP_NAME);
    startFlow(SCHEDULER_APP_NAME, FLOW_NAME);
    schedulerAssertNotifications(SCHEDULER_APP_NAME);
  }

  @Test
  @Description("Streaming with Email connector")
  public void collectNotificationsStreamingWithEmail() throws Exception {
    setUpEmailServer();
    deployApplication(STREAMING_APP_NAME, STREAMING_APP_LOCATION);
    toolingRuntimeClient.messageHistoryService().enable(STREAMING_APP_NAME);
    sendRequest();
    assertEmailNotifications(STREAMING_APP_NAME);
  }

  @Test
  @Description("Streaming with DB connector")
  public void collectNotificationsStreamingWithDB() throws Exception {
    setupDBServer();
    deployApplication(DB_APP_NAME, DB_APP_LOCATION);
    toolingRuntimeClient.messageHistoryService().enable(DB_APP_NAME);
    sendRequest();
    streamingAssertNotifications(DB_APP_NAME);
    deleteDB();
  }

  @Test
  @Description("Checks message history error message for a valid application when a an error happened")
  public void collectErrorNotifications() throws Exception {
    deployApplication(APP_ERROR_NAME, APP_ERROR_LOCATION);
    toolingRuntimeClient.messageHistoryService().enable(APP_ERROR_NAME);
    sendRequest(false);

    final List<Transaction> transactions = collectNotifications(APP_ERROR_NAME);
    assertThat(consumeNotifications(APP_ERROR_NAME).getTransactions(), emptyCollectionOf(Transaction.class));

    assertThat(transactions, hasSize(1));
    Transaction transaction = transactions.get(0);
    assertThat(transaction.getTransactionStack(), hasSize(1));
    TransactionStackEntry transactionStackEntry = transaction.getTransactionStack().get(0);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0"));
    assertThat(transactionStackEntry.getData().getError().getType(), equalTo("MULE:EXPRESSION"));
  }

  @Test
  @Description("Checks tryIt cannot be enabled when an application is invalid and failed to be deployed")
  public void collectNotificationsInvalidApp() throws Exception {
    deployApplication(INVALID_APP_NAME, INVALID_APP_LOCATION, false);
    expectedException.expect(ToolingException.class);
    toolingRuntimeClient.messageHistoryService().enable(INVALID_APP_NAME);
  }

  @Test
  @Description("Checks message history notifications cannot be consumed when an application is has not been deployed")
  public void consumeNotificationsNonExistingApp() throws Exception {
    expectedException.expect(NoSuchApplicationException.class);
    consumeNotifications(INVALID_APP_NAME);
  }

  private void startFtpServer() {
    ftpServer = new FakeFtpServer();
    ftpServer.addUserAccount(new UserAccount("username", "password", "/tmp"));
    ftpServer.setServerControlPort(ftpServerPort.getNumber());
    FileSystem fileSystem = new UnixFakeFileSystem();
    fileSystem.add(new FileEntry("/tmp/sample.txt", "abc123"));
    ftpServer.setFileSystem(fileSystem);
    ftpServer.start();
  }

  private void stopFtpServer() {
    ftpServer.stop();
  }

  @Test
  @Description("Checks tryIt with collection of messages (FTP list)")
  public void collectNotificationsFtpList() throws Exception {
    startFtpServer();
    deployApplication(FTP_APP_NAME, FTP_APP_LOCATION, false);

    toolingRuntimeClient.messageHistoryService().enable(FTP_APP_NAME);
    sendRequest();
    ftpAssertNotifications(FTP_APP_NAME);
    stopFtpServer();
  }

  @Test
  public void collectNotificationsHttpListenerResponseSendError() throws Exception {
    deployApplication(HTTP_LISTENER_SOURCE_RESPONSE_ERROR_APP_NAME, HTTP_LISTENER_SOURCE_RESPONSE_ERROR_APP_LOCATION, false);

    toolingRuntimeClient.messageHistoryService().enable(HTTP_LISTENER_SOURCE_RESPONSE_ERROR_APP_NAME);
    sendRequest(false);

    final List<Transaction> transactions = collectNotifications(HTTP_LISTENER_SOURCE_RESPONSE_ERROR_APP_NAME);
    assertThat(consumeNotifications(HTTP_LISTENER_SOURCE_RESPONSE_ERROR_APP_NAME).getTransactions(),
               emptyCollectionOf(Transaction.class));
    assertThat(transactions, hasSize(1));
    Transaction transaction = transactions.get(0);
    assertThat(transaction.getId(), is(notNullValue()));
    assertThat(transaction.getTimestamp(), is(notNullValue()));
    assertThat(transaction.getGlobalName(), is(FLOW_NAME));
    // Source - Attributes
    assertThat(transaction.getMessage(), is(notNullValue()));
    assertThat(transaction.getMessage().getAttributes().getDataType().getMediaType(),
               startsWith(MediaType.create("application", "dw").toString()));
    // Source - Payload
    assertThat(transaction.getMessage().getPayload().getDataType().getMediaType(),
               equalToIgnoringCase(APP_XML_MEDIA_TYPE));
    assertThat(new String(transaction.getMessage().getPayload().getContent(), inputCharset),
               isSimilarTo(INPUT_XML));

    // Stack Entries
    assertThat(transaction.getTransactionStack(), hasSize(1));

    // set-payload
    TransactionStackEntry transactionStackEntry = transaction.getTransactionStack().get(0);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("set-payload"));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));

    assertThat(new String(transactionStackEntry.getData().getMessage().getPayload().getContent(),
                          defaultCharset()),
               equalTo("Hi!"));
  }

  @Test
  @Description("Checks error notifications on try scope using on error propagate")
  public void collectErrorOnTryWithOnErrorPropagate() throws Exception {
    deployApplication(TRY_APP_NAME, TRY_APP_LOCATION, false);
    toolingRuntimeClient.messageHistoryService().enable(TRY_APP_NAME);
    sendRequest(false);
    assertTryOnErrorPropagateNotifications(TRY_APP_NAME);
  }

  @Test
  @Description("Checks error notifications on try scope using on error continue")
  public void collectErrorOnTryWithOnErrorContinue() throws Exception {
    deployApplication(TRY_ON_ERROR_CONTINUE_APP_NAME, TRY_ON_ERROR_CONTINUE_APP_LOCATION, false);
    toolingRuntimeClient.messageHistoryService().enable(TRY_ON_ERROR_CONTINUE_APP_NAME);
    sendRequest(false);
    assertTryOnErrorContinueNotifications(TRY_ON_ERROR_CONTINUE_APP_NAME);
  }

  @Test
  @Description("Checks error notifications on for each scenario, having only one notification")
  public void collectErrorOnForEach() throws Exception {
    deployApplication(FOR_EACH_ERROR_NAME, FOR_EACH_ERROR_LOCATION, false);
    toolingRuntimeClient.messageHistoryService().enable(FOR_EACH_ERROR_NAME);
    sendRequest(false);
    assertForEachErrorNotifications(FOR_EACH_ERROR_NAME);
  }

  @Test
  @Description("Checks for each scenario, retrieving one notification per iteration")
  public void collectNotificationsForEach() throws Exception {
    deployApplication(FOR_EACH_NAME, FOR_EACH_LOCATION);
    toolingRuntimeClient.messageHistoryService().enable(FOR_EACH_NAME);
    sendRequest(true);
    assertForEachNotifications(FOR_EACH_NAME);
  }


  @Test
  @Description("Checks tryIt could be enabled more than one time for the same application")
  public void collectNotificationsEnableTwoTimesSameApp() throws Exception {
    deployApplication(APP_NAME, APP_LOCATION);
    toolingRuntimeClient.messageHistoryService().enable(APP_NAME);
    sendRequest();
    assertNotifications(APP_NAME);

    sendRequest();
    toolingRuntimeClient.messageHistoryService().enable(APP_NAME);
    assertNotifications(APP_NAME);
  }

  @Test
  @Description("Checks tryIt keeps getting transactions between redeploys")
  public void collectNotificationsWhileRedeployHappens() throws Exception {
    deployApplication(APP_NAME, APP_LOCATION);
    toolingRuntimeClient.messageHistoryService().enable(APP_NAME);
    sendRequest();
    assertNotifications(APP_NAME);

    deployApplication(APP_NAME, APP_LOCATION);
    PollingProber prober = new PollingProber(5000, 200);
    prober.check(new JUnitLambdaProbe(() -> {
      try {
        sendRequest();
        assertNotifications(APP_NAME);
        return true;
      } catch (Exception e) {
        return false;
      }
    }, "Application couldn't redeploy"));
  }

  @Test
  @Description("Checks tryIt handles WSC payload")
  public void collectWscPayload() throws Exception {
    deployApplication(WSC_APP_NAME, WSC_APP_LOCATION);
    toolingRuntimeClient.messageHistoryService().enable(WSC_APP_NAME);
    startFlow(WSC_APP_NAME, "flow");
    final MessageHistory messageHistory = new MessageHistory();
    PollingProber prober = new PollingProber(8000, 500);
    prober.check(new JUnitLambdaProbe(() -> {
      MessageHistory consumedMessageHistory = consumeNotifications(WSC_APP_NAME);
      if (consumedMessageHistory.getTransactions().isEmpty()) {
        if (messageHistory.getTransactions().isEmpty()) {
          return false;
        }
        Transaction transaction = messageHistory.getTransactions().get(0);
        return transaction.getMessage() != null && transaction.getTransactionStack().size() == 1;
      }

      if (messageHistory.getTransactions().isEmpty()) {
        final Transaction consumedTransaction = consumedMessageHistory.getTransactions().get(0);
        messageHistory.getTransactions().add(consumedTransaction);
        if (consumedTransaction.getMessage() != null) {
          final Transaction transaction = messageHistory.getTransactions().get(0);
          transaction.setMessage(consumedTransaction.getMessage());
          transaction.setTimestamp(consumedTransaction.getTimestamp());
        }
      } else {
        final List<TransactionStackEntry> transactionStack = messageHistory.getTransactions().get(0).getTransactionStack();
        final Transaction consumedTransaction = consumedMessageHistory.getTransactions().get(0);
        transactionStack.addAll(consumedTransaction.getTransactionStack());
      }
      return isPollingDone(messageHistory);
    }, "Notifications from WSC flow couldn't be retrieved or were the ones expected"));

    final List<Transaction> transactions = messageHistory.getTransactions();
    assertThat(transactions, hasSize(1));
    Transaction transaction = transactions.get(0);
    assertThat(transaction.getId(), is(notNullValue()));
    assertThat(transaction.getTimestamp(), is(notNullValue()));
    // Source - Attributes
    assertThat(transaction.getMessage().getPayload().getContent(), is(nullValue()));
    assertThat(transaction.getMessage().getAttributes().getContent(), is(nullValue()));

    // Stack Entries
    assertThat(transaction.getTransactionStack(), hasSize(1));

    // wsc:consume
    TransactionStackEntry transactionStackEntry = transaction.getTransactionStack().get(0);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), equalTo("flow/processors/0"));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));
    assertThat(new String(transactionStackEntry.getData().getMessage().getPayload().getContent(), MediaType
        .parse(transactionStackEntry.getData().getMessage().getPayload().getDataType().getMediaType()).charset().get()),
               equalTo("{\n"
                   + "  headers: {},\n"
                   + "  attachments: {},\n"
                   + "  body: do {\n"
                   + "    ns ns2 http://soap.training.mulesoft.com/\n"
                   + "    ---\n"
                   + "    {\n"
                   + "      ns2#listAllFlightsResponse: {\n"
                   + "        return: {\n"
                   + "          airlineName: \"Delta\",\n"
                   + "          code: \"A1B2C3\",\n"
                   + "          departureDate: \"2015/03/20\",\n"
                   + "          destination: \"SFO\",\n"
                   + "          emptySeats: \"40\",\n"
                   + "          origin: \"MUA\",\n"
                   + "          planeType: \"Boing 737\",\n"
                   + "          price: \"400.0\"\n"
                   + "        },\n"
                   + "        return: {\n"
                   + "          airlineName: \"Delta\",\n"
                   + "          code: \"A1B2C4\",\n"
                   + "          departureDate: \"2015/02/11\",\n"
                   + "          destination: \"LAX\",\n"
                   + "          emptySeats: \"10\",\n"
                   + "          origin: \"MUA\",\n"
                   + "          planeType: \"Boing 737\",\n"
                   + "          price: \"199.99\"\n"
                   + "        },\n"
                   + "        return: {\n"
                   + "          airlineName: \"Delta\",\n"
                   + "          code: \"A134DS\",\n"
                   + "          departureDate: \"2015/04/11\",\n"
                   + "          destination: \"LAX\",\n"
                   + "          emptySeats: \"40\",\n"
                   + "          origin: \"MUA\",\n"
                   + "          planeType: \"Boing 777\",\n"
                   + "          price: \"750.0\"\n"
                   + "        },\n"
                   + "        return: {\n"
                   + "          airlineName: \"Delta\",\n"
                   + "          code: \"A1B34S\",\n"
                   + "          departureDate: \"2015/06/11\",\n"
                   + "          destination: \"CLE\",\n"
                   + "          emptySeats: \"50\",\n"
                   + "          origin: \"MUA\",\n"
                   + "          planeType: \"Boing 707\",\n"
                   + "          price: \"420.0\"\n"
                   + "        },\n"
                   + "        return: {\n"
                   + "          airlineName: \"Delta\",\n"
                   + "          code: \"A12342\",\n"
                   + "          departureDate: \"2015/07/11\",\n"
                   + "          destination: \"CLE\",\n"
                   + "          emptySeats: \"17\",\n"
                   + "          origin: \"MUA\",\n"
                   + "          planeType: \"Boing 727\",\n"
                   + "          price: \"308.0\"\n"
                   + "        },\n"
                   + "        return: {\n"
                   + "          airlineName: \"Delta\",\n"
                   + "          code: \"A1QWER\",\n"
                   + "          departureDate: \"2015/08/11\",\n"
                   + "          destination: \"LAX\",\n"
                   + "          emptySeats: \"18\",\n"
                   + "          origin: \"MUA\",\n"
                   + "          planeType: \"Boing 747\",\n"
                   + "          price: \"496.0\"\n"
                   + "        },\n"
                   + "        return: {\n"
                   + "          airlineName: \"Delta\",\n"
                   + "          code: \"A1ASD4\",\n"
                   + "          departureDate: \"2015/09/11\",\n"
                   + "          destination: \"CLE\",\n"
                   + "          emptySeats: \"40\",\n"
                   + "          origin: \"MUA\",\n"
                   + "          planeType: \"Boing 757\",\n"
                   + "          price: \"736.0\"\n"
                   + "        },\n"
                   + "        return: {\n"
                   + "          airlineName: \"Delta\",\n"
                   + "          code: \"A1BTT4\",\n"
                   + "          departureDate: \"2015/02/12\",\n"
                   + "          destination: \"SFO\",\n"
                   + "          emptySeats: \"30\",\n"
                   + "          origin: \"MUA\",\n"
                   + "          planeType: \"Boing 777\",\n"
                   + "          price: \"593.0\"\n"
                   + "        },\n"
                   + "        return: {\n"
                   + "          airlineName: \"Delta\",\n"
                   + "          code: \"A14244\",\n"
                   + "          departureDate: \"2015/02/12\",\n"
                   + "          destination: \"SFO\",\n"
                   + "          emptySeats: \"10\",\n"
                   + "          origin: \"MUA\",\n"
                   + "          planeType: \"Boing 787\",\n"
                   + "          price: \"294.0\"\n"
                   + "        },\n"
                   + "        return: {\n"
                   + "          airlineName: \"Delta\",\n"
                   + "          code: \"A1FGF4\",\n"
                   + "          departureDate: \"2015/02/13\",\n"
                   + "          destination: \"PDX\",\n"
                   + "          emptySeats: \"80\",\n"
                   + "          origin: \"MUA\",\n"
                   + "          planeType: \"Boing 777\",\n"
                   + "          price: \"958.0\"\n"
                   + "        },\n"
                   + "        return: {\n"
                   + "          airlineName: \"Delta\",\n"
                   + "          code: \"AFFFC4\",\n"
                   + "          departureDate: \"2015/02/20\",\n"
                   + "          destination: \"PDX\",\n"
                   + "          emptySeats: \"30\",\n"
                   + "          origin: \"MUA\",\n"
                   + "          planeType: \"Boing 777\",\n"
                   + "          price: \"283.0\"\n"
                   + "        },\n"
                   + "        return: {\n"
                   + "          airlineName: \"Delta\",\n"
                   + "          code: \"A1B3D4\",\n"
                   + "          departureDate: \"2015/02/12\",\n"
                   + "          destination: \"PDX\",\n"
                   + "          emptySeats: \"10\",\n"
                   + "          origin: \"MUA\",\n"
                   + "          planeType: \"Boing 777\",\n"
                   + "          price: \"385.0\"\n"
                   + "        }\n"
                   + "      }\n"
                   + "    }\n"
                   + "  }\n"
                   + "}"));
  }

  @Test
  @Description("Checks tryIt stops collecting transactions when application is undeployed")
  public void stopCollectingNotificationsOnUndeploy() throws Exception {
    deployApplication(APP_NAME, APP_LOCATION);
    toolingRuntimeClient.messageHistoryService().enable(APP_NAME);
    sendRequest();
    assertNotifications(APP_NAME);

    unDeployApplication(APP_NAME);
    assertNotNotifications(APP_NAME);

    deployApplication(APP_NAME, APP_LOCATION);
    PollingProber prober = new PollingProber(5000, 200);
    prober.check(new JUnitLambdaProbe(() -> {
      try {
        sendRequest();
        assertNotifications(APP_NAME);
        return true;
      } catch (Exception e) {
        return false;
      }
    }, "Application couldn't redeploy"));
  }

  private void assertNotNotifications(String appName) {
    PollingProber prober = new PollingProber(5000, 200);
    prober.check(new JUnitLambdaProbe(() -> {
      MessageHistory messageHistory = consumeNotifications(appName);
      if (messageHistory.getTransactions().isEmpty()) {
        return true;
      }
      return false;
    }, "Notifications couldn't be retrieved"));
  }

  private MessageHistory consumeNotifications(String appName) {
    return toolingRuntimeClient.messageHistoryService().consume(appName, 100);
  }

  private void schedulerAssertNotifications(String appName) {
    final MessageHistory messageHistory = new MessageHistory();
    PollingProber prober = new PollingProber(5000, 200);
    prober.check(new JUnitLambdaProbe(() -> {
      MessageHistory consumedMessageHistory = consumeNotifications(appName);
      if (consumedMessageHistory.getTransactions().isEmpty()) {
        return isPollingDone(messageHistory);
      }

      if (messageHistory.getTransactions().isEmpty()) {
        final Transaction consumedTransaction = consumedMessageHistory.getTransactions().get(0);
        messageHistory.getTransactions().add(consumedTransaction);
        if (consumedTransaction.getMessage() != null) {
          final Transaction transaction = messageHistory.getTransactions().get(0);
          transaction.setMessage(consumedTransaction.getMessage());
          transaction.setTimestamp(consumedTransaction.getTimestamp());
        }
      } else {
        final List<TransactionStackEntry> transactionStack = messageHistory.getTransactions().get(0).getTransactionStack();
        final Transaction consumedTransaction = consumedMessageHistory.getTransactions().get(0);
        transactionStack.addAll(consumedTransaction.getTransactionStack());
      }
      return isPollingDone(messageHistory);
    }, "Notifications from Poll flow couldn't be retrieved or were the ones expected"));

    checkSchedulerTransactions(messageHistory.getTransactions());
  }

  private boolean isPollingDone(MessageHistory messageHistory) {
    if (messageHistory.getTransactions().isEmpty()) {
      return false;
    }
    Transaction transaction = messageHistory.getTransactions().get(0);
    return transaction.getMessage() != null && transaction.getTransactionStack().size() == 3;
  }

  private void assertNotifications(String appName) {
    final List<Transaction> transactions = collectNotifications(appName);
    assertThat(consumeNotifications(appName).getTransactions(), emptyCollectionOf(Transaction.class));
    checkHttpTransactions(transactions);
  }

  private void assertEmailNotifications(String appName) {
    final List<Transaction> transactions = collectNotifications(appName);
    assertThat(consumeNotifications(appName).getTransactions(), emptyCollectionOf(Transaction.class));
    checkEmailTransactions(transactions);
  }

  private void assertDWNotifications(String appName) {
    final List<Transaction> transactions = collectNotifications(appName);
    assertThat(consumeNotifications(appName).getTransactions(), emptyCollectionOf(Transaction.class));
    checkDWTransactions(transactions);
  }

  private void assertAsyncNotifications(String appName) {
    final List<Transaction> transactions = collectNotifications(appName);
    assertThat(consumeNotifications(appName).getTransactions(), emptyCollectionOf(Transaction.class));
    checkAsyncTransactions(transactions);
  }

  private void assertTryOnErrorPropagateNotifications(String appName) {
    final List<Transaction> transactions = collectNotifications(appName);
    assertThat(consumeNotifications(appName).getTransactions(), emptyCollectionOf(Transaction.class));
    checkTryOnErrorPropagateTransactions(transactions);
  }

  private void assertTryOnErrorContinueNotifications(String appName) {
    final List<Transaction> transactions = collectNotifications(appName);
    assertThat(consumeNotifications(appName).getTransactions(), emptyCollectionOf(Transaction.class));
    checkTryOnErrorContinueTransactions(transactions);
  }

  private void assertForEachErrorNotifications(String appName) {
    final List<Transaction> transactions = collectNotifications(appName);
    assertThat(consumeNotifications(appName).getTransactions(), emptyCollectionOf(Transaction.class));
    checkForEachErrorTransactions(transactions);
  }

  private void assertForEachNotifications(String appName) {
    final List<Transaction> transactions = collectNotifications(appName);
    assertThat(consumeNotifications(appName).getTransactions(), emptyCollectionOf(Transaction.class));
    checkForEachTransactions(transactions);
  }

  private void assertSmartConnectorNotifications(String appName) {
    final List<Transaction> transactions = collectNotifications(appName);
    assertThat(consumeNotifications(appName).getTransactions(), emptyCollectionOf(Transaction.class));

    assertThat(transactions, hasSize(1));
    Transaction transaction = transactions.get(0);
    assertThat(transaction.getId(), is(notNullValue()));
    assertThat(transaction.getTimestamp(), is(notNullValue()));
    // Source - Attributes
    assertThat(transaction.getMessage(), is(notNullValue()));
    assertThat(transaction.getMessage().getAttributes().getDataType().getType(),
               equalTo(String.class.getName()));
    assertThat(transaction.getMessage().getAttributes().getDataType().getMediaType(),
               startsWith(MediaType.create("application", "dw").toString()));
    // Source - Payload
    assertThat(transaction.getMessage().getPayload().getDataType().getMediaType(),
               equalToIgnoringCase(APP_XML_MEDIA_TYPE));
    assertThat(new String(transaction.getMessage().getPayload().getContent(), inputCharset), isSimilarTo(INPUT_XML));

    // Stack Entries
    assertThat(transaction.getTransactionStack(), hasSize(2));

    // set-payload inside sc:set-payload-hardcoded
    TransactionStackEntry transactionStackEntry = transaction.getTransactionStack().get(0);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), equalTo(SET_PAYLOAD_HARDCODED + "/processors/0"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(),
               equalTo("set-payload"));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));
    assertThat(new String(transactionStackEntry.getData().getMessage().getPayload().getContent(), MediaType
        .parse(transactionStackEntry.getData().getMessage().getPayload().getDataType().getMediaType()).charset().get()),
               equalTo(WUBBA_LUBBA_DUB_DUB));

    // sc:set-payload-hardcoded
    transactionStackEntry = transaction.getTransactionStack().get(1);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(),
               equalTo(SET_PAYLOAD_HARDCODED));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));
    assertThat(new String(transactionStackEntry.getData().getMessage().getPayload().getContent(), MediaType
        .parse(transactionStackEntry.getData().getMessage().getPayload().getDataType().getMediaType()).charset().get()),
               equalTo(WUBBA_LUBBA_DUB_DUB));
  }

  private void assertSmartConnectorFailingNotifications(String appName) {
    final List<Transaction> transactions = collectNotifications(appName);
    assertThat(consumeNotifications(appName).getTransactions(), emptyCollectionOf(Transaction.class));

    assertThat(transactions, hasSize(1));
    Transaction transaction = transactions.get(0);
    assertThat(transaction.getId(), is(notNullValue()));
    assertThat(transaction.getTimestamp(), is(notNullValue()));
    // Source - Attributes
    assertThat(transaction.getMessage(), is(notNullValue()));
    assertThat(transaction.getMessage().getAttributes().getDataType().getType(),
               equalTo(String.class.getName()));
    assertThat(transaction.getMessage().getAttributes().getDataType().getMediaType(),
               startsWith(MediaType.create("application", "dw").toString()));
    // Source - Payload
    assertThat(transaction.getMessage().getPayload().getDataType().getMediaType(),
               equalToIgnoringCase(APP_XML_MEDIA_TYPE));
    assertThat(new String(transaction.getMessage().getPayload().getContent(), inputCharset), isSimilarTo(INPUT_XML));

    // Stack Entries
    assertThat(transaction.getTransactionStack(), hasSize(2));

    // list inside sc:list
    TransactionStackEntry transactionStackEntry = transaction.getTransactionStack().get(0);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), equalTo(SC_LIST + "/processors/0"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(),
               equalTo(SC_LIST));
    assertThat(transactionStackEntry.getData().getError(), is(notNullValue()));
    assertThat(transactionStackEntry.getData().getError().getType(), is(FILE_CONNECTIVITY_ERROR));
    assertThat(transactionStackEntry.getData().isSuccessful(), is(false));


    // sc:set-payload-hardcoded
    transactionStackEntry = transaction.getTransactionStack().get(1);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(),
               equalTo(SC_LIST));
    assertThat(transactionStackEntry.getData().getError(), is(notNullValue()));
    assertThat(transactionStackEntry.getData().getError().getType(), is(FILE_CONNECTIVITY_ERROR));
    // TODO (https://www.mulesoft.org/jira/browse/MULE-14647)
    // assertThat(transactionStackEntry.getData().isSuccessful(), is(false));
  }

  private void checkHttpTransactions(List<Transaction> transactions) {
    assertThat(transactions, hasSize(1));
    Transaction transaction = transactions.get(0);
    assertThat(transaction.getId(), is(notNullValue()));
    assertThat(transaction.getTimestamp(), is(notNullValue()));
    assertThat(transaction.getGlobalName(), is(FLOW_NAME));
    // Source - Attributes
    assertThat(transaction.getMessage(), is(notNullValue()));
    assertThat(transaction.getMessage().getAttributes().getDataType().getType(),
               equalTo(String.class.getName()));
    assertThat(transaction.getMessage().getAttributes().getDataType().getMediaType(),
               startsWith(MediaType.create("application", "dw").toString()));
    // Source - Payload
    assertThat(transaction.getMessage().getPayload().getDataType().getMediaType(),
               equalToIgnoringCase(APP_XML_MEDIA_TYPE));
    assertThat(new String(transaction.getMessage().getPayload().getContent(), inputCharset), isSimilarTo(INPUT_XML));

    // Stack Entries
    assertThat(transaction.getTransactionStack(), hasSize(2));

    // dw:transform
    TransactionStackEntry transactionStackEntry = transaction.getTransactionStack().get(0);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("transform"));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));
    assertThat(new String(transactionStackEntry.getData().getMessage().getPayload().getContent(), MediaType
        .parse(transactionStackEntry.getData().getMessage().getPayload().getDataType().getMediaType()).charset().get()),
               containsStringIgnoringLineBreaks(CSV_PAYLOAD));

    // file:read
    transactionStackEntry = transaction.getTransactionStack().get(1);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "1"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getNamespace(), equalTo("file"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("read"));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));
    assertThat(new String(transactionStackEntry.getData().getMessage().getPayload().getContent(), MediaType
        .parse(transactionStackEntry.getData().getMessage().getPayload().getDataType().getMediaType()).charset().get()),
               equalTo(PAYLOAD_EXPECTED));
  }

  private void checkTryOnErrorPropagateTransactions(List<Transaction> transactions) {
    assertThat(transactions, hasSize(1));
    // try
    Transaction transaction = transactions.get(0);
    assertThat(transaction.getTransactionStack(), hasSize(3));

    TransactionStackEntry transactionStackEntry = transaction.getTransactionStack().get(0);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(),
               equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0/processors/0"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("transform"));

    assertThat(transactionStackEntry.getData().getError(), is(notNullValue()));
    assertThat(transactionStackEntry.getData().isSuccessful(), is(false));

    transactionStackEntry = transaction.getTransactionStack().get(1);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(),
               equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0/errorHandler/0/processors/0"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("logger"));

    assertThat(transactionStackEntry.getData().getError(), is(notNullValue()));
    assertThat(transactionStackEntry.getData().isSuccessful(), is(true));

    transactionStackEntry = transaction.getTransactionStack().get(2);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("try"));

    assertThat(transactionStackEntry.getData().getError(), is(notNullValue()));
    assertThat(transactionStackEntry.getData().isSuccessful(), is(false));
  }

  private void checkTryOnErrorContinueTransactions(List<Transaction> transactions) {
    assertThat(transactions, hasSize(1));
    // try
    Transaction transaction = transactions.get(0);
    assertThat(transaction.getTransactionStack(), hasSize(4));

    TransactionStackEntry transactionStackEntry = transaction.getTransactionStack().get(0);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(),
               equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0/processors/0"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("transform"));

    assertThat(transactionStackEntry.getData().getError(), is(notNullValue()));
    assertThat(transactionStackEntry.getData().isSuccessful(), is(false));

    transactionStackEntry = transaction.getTransactionStack().get(1);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(),
               equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0/errorHandler/0/processors/0"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("logger"));

    assertThat(transactionStackEntry.getData().getError(), is(notNullValue()));
    assertThat(transactionStackEntry.getData().isSuccessful(), is(true));

    transactionStackEntry = transaction.getTransactionStack().get(2);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("try"));

    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));
    assertThat(transactionStackEntry.getData().isSuccessful(), is(true));

    transactionStackEntry = transaction.getTransactionStack().get(3);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(),
               equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "1"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("logger"));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));
    assertThat(transactionStackEntry.getData().isSuccessful(), is(true));

  }

  private void checkForEachErrorTransactions(List<Transaction> transactions) {
    assertThat(transactions, hasSize(1));
    Transaction transaction = transactions.get(0);
    assertThat(transaction.getTransactionStack(), hasSize(3));

    TransactionStackEntry transactionStackEntry = transaction.getTransactionStack().get(0);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(),
               equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0/processors/0"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("logger"));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));
    assertThat(transactionStackEntry.getData().isSuccessful(), is(true));

    transactionStackEntry = transaction.getTransactionStack().get(1);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(),
               equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0/processors/1"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("request"));
    assertThat(transactionStackEntry.getData().getError(), is(notNullValue()));
    assertThat(transactionStackEntry.getData().isSuccessful(), is(false));

    transactionStackEntry = transaction.getTransactionStack().get(2);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(),
               equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("foreach"));
    assertThat(transactionStackEntry.getData().getError(), is(notNullValue()));
    assertThat(transactionStackEntry.getData().isSuccessful(), is(false));
  }

  private void checkForEachTransactions(List<Transaction> transactions) {
    assertThat(transactions, hasSize(1));
    Transaction transaction = transactions.get(0);
    assertThat(transaction.getTransactionStack(), hasSize(5));

    TransactionStackEntry transactionStackEntry = transaction.getTransactionStack().get(0);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(),
               equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0/processors/0"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("logger"));
    assertThat(transactionStackEntry.getInput().getMessage().getPayload().getDataType().getMediaType(),
               containsString("application/dw; charset="));
    assertThat(new String(transactionStackEntry.getInput().getMessage().getPayload().getContent(), defaultCharset()),
               is("1"));
    assertThat(transactionStackEntry.getInput().getError(), is(nullValue()));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));
    assertThat(transactionStackEntry.getData().isSuccessful(), is(true));

    transactionStackEntry = transaction.getTransactionStack().get(1);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(),
               equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0/processors/1"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("transform"));
    assertThat(transactionStackEntry.getInput(), is(nullValue()));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));
    assertThat(transactionStackEntry.getData().isSuccessful(), is(true));

    transactionStackEntry = transaction.getTransactionStack().get(2);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(),
               equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0/processors/0"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("logger"));
    assertThat(transactionStackEntry.getInput().getMessage().getPayload().getDataType().getMediaType(),
               containsString("application/dw; charset="));
    assertThat(new String(transactionStackEntry.getInput().getMessage().getPayload().getContent(), defaultCharset()),
               is("2"));
    assertThat(transactionStackEntry.getInput().getError(), is(nullValue()));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));
    assertThat(transactionStackEntry.getData().isSuccessful(), is(true));

    transactionStackEntry = transaction.getTransactionStack().get(3);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(),
               equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0/processors/1"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("transform"));
    assertThat(transactionStackEntry.getInput(), is(nullValue()));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));
    assertThat(transactionStackEntry.getData().isSuccessful(), is(true));

    transactionStackEntry = transaction.getTransactionStack().get(4);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(),
               equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("foreach"));
    assertThat(transactionStackEntry.getInput().getMessage().getPayload().getDataType().getMediaType(),
               is("application/xml; charset=UTF-16"));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));
    assertThat(transactionStackEntry.getData().isSuccessful(), is(true));
  }

  private void checkEmailTransactions(List<Transaction> transactions) {
    assertThat(transactions, hasSize(1));
    Transaction transaction = transactions.get(0);
    assertThat(transaction.getId(), is(notNullValue()));
    assertThat(transaction.getTimestamp(), is(notNullValue()));
    // Source - Attributes
    assertThat(transaction.getMessage(), is(notNullValue()));
    assertThat(transaction.getMessage().getAttributes().getDataType().getType(),
               equalTo(String.class.getName()));
    assertThat(transaction.getMessage().getAttributes().getDataType().getMediaType(),
               startsWith(MediaType.create("application", "dw").toString()));
    // Source - Payload
    assertThat(transaction.getMessage().getPayload().getDataType().getMediaType(),
               equalToIgnoringCase(APP_XML_MEDIA_TYPE));
    assertThat(new String(transaction.getMessage().getPayload().getContent(), inputCharset), isSimilarTo(INPUT_XML));

    // Stack Entries
    assertThat(transaction.getTransactionStack(), hasSize(2));

    // dw:transform
    TransactionStackEntry transactionStackEntry = transaction.getTransactionStack().get(0);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("list-imap"));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));

    String content = new String(transactionStackEntry.getData().getMessage().getPayload().getContent(), MediaType
        .parse(transactionStackEntry.getData().getMessage().getPayload().getDataType().getMediaType()).charset().get());

    assertThat(content.contains(GUILLE_MAIL), is(true));
    assertThat(content.contains(MARTIN_MAIL), is(true));
    assertThat(content.contains(EMAIL_SUBJECT), is(true));
    assertThat(content.contains(EMAIL_CONTENT), is(true));

    transactionStackEntry = transaction.getTransactionStack().get(1);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "1"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("set-payload"));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));

    content = new String(transactionStackEntry.getData().getMessage().getPayload().getContent(), MediaType
        .parse(transactionStackEntry.getData().getMessage().getPayload().getDataType().getMediaType()).charset().get());

    assertThat(content.contains(GUILLE_MAIL), is(true));
    assertThat(content.contains(MARTIN_MAIL), is(true));
    assertThat(content.contains(EMAIL_SUBJECT), is(true));
    assertThat(content.contains(EMAIL_CONTENT), is(true));
  }

  private void checkAsyncTransactions(List<Transaction> transactions) {
    assertThat(transactions, hasSize(1));
    Transaction transaction = transactions.get(0);
    assertThat(transaction.getId(), is(notNullValue()));
    assertThat(transaction.getTimestamp(), is(notNullValue()));
    // Source - Attributes
    assertThat(transaction.getMessage(), is(notNullValue()));
    assertThat(transaction.getMessage().getAttributes().getDataType().getType(),
               equalTo(String.class.getName()));
    assertThat(transaction.getMessage().getAttributes().getDataType().getMediaType(),
               startsWith(MediaType.create("application", "dw").toString()));
    // Source - Payload
    assertThat(transaction.getMessage().getPayload().getDataType().getMediaType(),
               equalToIgnoringCase(APP_XML_MEDIA_TYPE));
    assertThat(new String(transaction.getMessage().getPayload().getContent(), inputCharset), isSimilarTo(INPUT_XML));

    // Stack Entries
    assertThat(transaction.getTransactionStack(), hasSize(3));

    // dw:transform
    TransactionStackEntry transactionStackEntry = transaction.getTransactionStack().get(0);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("transform"));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));
    assertThat(transactionStackEntry.getData().getMessage().getPayload().getDataType().getMediaType(),
               startsWith("application/dw"));
    assertThat(new String(transactionStackEntry.getData().getMessage().getPayload().getContent(), MediaType
        .parse(transactionStackEntry.getData().getMessage().getPayload().getDataType().getMediaType()).charset().get()),
               containsStringIgnoringLineBreaks(DATAWEAVE_COLLECTION_FIRST_TRANSFORMATION));
    // async
    transactionStackEntry = transaction.getTransactionStack().get(1);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "1"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("async"));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));
    assertThat(transactionStackEntry.getData().getMessage().getPayload().getDataType().getMediaType(),
               startsWith("application/dw"));
    assertThat(new String(transactionStackEntry.getData().getMessage().getPayload().getContent(), MediaType
        .parse(transactionStackEntry.getData().getMessage().getPayload().getDataType().getMediaType()).charset().get()),
               containsStringIgnoringLineBreaks(DATAWEAVE_COLLECTION_FIRST_TRANSFORMATION));

    // logger
    transactionStackEntry = transaction.getTransactionStack().get(2);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(),
               equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "1/processors/0"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("logger"));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));
    assertThat(transactionStackEntry.getData().getMessage().getPayload().getDataType().getMediaType(),
               startsWith("application/dw"));
    assertThat(new String(transactionStackEntry.getData().getMessage().getPayload().getContent(), MediaType
        .parse(transactionStackEntry.getData().getMessage().getPayload().getDataType().getMediaType()).charset().get()),
               containsStringIgnoringLineBreaks(DATAWEAVE_COLLECTION_FIRST_TRANSFORMATION));
  }

  private void checkDWTransactions(List<Transaction> transactions) {
    assertThat(transactions, hasSize(1));
    Transaction transaction = transactions.get(0);
    assertThat(transaction.getId(), is(notNullValue()));
    assertThat(transaction.getTimestamp(), is(notNullValue()));
    // Source - Attributes
    assertThat(transaction.getMessage(), is(notNullValue()));
    assertThat(transaction.getMessage().getAttributes().getDataType().getType(),
               equalTo(String.class.getName()));
    assertThat(transaction.getMessage().getAttributes().getDataType().getMediaType(),
               startsWith(MediaType.create("application", "dw").toString()));
    // Source - Payload
    assertThat(transaction.getMessage().getPayload().getDataType().getMediaType(),
               equalToIgnoringCase(APP_XML_MEDIA_TYPE));
    assertThat(new String(transaction.getMessage().getPayload().getContent(), inputCharset), isSimilarTo(INPUT_XML));

    // Stack Entries
    assertThat(transaction.getTransactionStack(), hasSize(2));

    // dw:transform
    TransactionStackEntry transactionStackEntry = transaction.getTransactionStack().get(0);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("transform"));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));
    assertThat(transactionStackEntry.getData().getMessage().getPayload().getDataType().getMediaType(),
               startsWith("application/dw"));
    assertThat(new String(transactionStackEntry.getData().getMessage().getPayload().getContent(), MediaType
        .parse(transactionStackEntry.getData().getMessage().getPayload().getDataType().getMediaType()).charset().get()),
               containsStringIgnoringLineBreaks(DATAWEAVE_COLLECTION_FIRST_TRANSFORMATION));

    // dw:transform 2
    transactionStackEntry = transaction.getTransactionStack().get(1);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "1"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("transform"));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));
    assertThat(transactionStackEntry.getData().getMessage().getPayload().getDataType().getMediaType(),
               startsWith("application/dw"));
    assertThat(new String(transactionStackEntry.getData().getMessage().getPayload().getContent(), MediaType
        .parse(transactionStackEntry.getData().getMessage().getPayload().getDataType().getMediaType()).charset().get()),
               containsStringIgnoringLineBreaks(DATAWEAVE_COLLECTION_SECOND_TRANSFORMATION));
  }

  private void ftpAssertNotifications(String appName) throws IOException {
    final List<Transaction> transactions = collectNotifications(appName);
    assertThat(consumeNotifications(appName).getTransactions(), emptyCollectionOf(Transaction.class));
    checkFtpTransactions(transactions);
  }

  private void checkFtpTransactions(List<Transaction> transactions) throws IOException {
    assertThat(transactions, hasSize(1));
    Transaction transaction = transactions.get(0);
    assertThat(transaction.getId(), is(notNullValue()));
    assertThat(transaction.getTimestamp(), is(notNullValue()));
    assertThat(transaction.getGlobalName(), is(FLOW_NAME));
    // Source - Attributes
    assertThat(transaction.getMessage(), is(notNullValue()));
    assertThat(transaction.getMessage().getAttributes().getDataType().getMediaType(),
               startsWith(MediaType.create("application", "dw").toString()));
    // Source - Payload
    assertThat(transaction.getMessage().getPayload().getDataType().getMediaType(),
               equalToIgnoringCase(APP_XML_MEDIA_TYPE));
    assertThat(new String(transaction.getMessage().getPayload().getContent(), inputCharset),
               isSimilarTo(INPUT_XML));

    // Stack Entries (ignored the set-payload)
    assertThat(transaction.getTransactionStack(), hasSize(2));

    // ftp:list
    TransactionStackEntry transactionStackEntry = transaction.getTransactionStack().get(0);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0"));
    assertThat(transactionStackEntry.getComponentLocation().getTypedComponentIdentifier().getName(), equalTo("list"));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));

    List<String> actualContentLines =
        readLines(new StringReader(new String(transactionStackEntry.getData().getMessage().getPayload().getContent(),
                                              defaultCharset())));
    List<String> expectedContentLines =
        IOUtils.readLines(this.getClass().getClassLoader().getResourceAsStream("messageHistory/ftpList.dw.output"),
                          defaultCharset().name());
    assertThat(actualContentLines, hasSize(expectedContentLines.size()));

    actualContentLines.stream().filter(line -> !line.contains("timestamp:")).collect(toList())
        .containsAll(expectedContentLines.stream().filter(line -> !line.contains("timestamp:")).collect(toList()));
  }

  private void streamingAssertNotifications(String appName) {
    final List<Transaction> transactions = collectNotifications(appName);
    assertThat(consumeNotifications(appName).getTransactions(), emptyCollectionOf(Transaction.class));
    checkStreamingTransactions(transactions);
  }

  private void checkStreamingTransactions(List<Transaction> transactions) {
    assertThat(transactions, hasSize(1));
    Transaction transaction = transactions.get(0);
    assertThat(transaction.getId(), is(notNullValue()));
    assertThat(transaction.getGlobalName(), is(FLOW_NAME));
    assertThat(transaction.getTimestamp(), is(notNullValue()));
    // Source - Payload
    assertThat(transaction.getMessage().getPayload().getDataType().getMediaType(),
               equalToIgnoringCase(APP_XML_MEDIA_TYPE));
    assertThat(new String(transaction.getMessage().getPayload().getContent(), inputCharset), isSimilarTo(INPUT_XML));

    // Stack Entries
    assertThat(transaction.getTransactionStack(), hasSize(2));

    TransactionStackEntry transactionStackEntry = transaction.getTransactionStack().get(0);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0"));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));
    assertThat(transactionStackEntry.getData().getMessage().getPayload().getDataType().getMediaType(),
               startsWith(MediaType.create("application", "dw").toString()));
    assertThat(new String(transactionStackEntry.getData().getMessage().getPayload().getContent(), MediaType
        .parse(transactionStackEntry.getData().getMessage().getPayload().getDataType().getMediaType()).charset().get()),
               containsStringIgnoringLineBreaks(JSON_PAYLOAD));
  }

  private void checkSchedulerTransactions(List<Transaction> transactions) {
    assertThat(transactions, hasSize(1));
    Transaction transaction = transactions.get(0);
    assertThat(transaction.getId(), is(notNullValue()));
    assertThat(transaction.getTimestamp(), is(notNullValue()));
    assertThat(transaction.getGlobalName(), is(FLOW_NAME));
    // Source - Attributes and Payload
    assertThat(transaction.getMessage().getAttributes().getContent(), is(nullValue()));
    assertThat(transaction.getMessage().getPayload().getContent(), is(nullValue()));

    // Stack Entries
    assertThat(transaction.getTransactionStack(), hasSize(3));

    // set-payload
    TransactionStackEntry transactionStackEntry = transaction.getTransactionStack().get(0);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "0"));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));
    assertThat(new String(transactionStackEntry.getData().getMessage().getPayload().getContent(), MediaType
        .parse(transactionStackEntry.getData().getMessage().getPayload().getDataType().getMediaType()).charset().get()),
               isSimilarTo(INPUT_XML));
    // dw:transform
    transactionStackEntry = transaction.getTransactionStack().get(1);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "1"));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));
    assertThat(new String(transactionStackEntry.getData().getMessage().getPayload().getContent(), MediaType
        .parse(transactionStackEntry.getData().getMessage().getPayload().getDataType().getMediaType()).charset().get()),
               containsStringIgnoringLineBreaks(CSV_PAYLOAD));

    // file:read
    transactionStackEntry = transaction.getTransactionStack().get(2);
    assertThat(transactionStackEntry.getComponentLocation().getLocation(), equalTo(SAMPLE_FLOW_PROCESSORS_PATH + "2"));
    assertThat(transactionStackEntry.getData().getError(), is(nullValue()));
    assertThat(new String(transactionStackEntry.getData().getMessage().getPayload().getContent(), MediaType
        .parse(transactionStackEntry.getData().getMessage().getPayload().getDataType().getMediaType()).charset().get()),
               equalTo(PAYLOAD_EXPECTED));
  }

  private List<Transaction> collectNotifications(String appName) {
    final List<Transaction> transactions = new ArrayList<>();
    try {
      Thread.sleep(1000);
    } catch (InterruptedException e) {
      throw new IllegalStateException("unexpected interrupt", e);
    }
    PollingProber prober = new PollingProber();
    final JUnitLambdaProbe proberAction = new JUnitLambdaProbe(() -> {
      MessageHistory messageHistory = consumeNotifications(appName);
      if (messageHistory.getTransactions().isEmpty() && transactions.isEmpty()) {
        return false;
      }
      transactions.addAll(messageHistory.getTransactions());
      return true;
    }, "Notifications couldn't be retrieved");
    prober.check(proberAction);
    return transactions;
  }

  private void deployApplication(String appName, String appLocation) throws Exception {
    deployApplication(appName, appLocation, true);
  }

  private void deployApplication(String appName, String appLocation, boolean assertResponse) throws Exception {
    String url = getApplicationService(appName);

    HttpClient client = HttpClientBuilder.create().build();
    HttpPut request = new HttpPut(url);
    request.setEntity(new StringEntity("{\"url\": \"" + getAppUrlLocation(appLocation).toURI() + "\"}", APPLICATION_JSON));

    HttpResponse response = client.execute(request);

    logger.info("Application deployed response: " + IOUtils.toString(response.getEntity().getContent()));
    if (assertResponse) {
      assertThat(response.getStatusLine().getStatusCode(), is(202));
    }
  }

  private void startFlow(String appName, String flowName) throws Exception {
    String url = getApplicationService(appName) + "/flows/" + flowName + "/start";

    HttpClient client = HttpClientBuilder.create().build();
    HttpPut request = new HttpPut(url);

    HttpResponse response = client.execute(request);

    assertThat(response.getStatusLine().getStatusCode(), is(202));
  }

  private String getApplicationService(String appName) {
    return "http://localhost:" + agentPort.getNumber() + "/mule/applications/" + appName;
  }

  private void unDeployApplication(String appName) throws Exception {
    unDeployApplication(appName, true);
  }

  private void unDeployApplication(String appName, boolean assertResponse) throws Exception {
    String url = getApplicationService(appName);

    HttpClient client = HttpClientBuilder.create().build();
    HttpDelete request = new HttpDelete(url);

    HttpResponse response = client.execute(request);

    logger.debug("Application undeployed response: " + IOUtils.toString(response.getEntity().getContent()));
    if (assertResponse) {
      assertThat(response.getStatusLine().getStatusCode(), is(202));
    }
  }

  private void sendRequest() throws IOException {
    sendRequest(true);
  }

  private void sendRequest(boolean checkResponse) throws IOException {
    String url = "http://localhost:" + httpServerPort.getNumber() + "/test";

    int timeout = 5;
    RequestConfig config = RequestConfig.custom()
        .setConnectTimeout(timeout * 1000)
        .setConnectionRequestTimeout(timeout * 1000)
        .setSocketTimeout(timeout * 5000).build();

    HttpClient client = HttpClientBuilder.create().build();
    HttpPost request = new HttpPost(url);
    request.setConfig(config);
    request.addHeader("Content-Type", MediaType.create("application", "xml").withCharset(inputCharset).toString());
    String xml = INPUT_XML;
    HttpEntity entity = new ByteArrayEntity(xml.getBytes(inputCharset.name()));
    request.setEntity(entity);

    HttpResponse response = client.execute(request);

    logger.debug("Send request response: " + IOUtils.toString(response.getEntity().getContent()));
    if (checkResponse) {
      assertThat(response.getStatusLine().getStatusCode(), is(200));
    }
  }

  public URL getAppUrlLocation(String appLocation) throws Exception {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File applicationFolder = new File(targetTestClassesFolder, appLocation);
    File appZipFile = temporaryFolder.newFile();
    compress(appZipFile, applicationFolder);
    return appZipFile.toURI().toURL();
  }

  private void sendEmails() throws Exception {
    MimeMessage mimeMessage = new MimeMessage(testSession);
    mimeMessage.setFrom(new InternetAddress(GUILLE_MAIL));
    mimeMessage.setRecipients(TO, MARTIN_MAIL);
    mimeMessage.setText(EMAIL_CONTENT);
    mimeMessage.setDescription(EMAIL_CONTENT);
    mimeMessage.setSubject(EMAIL_SUBJECT);
    user.deliver(mimeMessage);
  }
}
