/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.tests.integration.tooling.client;

import static java.lang.String.format;
import static java.util.Collections.emptyMap;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.apache.tika.io.IOUtils.readLines;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;
import static org.junit.Assert.assertThat;
import static org.junit.rules.ExpectedException.none;
import static org.mule.maven.client.test.MavenTestHelper.createDefaultEnterpriseMavenConfiguration;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getMuleVersion;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getTestLog4JConfigurationFile;
import static org.mule.tooling.client.test.AbstractMuleRuntimeTestCase.getToolingVersion;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.artifact.resources.ResourceLoader;
import org.mule.tooling.client.api.component.location.Location;
import org.mule.tooling.client.api.datasense.DataSenseRequest;
import org.mule.tooling.client.api.exception.ToolingArtifactNotFoundException;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.feature.Feature;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrap;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapConfiguration;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapFactory;
import org.mule.tooling.client.test.utils.ZipUtils;
import org.mule.tooling.client.tests.integration.category.NeedMuleRuntimeTest;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;
import java.util.Optional;

import io.qameta.allure.Description;
import io.qameta.allure.Story;
import org.apache.commons.io.FileUtils;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;

@io.qameta.allure.Feature("ConnectivityTestingService")
@Story("Integration tests for ConnectivityTesting that use the ToolingBootstrap and ToolingRuntimeClient")
@Category(NeedMuleRuntimeTest.class)
public class ToolingArtifactTestCase {

  @Rule
  public ExpectedException expectedException = none();

  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  private ToolingRuntimeClient toolingRuntimeClient;
  private ToolingRuntimeClientBootstrap bootstrap;

  @Before
  public void setUpToolingRuntimeClient() throws Exception {
    bootstrap = ToolingRuntimeClientBootstrapFactory.newToolingRuntimeClientBootstrap(
                                                                                      ToolingRuntimeClientBootstrapConfiguration
                                                                                          .builder()
                                                                                          .muleVersion(getMuleVersion())
                                                                                          .toolingVersion(getToolingVersion())
                                                                                          .mavenConfiguration(createDefaultEnterpriseMavenConfiguration())
                                                                                          .workingFolder(temporaryFolder
                                                                                              .newFolder("workingDir"))
                                                                                          .log4jConfiguration(getTestLog4JConfigurationFile())
                                                                                          .build());
    toolingRuntimeClient = bootstrap.getToolingRuntimeClientBuilderFactory().create().build();
  }

  @After
  public void disposeToolingBootstrap() throws Exception {
    if (bootstrap != null) {
      bootstrap.dispose();
    }
  }

  @Test
  @Description("Checks that an application cannot be created with none compatible version of plugins declared on domain")
  public void shouldFailToCreateApplicationToolingArtifactDueToWrongVersionOfPlugins() throws Exception {
    expectedException.expect(ToolingException.class);
    expectedException
        .expectMessage(allOf(containsString("Error while creating application"),
                             containsString("Incompatible version of plugin 'Email' (org.mule.connectors:mule-email-connector) found")));
    toolingRuntimeClient.newToolingArtifact(toUrl("applications/email-with-domain-exploded-invalid-version"),
                                            emptyMap(),
                                            toolingRuntimeClient
                                                .newToolingArtifact(toUrl("domains/email-domain-invalid-version"), emptyMap())
                                                .getId());
  }

  @Test
  @Description("Checks that an application cannot be created with none compatible version of plugins declared on domain")
  public void shouldFailToCreateApplicationToolingArtifactDueToDifferentDomainDependency() throws Exception {
    expectedException.expect(ToolingException.class);
    expectedException
        .expectMessage(allOf(containsString("Error while creating application"),
                             containsString("Application declares a different domain dependency on its pom.xml")));
    toolingRuntimeClient.newToolingArtifact(toUrl("applications/email-with-domain"),
                                            emptyMap(),
                                            toolingRuntimeClient.newToolingArtifact(toUrl("domains/email-domain"), emptyMap())
                                                .getId());
  }

  @Test
  @Description("Checks that an application dependencies are retrieved from remote repositories declared on pom application")
  public void shouldUseRemoteRepositoriesFromApplicationPomFile() throws Exception {
    File localMavenRepository = temporaryFolder.newFolder();
    ToolingRuntimeClient toolingRuntimeClient = bootstrap.getToolingRuntimeClientBuilderFactory().create()
        .withMavenConfiguration(MavenConfiguration.newMavenConfigurationBuilder()
            .localMavenRepositoryLocation(localMavenRepository)
            .ignoreArtifactDescriptorRepositories(false)
            .build())
        .build();
    ToolingArtifact toolingArtifact =
        toolingRuntimeClient.newToolingArtifact(toUrl("applications/app-pom-remote-repositories"), emptyMap());
    assertThat(toolingArtifact, not(nullValue()));

    assertThat(new File(localMavenRepository, "com/amazon/redshift/redshift-jdbc42/1.2.10.1009/redshift-jdbc42-1.2.10.1009.jar")
        .exists(), is(true));
  }

  private URL toUrl(String appFolder) {
    try {
      File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
      return new File(targetTestClassesFolder, appFolder).toURI().toURL();
    } catch (Exception e) {
      throw new IllegalStateException(format("Couldn't get URL for application %s", appFolder));
    }
  }

  @Test
  @Description("Checks a ToolingArtifactNotFoundException is thrown when fetching by using a non existing id")
  public void toolingArtifactNotFoundException() throws Exception {
    expectedException.expect(ToolingArtifactNotFoundException.class);
    toolingRuntimeClient.fetchToolingArtifact("invalid-id");
  }

  @Test
  @Description("Checks that an application can be created with dependencies declared on the application, having empty dependencies on domain")
  public void shouldCreateToolingArtifactWithDependenciesOnlyOnArtifact() throws Exception {
    DataSenseRequest request = new DataSenseRequest();
    request.setLocation(Location.builder().globalName("flow").addProcessorsPart().addIndexPart(0).build());
    toolingRuntimeClient.newToolingArtifact(toUrl("applications/domain-simple-app"),
                                            emptyMap(),
                                            toolingRuntimeClient
                                                .newToolingArtifact(toUrl("domains/empty-domain"), emptyMap())
                                                .getId())
        .dataSenseService().resolveDataSense(request);
  }

  @Test
  @Description("Checks that an resource loader exposed by ToolingArtifact can access resources using Resouces::")
  public void loadResourcesUsingResourcePrefix() throws IOException {
    Feature<ResourceLoader> resourceLoaderFeature =
        toolingRuntimeClient.newToolingArtifact(toUrl("applications/test-api-v8"), emptyMap()).getResourceLoader();
    assertThat(resourceLoaderFeature.isEnabled(), is(true));
    // getResourceAsStream
    try (InputStream inputStream =
        resourceLoaderFeature.get()
            .getResourceAsStream("resource::com.mycompany:test-api:1.1.0-SNAPSHOT:raml:zip:test-api.raml")) {
      assertThat(inputStream, is(notNullValue()));
      assertThat(readLines(inputStream).get(1), is("title: Test API"));
    }
    // loadFrom
    try (InputStream inputStream =
        resourceLoaderFeature.get()
            .loadFrom("test-api.raml", "com.mycompany", "test-api", of("1.1.0-SNAPSHOT"), "raml", "zip").orElse(null)) {
      assertThat(inputStream, is(notNullValue()));
      assertThat(readLines(inputStream).get(1), is("title: Test API"));
    }
    // loadFrom (default -> jar)
    assertThat(resourceLoaderFeature.get()
        .loadFrom("test-api.raml", "com.mycompany", "test-api", of("1.1.0-SNAPSHOT")), is(empty()));
    // getResource
    URL resourceUrl =
        resourceLoaderFeature.get().getResource("resource::com.mycompany:test-api:1.1.0-SNAPSHOT:raml:zip:test-api.raml");
    assertThat(resourceUrl, is(notNullValue()));

    // findIn
    Optional<URL> resourceUrlOptional =
        resourceLoaderFeature.get().findIn("test-api.raml", "com.mycompany", "test-api", of("1.1.0-SNAPSHOT"), "raml", "zip");
    assertThat(resourceUrlOptional, not(empty()));

    assertThat(resourceUrlOptional.get(), equalTo(resourceUrl));

    // findIn (default -> jar)
    assertThat(resourceLoaderFeature.get().findIn("test-api.raml", "com.mycompany", "test-api", of("1.1.0-SNAPSHOT")),
               equalTo(empty()));
  }

  @Test
  public void refreshFragmentSnapshotOnAPISync() throws IOException, URISyntaxException {
    String appFolder = "applications/apisync-refresh-snapshot-version";

    Feature<ResourceLoader> resourceLoaderFeature =
        toolingRuntimeClient.newToolingArtifact(toUrl(appFolder), emptyMap()).getResourceLoader();
    assertThat(resourceLoaderFeature.isEnabled(), is(true));
    String resource = "resource::com.mycompany:reused-library:1.3.0-SNAPSHOT:raml-fragment:zip:reused-fragment.raml";

    // getResourceAsStream
    try (InputStream inputStream =
        resourceLoaderFeature.get()
            .getResourceAsStream(resource)) {
      assertThat(inputStream, is(notNullValue()));
      assertThat(readLines(inputStream).get(1), is("usage: This library contains some data types"));
    }
    File localRepository = createDefaultEnterpriseMavenConfiguration().getLocalMavenRepositoryLocation();
    Path artifactFolder = localRepository.toPath().resolve(Paths.get("com", "mycompany", "reused-library", "1.3.0-SNAPSHOT"));
    Path zipFilePath = artifactFolder.resolve("reused-library-1.3.0-SNAPSHOT-raml-fragment.zip");
    Path tempArtifactZip = artifactFolder.resolve("__reused-library-1.3.0-SNAPSHOT-raml-fragment__.zip");
    FileUtils.deleteQuietly(tempArtifactZip.toFile());
    Files.copy(zipFilePath, tempArtifactZip);

    File temp = temporaryFolder.newFolder();
    Files.copy(Paths.get(this.getClass().getClassLoader()
        .getResource("apisync/reused-library-1.3.0-SNAPSHOT/reused-fragment.modified").toURI()),
               new File(temp, "reused-fragment.raml").toPath());
    ZipUtils.compress(zipFilePath.toFile(), temp);
    try {
      resourceLoaderFeature =
          toolingRuntimeClient.newToolingArtifact(toUrl(appFolder), emptyMap()).getResourceLoader();
      assertThat(resourceLoaderFeature.isEnabled(), is(true));
      // getResource
      try (InputStream inputStream =
          resourceLoaderFeature.get()
              .getResource(resource).openStream()) {
        assertThat(inputStream, is(notNullValue()));
        List<String> content = readLines(inputStream);
        assertThat(content.get(1), is("usage: This library contains some data types for reuse"));
      }

      resourceLoaderFeature =
          toolingRuntimeClient.newToolingArtifact(toUrl(appFolder), emptyMap()).getResourceLoader();
      assertThat(resourceLoaderFeature.isEnabled(), is(true));
      // getResourceAsStream
      try (InputStream inputStream =
          resourceLoaderFeature.get()
              .getResourceAsStream(resource)) {
        assertThat(inputStream, is(notNullValue()));
        List<String> content = readLines(inputStream);
        assertThat(content.get(1), is("usage: This library contains some data types for reuse"));
      }
    } finally {
      FileUtils.deleteQuietly(zipFilePath.toFile());
      Files.copy(tempArtifactZip, zipFilePath);
      FileUtils.deleteQuietly(tempArtifactZip.toFile());
    }
  }

  @Test
  @Description("Checks that an resource loader exposed by ToolingArtifact can access resources exported by the application")
  public void loadExportedResource() throws IOException {
    Feature<ResourceLoader> resourceLoaderFeature =
        toolingRuntimeClient.newToolingArtifact(toUrl("applications/value-providers"), emptyMap()).getResourceLoader();
    assertThat(resourceLoaderFeature.isEnabled(), is(true));
    try (InputStream inputStream =
        resourceLoaderFeature.get().getResourceAsStream("human.wsdl")) {
      assertThat(inputStream, is(notNullValue()));
      assertThat(readLines(inputStream).get(2), containsString("The Human Resources Web Service "));
    }
    assertThat(resourceLoaderFeature.get().getResource("human.wsdl"), is(notNullValue()));
  }

  @Test
  @Description("Checks that an resource loader exposed by ToolingArtifact cannot access resources not exported by the application")
  public void shouldNotLoadNotExportedResource() throws IOException {
    Feature<ResourceLoader> resourceLoaderFeature =
        toolingRuntimeClient.newToolingArtifact(toUrl("applications/datasense-declarations"), emptyMap()).getResourceLoader();
    assertThat(resourceLoaderFeature.isEnabled(), is(true));
    assertThat(resourceLoaderFeature.get().getResource("304ff.ffd"), is(nullValue()));
    assertThat(resourceLoaderFeature.get().getResourceAsStream("304ff.ffd"), is(nullValue()));
  }

}
